import os.path
import tempfile


class SafetyWriteFile(object):
    def __init__(self, filename):
        self.is_first_write = True

        self.original_filename = filename
        dirname = os.path.dirname(filename)

        self.tmp_file = tempfile.NamedTemporaryFile(mode='w', dir=dirname, delete=False)

    def __getattr__(self, name):
        # NOTE(lavrukov): get from _TemporaryFileWrapper
        # Attribute lookups are delegated to the underlying file
        # and cached for non-numeric results
        # (i.e. methods are cached, closed and friends are not)
        tmp_file = self.__dict__['tmp_file']
        attr = getattr(tmp_file, name)
        if not issubclass(type(attr), type(0)):
            setattr(self, name, attr)
        return attr

    def __enter__(self):
        self.tmp_file.__enter__()
        return self

    def __exit__(self, exc, value, tb):
        if exc is not None:
            result = self.tmp_file.__exit__(exc, value, tb)
            os.unlink(self.tmp_file.name)
            return result

        result = self.tmp_file.__exit__(exc, value, tb)

        os.rename(self.tmp_file.name, self.original_filename)
        os.chmod(self.original_filename, 0o644)

        return result
