import sys
import requests

from requests import urllib3
from requests import adapters

from django.utils import six
from django.conf import settings
from rest_framework import status

from intranet.crt.exceptions import CrtRequestError


class CrtSession(requests.Session):
    RETRY_TOTAL_COUNT = 5
    BACKOFF_FACTOR = 0.1
    STATUS_FORCELIST = [
        status.HTTP_500_INTERNAL_SERVER_ERROR,
        status.HTTP_502_BAD_GATEWAY,
    ]
    DEFAULT_TIMEOUT = 3

    def __init__(self, max_retries=None, backoff_factor=None, status_forcelist=None, timeout=None):
        if max_retries is None:
            max_retries = self.RETRY_TOTAL_COUNT
        if backoff_factor is None:
            backoff_factor = self.BACKOFF_FACTOR
        if status_forcelist is None:
            status_forcelist = self.STATUS_FORCELIST

        self._timeout = timeout if timeout is not None else self.DEFAULT_TIMEOUT

        retry = urllib3.Retry(
            total=max_retries,
            backoff_factor=backoff_factor,
            status_forcelist=status_forcelist,
        )
        adapter = adapters.HTTPAdapter(max_retries=retry)

        super(CrtSession, self).__init__()

        self.mount('https://', adapter)
        self.verify = settings.ROOT_CA_BUNDLE

    def request(self, method, url, timeout=None, **kwargs):
        timeout = timeout if timeout is not None else self._timeout

        try:
            return super(CrtSession, self).request(method, url, timeout=timeout, **kwargs)
        except requests.RequestException as error:
            six.reraise(CrtRequestError, error, sys.exc_info()[2])

    def set_oauth(self):
        self.headers.update({'Authorization': 'OAuth {}'.format(settings.CRT_OAUTH_TOKEN)})
        return self

    def set_basic_auth(self, username, password):
        self.auth = (username, password)
        return self

    def set_basic_robot_auth(self):
        return self.set_basic_auth(settings.CRT_ROBOT, settings.CRT_ROBOT_PASSWORD)
