import re
from contextlib import contextmanager

import ldap
import os.path
import logging

from django.conf import settings


username_pattern = re.compile(r'^[-.\w]+$')
EXCHANGE_GROUP = 'CN=ExchangeMobileAccess,OU=Special,OU=Base,OU=Groups,DC=ld,DC=yandex,DC=ru'
CA_CERT_FILE = os.path.join(settings.CRT_CA_CHAINS_PATH, settings.YANDEX_CHAIN)


def get_client():
    cl = ldap.initialize('ldap://' + settings.LDAP_HOST)

    cl.set_option(ldap.OPT_X_TLS, ldap.OPT_X_TLS_DEMAND)
    cl.set_option(ldap.OPT_X_TLS_CACERTFILE, CA_CERT_FILE)
    # FIXME: Fix later, PASSP-11587
    cl.set_option(ldap.OPT_X_TLS_REQUIRE_CERT, ldap.OPT_X_TLS_NEVER)
    cl.set_option(ldap.OPT_NETWORK_TIMEOUT, 5)
    cl.set_option(ldap.OPT_TIMEOUT, 5)
    cl.set_option(ldap.OPT_X_TLS_NEWCTX, ldap.OPT_OFF)  # Should be the last
    cl.protocol_version = ldap.VERSION3

    return cl


@contextmanager
def ldap_client(username=settings.CRT_ROBOT, password=settings.CRT_ROBOT_PASSWORD):
    cl = get_client()
    cl.start_tls_s()
    cl.simple_bind_s('{0}@ld.yandex.ru'.format(username), password.encode('utf-8'))
    yield cl
    cl.unbind()


def check_username_and_password_in_ldap(username, password):
    if not username or not password or username_pattern.match(username) is None:
        return False

    try:
        with ldap_client(username, password):
            return True
    except ldap.INVALID_CREDENTIALS:
        return False
    except ldap.LDAPError:
        logging.exception('Catch ldap error')
        return False


def check_membership_in_ad_group(username, group):
    with ldap_client() as cl:
        filter_str = '(&(sAMAccountName={})(memberOf={}))'.format(username, group)
        result = cl.search_s('cn=users,dc=ld,dc=yandex,dc=ru', ldap.SCOPE_SUBTREE, filter_str)
    return len(result) > 0


def is_exchange_user(username):
    return check_membership_in_ad_group(username, EXCHANGE_GROUP)


def get_username_by_dn(dn_str):
    with ldap_client() as cl:
        user_bases = [
            settings.LDAP_USERS_BASE,
            settings.LDAP_FOREIGN_USERS_BASE,
        ]
        for base in user_bases:
            result = cl.search_s(
                base=base,
                scope=ldap.SCOPE_SUBTREE,
                filterstr='(distinguishedName={})'.format(dn_str),
                attrlist=['sAMAccountName'],
            )
            if result:
                break
    if len(result) == 0:
        raise ldap.NO_RESULTS_RETURNED

    return result[0][1]['sAMAccountName'][0]
