import urllib.parse

from rest_framework import status
from django.conf import settings

from intranet.crt.utils.http import CrtSession


def raise_for_wrong(response):
    """Внутреннее апи не предназначено для внешнего использования. Иногда оно может возвращать 200
    при невалидном запросе
    """
    try:
        response_data = response.json()
    except ValueError:
        raise RuntimeError('Invalid json on request "{}". Content: {}'.format(
            response.url,
            response.content,
        ))

    if response.status_code == status.HTTP_200_OK:
        if isinstance(response_data, list) or 'errors' not in list(response_data.keys()):
            return

    raise RuntimeError('Invalid status code with request "{}". Content: {}'.format(
        response.url,
        response.content,
    ))


class StaffFilterApi(object):
    """CERTOR-521
    Ходим во внутреннее api staff. По переданному хешу нам необходимо сохранить фильтр для робота и
    получить id сохраненного. По нему мы можем получить список пользователей, подходящих под фильтр.
    Для того, что бы не пытаться при каждом запросе создавать фильтр, сначала проверяем, существует
    ли сохраненный фильтр с таким хешом у нашего робота
    """
    filter_endpoint = urllib.parse.urljoin(settings.CRT_STAFF_URL, '/filters-api/saved-filters/')

    def __init__(self):
        self.session = CrtSession().set_oauth()

    def get_csrf_token(self):
        response = self.session.get(self.filter_endpoint)
        raise_for_wrong(response)
        return response.cookies['csrftoken']

    def create_filter(self, filter_hash):
        data = {
            'department_url': None,
            'person_login': None,
            'name': settings.CRT_ROBOT,
            'filter_id': filter_hash,
        }

        csrf_token = self.get_csrf_token()
        create_endpoint = urllib.parse.urljoin(self.filter_endpoint, 'create/')
        response = self.session.post(
            create_endpoint,
            json=data,
            headers={
                'X-CSRFToken': csrf_token,
                'Referer': create_endpoint,
            },
        )
        raise_for_wrong(response)

        return response.json()['saved_filter']['id']

    def get_existing_filters(self):
        response = self.session.get(self.filter_endpoint)
        raise_for_wrong(response)
        return {filter['filter_id']: filter['id'] for filter in response.json()}

    def get_filter_id(self, filter_hash):
        existing_filters = self.get_existing_filters()
        if filter_hash in existing_filters:
            return existing_filters[filter_hash]

        return self.create_filter(filter_hash)

    def get_persons_query_response(self, person_endpoint, page=1):
        params = {
            '_page': page,
            '_limit': settings.CRT_STAFF_FILTER_LIMIT,
        }
        response = self.session.get(person_endpoint, params=params)
        raise_for_wrong(response)
        return response.json()

    def get_filter_users(self, filter_hash):
        filter_id = self.get_filter_id(filter_hash)
        person_endpoint = urllib.parse.urljoin(self.filter_endpoint, '{}/persons/'.format(filter_id))

        response_data = self.get_persons_query_response(person_endpoint)
        logins = {person['login'] for person in response_data['result']}

        for page in range(2, response_data['pages'] + 1):
            response_data = self.get_persons_query_response(person_endpoint, page)
            logins |= {person['login'] for person in response_data['result']}

        return logins
