import constance
import logging

from django.conf import settings
from django.template.loader import render_to_string
from ids.exceptions import BackendError
from ids.registry import registry

from intranet.crt.constants import CERT_TYPE

log = logging.getLogger(__name__)


issues_repo = registry.get_repository(
    'startrek2',
    'issues',
    user_agent=settings.CRT_ROBOT,
    oauth_token=settings.CRT_OAUTH_TOKEN,
    timeout=settings.CRT_STARTREK_TIMEOUT,
)
queues_repo = registry.get_repository(
    'startrek2',
    'queues',
    user_agent=settings.CRT_ROBOT,
    oauth_token=settings.CRT_OAUTH_TOKEN,
    timeout=settings.CRT_STARTREK_TIMEOUT,
)


def get_component_id(queue, name):
    queue = queues_repo.get_one({'id': queue})
    component = [comp for comp in queue.components if comp.name == name][0]
    return component.id


def close_st_issue_for_cert_approval(certificate, comment=None, transition='close', resolution='fixed', approver=None):
    issues = issues_repo.get({'filter': 'key:{}'.format(certificate.approve_request.st_issue_key)})

    if len(issues) != 1:
        raise RuntimeError('StarTrack returned 0 or more 1 tickets')

    if approver is not None:
        issues[0].update(assignee=approver.username, ignore_version_change=True)

    issues[0].comments.create(text=comment)

    try:
        issues[0].transitions.get(transition).execute(resolution=resolution)
    except BackendError:
        pass


def create_st_issue_for_cert_approval(certificate):
    hosts = certificate.hosts.all()
    count = hosts.count()
    if count == 0:
        host_info = ''
    else:
        first = hosts.first().hostname
        if count == 1:
            host_info = ' ({first})'.format(first=first)
        else:
            host_info = ' ({first} и ещё {count})'.format(first=first, count=count - 1)

    summary = 'Запрос {cert_type}-сертификата требует подтверждения СИБ{host_info}'.format(
        cert_type=CERT_TYPE.HUMAN_PREFIX[certificate.type.name]['ru'],
        host_info=host_info,
    )

    description = render_to_string(
        'host-certificate-need-approve.txt',
        {
            'certificate': certificate,
            'hosts': hosts,
            'crt_api_base_url': settings.CRT_API_URL,
        }
    )

    components = settings.CRT_STARTREK_SEC_COMPONENTS
    if certificate.requester.username in settings.CRT_STARTREK_REQUESTER_TO_COMPONENTS:
        additional = settings.CRT_STARTREK_REQUESTER_TO_COMPONENTS[certificate.requester.username]
        components.extend(additional)

    issue = issues_repo.create(
        queue=settings.CRT_STARTREK_SECTASK_QUEUE,
        summary=summary,
        description=description,
        components=components,
        followers=[certificate.requester.username],
    )
    log.info('Created ST issue %s: %s', issue.key, issue.summary)
    return issue


def create_st_issue_for_expiring_cert(certificate, days, expiring_hosts):

    summary = 'Срок действия сертификата {} ({}) подходит к концу'.format(
        certificate.serial_number,
        certificate.ca_name,
    )

    filter_options = {
        'queue': settings.CRT_STARTREK_FIRE_QUEUE,
        'components': get_component_id(
            settings.CRT_STARTREK_FIRE_QUEUE,
            settings.CRT_STARTREK_EXPIRING_CERT_COMPONENT,
        ),
        'summary': summary,
    }
    filter_list = ['%s:%s' % (key, val) for (key, val) in filter_options.items()]
    query = issues_repo.get(
        {'filter': filter_list}
    )
    if len(query):
        return

    description = render_to_string(
        'host-certificate-expires-soon.txt',
        {
            'cert': certificate,
            'hosts': ', '.join(certificate.hosts.values_list('hostname', flat=True)),
            'days': days,
            'expiring_hosts': ', '.join(expiring_hosts),
        }
    )

    issue = issues_repo.create(
        queue=settings.CRT_STARTREK_FIRE_QUEUE,
        summary=summary,
        description=description,
        components=[settings.CRT_STARTREK_EXPIRING_CERT_COMPONENT],
    )
    log.info('Created ST issue %s: %s', issue.key, issue.summary)
    return issue


def create_issue_for_broken_filter(tag_filter, current_logins, fetched_logins):

    added_logins = sorted(set(fetched_logins) - set(current_logins))
    removed_logins = sorted(set(current_logins) - set(fetched_logins))

    summary = 'Изменения в фильтре "{}" превысили допустимый лимит'.format(
        tag_filter.name
    )
    description = render_to_string(
        'tag-filter-is-broken.txt',
        {
            'tag_filter': tag_filter,
            'added_logins': added_logins,
            'removed_logins': removed_logins,
            'crt_api_base_url': settings.CRT_API_URL,
        }
    )

    issue = issues_repo.create(
        queue=settings.CRT_STARTREK_SECTASK_QUEUE,
        summary=summary,
        description=description,
        followers=[
            username.strip() for username in constance.config.CRT_BROKEN_STAFF_FILTER_SECTASK_FOLLOWERS.split(',')
        ],
    )
    log.info('Created ST issue %s: %s', issue.key, issue.summary)
    return issue
