import logging
import traceback

import contextdecorator
from django.db import DatabaseError
from django.utils import timezone

from intranet.crt.core.models import TaskTimestamp
from intranet.crt.exceptions import CrtTimestampError

log = logging.getLogger(__name__)


class TimestampContext(contextdecorator.ContextDecorator):
    def __init__(self, task_type):
        self.task_type = task_type

    @classmethod
    def get_traceback(cls, exc_cls, exc, tb):
        if isinstance(exc, CrtTimestampError):
            return str(exc)

        return ''.join(traceback.format_exception(exc_cls, exc, tb))

    def __enter__(self):
        self.timestamp = TaskTimestamp.objects.create(type=self.task_type)
        return self

    def __exit__(self, exc_cls, exc, tb):
        if exc_cls is not None:
            self.timestamp.traceback = self.get_traceback(exc_cls, exc, tb)
        else:
            self.timestamp.is_success = True

        self.timestamp.finish = timezone.now()

        try:
            self.timestamp.save(update_fields=['finish', 'is_success', 'traceback'])
        except DatabaseError:
            log.exception('Can not write timestamp (id:{}) data to DB'.format(self.timestamp.id))

        return False
