'use strict';

const {readFileSync} = require('fs');
const express = require('express');
const path = require('path');
const morgan = require('morgan');
const RequestId = require('tools-access-express/express/middlewares/request-id');
const UserData = require('./express/middlewares/user-data');
const Html = require('./express/middlewares/html');
const Ping = require('./express/middlewares/ping');
const ErrorMw = require('./express/middlewares/error');
const Redirect = require('./express/middlewares/redirect');
const Rum = require('./express/middlewares/rum');
const config = require('./configs');
const cookieParser = require('cookie-parser');
const {version} = require('./package');

const app = express();
const getFileContent = (modulePath) => readFileSync(require.resolve(modulePath), {encoding: 'utf-8'});
const {
    LISTEN_PORT = 3000,
    NODE_ENV
} = process.env;

let pathPrefix = '';
let antiCache = version;

if (['development', 'auto-tests'].includes(NODE_ENV)) {
    pathPrefix = '';
    antiCache = Math.random();
}

const htmlData = {
    tmplPath: path.join(__dirname, 'dist/ru/index.pug'),
    pathPrefix,
    antiCache
};
const rumData = {
    interface: getFileContent('@yandex-int/rum-counter/dist/inline/interface.min.js'),
    send: getFileContent('@yandex-int/rum-counter/dist/bundle/send.min.js'),
    longtask: getFileContent('@yandex-int/rum-counter/dist/inline/longtask.min.js'),
    env: config.rum.env,
    pathPrefix,
    antiCache
};

app.use(morgan('common'));
app.use(express.static(path.join(__dirname, 'dist'), {index: false}));
app.use(Redirect.create());
app.get('/ping', Ping.create());
app.use('/',
    cookieParser(),
    UserData.create(config.hosts.backend),
    RequestId.create(),
    Rum.create(rumData),
    Rum.ErrorLogging.create({
        interfaceOverRum: getFileContent('@yandex-int/error-counter/dist/interfaceOverRum.min.js'),
        implementation: getFileContent('@yandex-int/error-counter/dist/implementation.min.js'),
        filters: getFileContent('@yandex-int/error-counter/dist/filters.min.js'),
        env: NODE_ENV,
        version
    }),
    Html.create(htmlData)
);
app.use(ErrorMw.except());

app.listen(LISTEN_PORT, () => {
    console.log(`App started on ${LISTEN_PORT}`); // eslint-disable-line no-console
});
