import React from 'react';
import {decl} from 'bem';
import propTypes from 'prop-types';
import memoizeOne from 'memoize-one';
import CertCard from 'b:cert-card m:loading m:is-error';
import Keys from 'b:keycodes';
import Component from 'b:ta-component';

const CERT_REQUEST_FORM_QUERY_PARAMS_PREFIX = 'cr-form-';
const MAIN_FIELDS_SLUGS = ['type', 'ca_name'];

export default decl(Component, {
    block: 'cert-card.container',

    willInit() {
        this.__base(...arguments);

        this.state = {
            certId: null,
            isFetching: false,
            isActionInProgress: false,
            actionError: {
                id: '',
                text: ''
            },
            fetchError: null,
            abcServiceId: '',
            abcServiceUpdateError: null,
            isAdditionalFieldsConfigFetching: false,
            additionalFieldsConfigError: null
        };

        this._refs = {};

        this.handleCloseClick = ::this.handleCloseClick;
        this.handleKeydown = ::this.handleKeydown;
        this.handleBPageClick = ::this.handleBPageClick;
        this.handleActionClick = ::this.handleActionClick;
        this.handleTooltipOutsideClick = ::this.handleTooltipOutsideClick;
        this.handleRetryClick = ::this.handleRetryClick;
        this.handleAbcServiceTooltipOutsideClick = ::this.handleAbcServiceTooltipOutsideClick;
        this.handleAbcServiceChange = ::this.handleAbcServiceChange;
        this.handleReissueButtonClick = ::this.handleReissueButtonClick;

        this.setRefCard = this.setRef('cert-card');
        this.getRefCard = this.getRef('cert-card');

        this.setAbcServiceId = memoizeOne(this.setAbcServiceId);
        this.requestCert = memoizeOne(this.requestCert);
    },
    async didMount() {
        this.__base(...arguments);

        window.addEventListener('keydown', this.handleKeydown);
        this.context.addBPageClickListener(this.handleBPageClick);

        await this.requestCert(this.props.certId);
    },
    didUpdate() {
        this.__base(...arguments);

        const newAbcService = this.props.cert.abc_service;

        this.setAbcServiceId(Object(newAbcService).id);

        this.requestCert(this.props.certId);
    },
    willUnmount() {
        this.__base(...arguments);

        window.removeEventListener('keydown', this.handleKeydown);
        this.context.deleteBPageClickListener(this.handleBPageClick);
    },
    replace() {
        const {
            isFetching,
            isActionInProgress,
            actionError,
            abcServiceUpdateError,
            abcServiceId,
            fetchError,
            isAdditionalFieldsConfigFetching,
            additionalFieldsConfigError
        } = this.state;

        return (
            <CertCard
                {...this.props}
                onCloseClick={this.handleCloseClick}
                onActionClick={this.handleActionClick}
                onTooltipOutsideClick={this.handleTooltipOutsideClick}
                onAbcServiceTooltipOutsideClick={this.handleAbcServiceTooltipOutsideClick}
                onRetryClick={this.handleRetryClick}
                onAbcServiceChange={this.handleAbcServiceChange}
                onReissueButtonClick={this.handleReissueButtonClick}
                loading={isFetching}
                isActionInProgress={isActionInProgress}
                isAdditionalFieldsConfigLoading={isAdditionalFieldsConfigFetching}
                actionError={actionError}
                fetchError={fetchError}
                additionalFieldsConfigError={additionalFieldsConfigError}
                is-error={Boolean(fetchError)}
                abcServiceUpdateError={abcServiceUpdateError}
                abcServiceId={abcServiceId}
                setRef={this.setRefCard}
            />
        );
    },
    handleCloseClick() {
        this.closeCard();
    },
    async handleRetryClick() {
        await this.requestCert(this.props.certId, Boolean(this.state.fetchError));
    },
    handleKeydown(e) {
        if (e.keyCode === Keys.ESC) {
            this.closeCard();
        }
    },
    handleBPageClick({target}) {
        const card = this.getRefCard();
        const targetFromTableTr = target.closest('.certs-table__tbody .certs-table__tr');

        if (!card.contains(target) && !targetFromTableTr) {
            this.closeCard();
        }
    },
    closeCard() {
        this.props.filterMatchStr('/certificates/:certId?', (v, k) => k !== 'certId');
    },
    setAbcServiceId(id) {
        this.setState({abcServiceId: id});
    },
    setRef(id) {
        return (elem) => {
            this._refs[id] = elem;
        };
    },
    getRef(id) {
        return () => this._refs[id];
    },
    async handleActionClick() {
        await this.requestCertAction(...arguments);
    },
    async requestCertAction(certId, actionId) {
        this.setState({isActionInProgress: true});

        try {
            await this.props.requestCertAction(...arguments);
        } catch (err) {
            this.setState({
                actionError: {
                    id: actionId,
                    text: err.toString()
                }
            });

            throw err;
        } finally {
            this.setState({isActionInProgress: false});
        }
    },
    async requestCert(certId) {
        const {requestCert} = this.props;

        this.setState({
            certId,
            isFetching: true,
            fetchError: null,
            abcServiceUpdateError: null
        });

        try {
            await requestCert(certId);
        } catch (fetchError) {
            // Todo: http://st/CERTOR-896
            this.setState({fetchError});

            throw fetchError;
        } finally {
            this.setState({isFetching: false});
        }
    },
    handleTooltipOutsideClick() {
        this.setState({
            actionError: {
                id: '',
                text: ''
            }
        });
    },
    async handleAbcServiceChange(value) {
        const {id: newId = ''} = {...value};
        const {id: certId, abc_service} = this.props.cert;
        const {id: oldId = ''} = {...abc_service};

        if (newId !== oldId) {
            this.setState({abcServiceId: newId});

            try {
                await this.props.requestUpdateAbcService(certId, newId);
            } catch (err) {
                this.setState({abcServiceUpdateError: err.toString()});

                throw err;
            }
        }
    },
    handleAbcServiceTooltipOutsideClick() {
        this.setState({abcServiceUpdateError: null});
    },
    getFieldsPrefixedUrlParams() {
        const {additionalFieldsConfig} = this.props.cert;
        const slugs = [
            ...MAIN_FIELDS_SLUGS,
            ...additionalFieldsConfig.map(({slug}) => slug)
        ];

        return slugs.reduce((acc, slug) => {
            acc[`${CERT_REQUEST_FORM_QUERY_PARAMS_PREFIX}${slug}`] = this.getFieldValueBySlug(slug);

            return acc;
        }, {});
    },
    getFieldValueBySlug(slug) {
        const {cert} = this.props;

        if (slug === 'abc_service') {
            return Object(cert[slug]).id;
        }

        if (slug === 'hosts') {
            return String(cert[slug]);
        }

        return cert[slug];
    },
    async handleReissueButtonClick() {
        const {type, ca_name} = this.props.cert;

        this.setState({
            isAdditionalFieldsConfigFetching: true,
            additionalFieldsConfigError: null
        });

        try {
            await this.props.requestAdditionalFieldsConfig(ca_name, type);

            this.props.updateQueryStr({
                'cr-form': 1,
                ...this.getFieldsPrefixedUrlParams()
            });
        } catch (err) {
            this.setState({
                additionalFieldsConfigError: err.toString()
            });
            throw err;
        } finally {
            this.setState({isAdditionalFieldsConfigFetching: false});
        }
    }
}, {
    contextTypes: {
        addBPageClickListener: propTypes.func.isRequired,
        deleteBPageClickListener: propTypes.func.isRequired
    },
    propTypes: {
        certId: propTypes.string.isRequired,
        expr: propTypes.string.isRequired,
        filterMatchStr: propTypes.func.isRequired,
        path: propTypes.string.isRequired,
        requestCert: propTypes.func.isRequired,
        requestCertAction: propTypes.func.isRequired,
        requestUpdateAbcService: propTypes.func.isRequired,
        updateQueryStr: propTypes.func.isRequired
    }
});
