/* global process */

import React from 'react';
import propTypes from 'prop-types';
import {decl} from 'bem';
import Button from 'b:button2 m:theme=normal m:size=s m:view=default m:tone=default';
import Tooltip from 'b:tooltip m:theme=error m:size=s';
import i18n from 't:i18n';

const {BEM_LANG} = process.env;

export default decl({
    block: 'cert-card',
    elem: 'actions',

    willInit() {
        this.__base(...arguments);

        this._refs = {};
        this.setRefReissueButton = this.setRef('reissue-button');
        this.getRefReissueButton = this.getRef('reissue-button');
    },
    content({
        cert: {available_actions, type},
        isActionInProgress,
        actionError,
        onTooltipOutsideClick,
        onReissueButtonClick,
        isAdditionalFieldsConfigLoading,
        additionalFieldsConfigError
    }) {
        const content = available_actions
            .filter(({id: actionId}) => ['unhold', 'hold', 'revoke'].includes(actionId))
            .map(({id: actionId, name}) => (
                <Button
                    key={actionId}
                    theme="normal"
                    size="s"
                    view="default"
                    tone="default"
                    mix={{
                        block: this.block,
                        elem: 'button'
                    }}
                    onClick={this.getOnActionClick(actionId)}
                    disabled={isActionInProgress}
                    ref={this.setRef(actionId)}
                >
                    {name[BEM_LANG]}
                </Button>
            ));

        if (actionError.id) {
            content.push(
                <Tooltip
                    theme="error"
                    key="tooltip"
                    visible
                    anchor={this.getRef(actionError.id)}
                    to="left"
                    size="s"
                    autoclosable
                    onOutsideClick={onTooltipOutsideClick}
                    mix={{
                        block: this.block,
                        elem: 'action-error-tooltip'
                    }}
                >
                    {actionError.text}
                </Tooltip>
            );
        }

        if (type === 'host') {
            content.push(
                <Button
                    key="reissue"
                    theme="normal"
                    size="s"
                    view="default"
                    tone="default"
                    mix={{
                        block: this.block,
                        elem: 'button',
                        mods: {
                            type: 'reissue'
                        }
                    }}
                    onClick={onReissueButtonClick}
                    disabled={isAdditionalFieldsConfigLoading}
                    ref={this.setRefReissueButton}
                >
                    {i18n('reissue')}
                </Button>,
                <Tooltip
                    theme="error"
                    key="additional-fields-config-error-tooltip"
                    visible={Boolean(additionalFieldsConfigError)}
                    anchor={this.getRefReissueButton}
                    to="left"
                    size="s"
                    autoclosable
                    onOutsideClick={onTooltipOutsideClick}
                >
                    {additionalFieldsConfigError}
                </Tooltip>
            );
        }

        return content;
    },
    setRef(id) {
        return (elem) => {
            this._refs[id] = elem;
        };
    },
    getRef(id) {
        return /* istanbul ignore next */() => this._refs[id];
    },
    getOnActionClick(actionId) {
        const {onActionClick, cert} = this.props;

        return () => {
            onActionClick(cert.id, actionId);
        };
    }
}, {
    propTypes: {
        actionError: propTypes.shape({
            id: propTypes.string.isRequired,
            text: propTypes.string.isRequired
        }).isRequired,
        additionalFieldsConfigError: propTypes.string,
        cert: propTypes.shape({
            available_actions: propTypes.arrayOf(propTypes.shape({
                id: propTypes.string.isRequired,
                name: propTypes.shape({
                    [BEM_LANG]: propTypes.string.isRequired
                }).isRequired
            })).isRequired,
            id: propTypes.number.isRequired
        }).isRequired,
        isActionInProgress: propTypes.bool.isRequired,
        isAdditionalFieldsConfigLoading: propTypes.bool.isRequired,
        onActionClick: propTypes.func.isRequired,
        onReissueButtonClick: propTypes.func.isRequired,
        onTooltipOutsideClick: propTypes.func.isRequired
    }
});
