import React from 'react';
import {decl} from 'bem';
import propTypes from 'prop-types';
import memoizeOne from 'memoize-one';
import omit from 'lodash/omit';
import omitBy from 'lodash/omitBy';
import isEqual from 'lodash/isEqual';
import ModalContent from 'b:cert-request-form e:modal-content m:submitted';
import Component from 'b:ta-component';
import {format as urlFormat} from 'url';

const CERT_REQUEST_FORM_QUERY_PARAMS_PREFIX = 'cr-form-';
const MAIN_FIELDS_NAMES = ['type', 'ca_name'];

export default decl(Component, {
    block: 'cert-request-form',
    elem: 'modal-content.container',

    willInit() {
        this.state = {
            fields: this.getFields(),
            isIssueButtonAvailable: false,
            isAdditionalShown: false,
            isAdditionalFieldsConfigFetching: false,
            additionalFieldsConfigError: null,
            isRequestCertInProgress: false,
            requestError: null,
            isSubmitted: false
        };
        this.handleIssueButtonClick = this.handleIssueButtonClick.bind(this);
        this.handleFieldTypeChange = this.handleFieldTypeChange.bind(this);
        this.handleFieldCanChange = this.handleFieldCanChange.bind(this);
        this.handleFieldAdditionalChange = this.handleFieldAdditionalChange.bind(this);
        this.handleRetryButtonClick = this.handleRetryButtonClick.bind(this);
        this.getCopyLinkURL = ::this.getCopyLinkURL;

        this.setAdditionalFields = memoizeOne(this.setAdditionalFields, isEqual);
    },
    didMount() {
        this.__base(...arguments);

        this.clearQueryParams();
        this.checkForLoadAdditionalFields();
    },
    didUpdate() {
        this.__base(...arguments);

        this.setAdditionalFields(this.props.certRequestForm.additionalFieldsConfig);
    },
    replace({certRequestForm}) {
        const {
            isIssueButtonAvailable,
            isAdditionalShown,
            isAdditionalFieldsConfigFetching,
            additionalFieldsConfigError,
            fields,
            isRequestCertInProgress,
            requestError,
            isSubmitted
        } = this.state;

        return (
            <ModalContent
                {...omit(this.props, ['mix'])}
                isIssueButtonAvailable={isIssueButtonAvailable}
                isAdditionalShown={isAdditionalShown}
                isAdditionalFieldsConfigFetching={isAdditionalFieldsConfigFetching}
                additionalFieldsConfigError={additionalFieldsConfigError}
                fields={fields}
                additionalFieldsConfig={certRequestForm.additionalFieldsConfig}
                onIssueButtonClick={this.handleIssueButtonClick}
                onFieldTypeChange={this.handleFieldTypeChange}
                onFieldCanChange={this.handleFieldCanChange}
                onFieldAdditionalChange={this.handleFieldAdditionalChange}
                onRetryButtonClick={this.handleRetryButtonClick}
                getCopyLinkURL={this.getCopyLinkURL}
                isRequestCertInProgress={isRequestCertInProgress}
                requestError={requestError}
                submitted={isSubmitted}
            />
        );
    },
    setAdditionalFields(additionalFieldsConfig) {
        const {fields} = this.state;

        const additional = additionalFieldsConfig.reduce((acc, {slug}) => ({
            ...acc,
            [slug]: fields.additional[slug] || ''
        }), {});

        this.setState((prevState) => ({
            fields: {
                ...prevState.fields,
                additional
            }
        }), this.checkIssueButtonAvailability);
    },
    async handleIssueButtonClick() {
        let isRequestSuccessed = false;

        this.setState({isRequestCertInProgress: true});

        try {
            const fields = {
                ...this.state.fields.main,
                ...this.state.fields.additional
            };
            const notEmptyFields = omitBy(fields, (val) => val === '');

            await this.props.requestCreateCert(notEmptyFields);

            isRequestSuccessed = true;
        } catch (err) {
            this.setState({requestError: err});
        } finally {
            this.setState({
                isSubmitted: true,
                isRequestCertInProgress: false
            });
        }

        if (isRequestSuccessed) {
            try {
                await this.props.requestCerts(this.props.filtersQueryParams);
            } catch (err) {
                // Noop
            }
        }
    },
    handleFieldTypeChange(type) {
        if (!this.isMainFieldChanged('type', type)) {
            return;
        }

        this.setState((prevState) => ({
            fields: {
                ...prevState.fields,
                main: {
                    ...prevState.fields.main,
                    type
                }
            },
            additionalFieldsConfigError: null
        }), () => {
            this.checkForLoadAdditionalFields();
            this.checkIssueButtonAvailability();
        });
    },
    handleFieldCanChange(ca_name) {
        if (!this.isMainFieldChanged('ca_name', ca_name)) {
            return;
        }

        this.setState((prevState) => ({
            fields: {
                ...prevState.fields,
                main: {
                    ...prevState.fields.main,
                    ca_name
                }
            },
            additionalFieldsConfigError: null
        }), () => {
            this.checkForLoadAdditionalFields();
            this.checkIssueButtonAvailability();
        });
    },
    handleFieldAdditionalChange(slug, value) {
        this.setState((prevState) => ({
            fields: {
                ...prevState.fields,
                additional: {
                    ...prevState.fields.additional,
                    [slug]: value
                }
            }
        }), this.checkIssueButtonAvailability);
    },
    checkIssueButtonAvailability() {
        const {main, additional} = this.state.fields;
        const isEveryMainFieldFilled = Object.values(main).every((field) => field !== '');
        const isEveryRequiredAdditionalFieldFilled = this.props.certRequestForm.additionalFieldsConfig
            .filter(({isRequired}) => isRequired)
            .every(({slug}) => additional[slug] !== '');

        this.setState({isIssueButtonAvailable: isEveryMainFieldFilled && isEveryRequiredAdditionalFieldFilled});
    },
    handleRetryButtonClick() {
        this.setState({
            isSubmitted: false,
            requestError: null
        });
    },
    getCopyLinkURL() {
        const {protocol, host: hostname} = location;
        const pathname = '/certificates/';
        const {main, additional} = this.state.fields;
        const params = {...main, ...additional};
        const formParams = Object.entries(params).reduce((acc, [key, value]) => {
            if (params[key]) {
                acc[`${CERT_REQUEST_FORM_QUERY_PARAMS_PREFIX}${key}`] = value;
            }

            return acc;
        }, {});

        return urlFormat({
            protocol,
            hostname,
            pathname,
            query: {
                'cr-form': 1,
                ...formParams
            }
        });
    },
    getSeparatedFieldsInitialValues() {
        const {initialFieldsValues} = this.props;

        return Object.keys(initialFieldsValues).reduce((acc, key) => {
            if (MAIN_FIELDS_NAMES.includes(key)) {
                acc.main[key] = initialFieldsValues[key];
            } else {
                acc.additional[key] = initialFieldsValues[key];
            }

            return acc;
        }, {
            main: {},
            additional: {}
        });
    },
    isMainFieldChanged(slug, value) {
        return value !== this.state.fields.main[slug];
    },
    getFields() {
        const {
            main: initialMain,
            additional: initialAdditional
        } = this.getSeparatedFieldsInitialValues();

        return {
            main: {
                type: initialMain.type || 'host',
                ca_name: initialMain.ca_name || ''
            },
            additional: {
                ...initialAdditional
            }
        };
    },
    async checkForLoadAdditionalFields() {
        const {type, ca_name} = this.state.fields.main;
        const areMainFieldsFilled = type !== '' && ca_name !== '';

        this.setState({isAdditionalShown: areMainFieldsFilled});

        if (areMainFieldsFilled) {
            this.setState({isAdditionalFieldsConfigFetching: true});

            try {
                await this.props.requestAdditionalFieldsConfig(ca_name, type);
            } catch (err) {
                this.setState({
                    additionalFieldsConfigError: err,
                    isAdditionalShown: false,
                    isIssueButtonAvailable: false
                });
                throw err;
            } finally {
                this.setState({isAdditionalFieldsConfigFetching: false});
            }
        }
    },
    clearQueryParams() {
        this.props.filterQueryStr((v, k) => !k.startsWith(CERT_REQUEST_FORM_QUERY_PARAMS_PREFIX));
    }
}, {
    propTypes: {
        certRequestForm: propTypes.shape({
            additionalFieldsConfig: propTypes.arrayOf(propTypes.shape({
                isRequired: propTypes.bool.isRequired,
                slug: propTypes.string.isRequired
            })).isRequired
        }).isRequired,
        path: propTypes.string.isRequired,
        requestAdditionalFieldsConfig: propTypes.func.isRequired
    }
});
