/* global process */

import React from 'react';
import {decl} from 'bem';
import propTypes from 'prop-types';
import Title from 'b:ta-title';
import Fields from 'e:fields m:additional m:loading';
import Field from 'e:field m:type=suggest|textinput|bool';
import Controls from 'e:controls';
import CopyLink from 'b:copy-link';
import ErrorText from 'b:error-text';
import i18n from 't:i18n';

const {BEM_LANG} = process.env;

export default decl({
    block: 'cert-request-form',
    elem: 'modal-content',

    mods({submitted}) {
        return {submitted};
    },
    content(props) {
        const {
            isAdditionalShown,
            isAdditionalFieldsConfigFetching,
            additionalFieldsConfigError,
            onFieldTypeChange,
            onFieldCanChange,
            fields,
            additionalFieldsConfig,
            isRequestCertInProgress,
            getCopyLinkURL
        } = props;

        const content = [
            <CopyLink
                key="copy-link"
                getURL={getCopyLinkURL}
                mix={{
                    block: this.block,
                    elem: 'copy-link'
                }}
            />,
            <Title
                key="title"
                level={3}
                size="m"
                font="m"
            >
                {i18n('issue-cert')}
            </Title>,
            <Fields
                key="fields-main"
            >
                <Field
                    type="suggest"
                    title="CA name"
                    onFieldChange={onFieldCanChange}
                    disabled={isRequestCertInProgress}
                    dataprovider={{pathname: '/api/frontend/suggest/ca_names/'}}
                    value={fields.main.ca_name}
                    isRequired
                    slug="ca-name"
                />
                <Field
                    type="suggest"
                    title={i18n('cert-type')}
                    onFieldChange={onFieldTypeChange}
                    disabled
                    dataprovider={{
                        pathname: '/api/frontend/suggest/cert_types/',
                        query: {ca_name: fields.main.ca_name}
                    }}
                    value={fields.main.type}
                    isRequired
                    slug="cert-type"
                />
            </Fields>
        ];

        if (isAdditionalShown) {
            content.push(
                <Fields
                    key="fields-additional"
                    loading={isAdditionalFieldsConfigFetching}
                    additional
                >
                    {additionalFieldsConfig.map(({type, slug, title, suggestPath, isRequired}) => (
                        <Field
                            key={slug}
                            type={type}
                            title={title[BEM_LANG]}
                            value={fields.additional[slug]}
                            dataprovider={{pathname: suggestPath}}
                            onFieldChange={this.getOnFieldAdditionalChange(slug)}
                            disabled={isRequestCertInProgress}
                            isRequired={isRequired}
                            slug={slug}
                        />
                    ))}
                </Fields>
            );
        }

        if (additionalFieldsConfigError) {
            content.push(
                <Title
                    key="additional-title-error"
                    level={6}
                    size="xs"
                    font="xs"
                    mix={{
                        block: this.block,
                        elem: 'additional-title-error'
                    }}
                >
                    {i18n('additional-title-error')}
                </Title>,
                <ErrorText
                    key="fields-additional-error"
                >
                    {JSON.stringify(additionalFieldsConfigError, null, 2)}
                </ErrorText>
            );
        }

        content.push(
            <Controls
                {...this.props}
                key="controls"
            />
        );

        return content;
    },
    getOnFieldAdditionalChange(slug) {
        const {onFieldAdditionalChange} = this.props;

        return (value) => {
            onFieldAdditionalChange(slug, value);
        };
    }
}, {
    propTypes: {
        additionalFieldsConfig: propTypes.arrayOf(
            propTypes.shape({
                isRequired: propTypes.bool.isRequired,
                title: propTypes.shape({
                    [BEM_LANG]: propTypes.string.isRequired
                }).isRequired,
                type: propTypes.string.isRequired,
                slug: propTypes.string.isRequired,
                suggestPath: propTypes.string.isRequired
            })
        ).isRequired,
        additionalFieldsConfigError: propTypes.instanceOf(Error),
        fields: propTypes.shape({
            additional: propTypes.object.isRequired,
            main: propTypes.shape({
                ca_name: propTypes.string.isRequired,
                type: propTypes.string.isRequired
            }).isRequired
        }).isRequired,
        getCopyLinkURL: propTypes.func.isRequired,
        isAdditionalFieldsConfigFetching: propTypes.bool.isRequired,
        isAdditionalShown: propTypes.bool.isRequired,
        isRequestCertInProgress: propTypes.bool.isRequired,
        onFieldAdditionalChange: propTypes.func.isRequired,
        onFieldCanChange: propTypes.func.isRequired,
        onFieldTypeChange: propTypes.func.isRequired
    }
});
