import React from 'react';
import {decl} from 'bem';
import propTypes from 'prop-types';
import CertTable from 'b:certs-table';
import Component from 'b:ta-component';
import debounce from 'lodash/debounce';
import Bluebird from 'bluebird';

export default decl(Component, {
    block: 'certs-table.container',

    willInit() {
        this.__base(...arguments);

        this.state = {
            isFetching: false,
            filters: this.props.getFilters(),
            page: null
        };

        this.handlePagerChange = ::this.handlePagerChange;
        this.handleTrClick = ::this.handleTrClick;
        this.requestCerts = debounce(this.requestCerts, 200);
        this.requestCerts();
    },
    didUpdate() {
        this.__base(...arguments);

        const filters = this.props.getFilters();
        const page = this.getPage();

        const hasDifferenceFilters = Object.entries(filters)
            .some(([key, value]) => this.state.filters[key] !== value);

        // В случае изменения фильтров всегда сбрасываем страницу
        if (hasDifferenceFilters) {
            this.dropPage();
            this.setState({filters, page: null}, this.requestCerts);

            return;
        }

        if (this.state.page !== page) {
            this.setState({page}, this.requestCerts);
        }
    },
    handleTrClick({currentTarget}) {
        const certId = currentTarget.getAttribute('data-id');

        this.props.updateMatchStr('/certificates/:certId', {certId});
    },
    replace() {
        const {isFetching} = this.state;

        return (
            <CertTable
                {...this.props}
                isFetching={isFetching}
                onTrClick={this.handleTrClick}
                onPagerChange={this.handlePagerChange}
            />
        );
    },
    getPage() {
        return this.props.page;
    },
    handlePagerChange({page}) {
        this.props.updateQueryStr({page});
    },
    dropPage() {
        this.props.filterQueryStr((v, k) => k !== 'page');
    },
    async requestCerts() {
        this.setState({isFetching: true});

        try {
            await this.props.requestCerts(Object.assign({page: this.state.page}, this.state.filters));
        } catch (err) {
            if (err instanceof Bluebird.CancellationError) {
                console.warn(err); // eslint-disable-line no-console
                await new Promise(() => {
                    // Подвешиваем catch, чтобы при отмене запроса не вызвался finally
                });
            } else {
                // Todo: http://st/CERTOR-754
                throw err;
            }
        } finally {
            this.setState({isFetching: false});
        }
    }
}, {
    propTypes: {
        filterQueryStr: propTypes.func.isRequired,
        getFilters: propTypes.func.isRequired,
        page: propTypes.string.isRequired,
        requestCerts: propTypes.func.isRequired,
        updateMatchStr: propTypes.func.isRequired,
        updateQueryStr: propTypes.func.isRequired
    }
});
