'use strict';

const path = require('path');
const assert = require('assert');
const {NoEmitOnErrorsPlugin, DefinePlugin} = require('webpack');
const ExtractTextPlugin = require('extract-text-webpack-plugin');
const CopyWebpackPlugin = require('copy-webpack-plugin');
const {getLevels} = require('./tools');

const {BEM_LANG, NODE_ENV} = process.env;

assert(typeof BEM_LANG === 'string', 'Pass BEM_LANG env variable');
assert(typeof NODE_ENV === 'string', 'Pass NODE_ENV env variable');

const getStyleLoaders = (minimize = false, plugins = []) => [
    ExtractTextPlugin.loader(),
    {
        loader: 'css-loader',
        options: {
            minimize
        }
    },
    {
        loader: 'postcss-loader',
        options: {
            ident: 'postcss',
            plugins: () => plugins
        }
    }
];

const styleLoaders = getStyleLoaders();

module.exports.config = {
    entry: ['babel-polyfill', './src/index.js'],
    output: {
        filename: 'index.js',
        path: path.join(__dirname, 'dist', BEM_LANG),
        publicPath: `/${BEM_LANG}`
    },
    stats: {
        children: false
    },
    module: {
        rules: [
            {
                test: /\.css$/,
                use: styleLoaders
            },
            {
                test: /\.styl$/,
                use: [
                    ...styleLoaders,
                    {
                        loader: 'stylus-loader',
                        options: {linenos: true}
                    }
                ]
            },
            {
                test: /\.js$/,
                include: [
                    path.join(__dirname, 'src'),
                    path.join(__dirname, 'node_modules/awesome-icon'),
                    path.join(__dirname, 'node_modules/tools-access-react'),
                    path.join(__dirname, 'node_modules/tools-access-react-redux'),
                    path.join(__dirname, 'node_modules/tools-access-react-redux-router'),
                    path.join(__dirname, 'node_modules/islands')
                ],
                use: [
                    {
                        loader: 'webpack-bem-loader',
                        options: {
                            naming: 'origin',
                            levels: getLevels(),
                            techs: ['js', 'css', 'styl'],
                            techMap: {
                                js: ['react.js']
                            },
                            langs: [BEM_LANG]
                        }
                    },
                    {loader: 'babel-loader'}
                ]
            },
            {
                test: /\.i18n\//,
                use: ['webpack-bem-i18n-loader']
            },
            {
                test: /\.(?:woff2?|ttf|svg|eot|gif|jpg)$/,
                use: [
                    {
                        loader: 'file-loader',
                        options: {name: '/_/[hash].[ext]'}
                    }
                ]
            }
        ]
    },
    plugins: [
        new NoEmitOnErrorsPlugin(),
        new CopyWebpackPlugin([
            'src/favicon.ico',
            'src/index.pug',
            {
                from: 'node_modules/@yandex-int/rum-counter/dist/bundle/implementation.min.js',
                to: 'rum'
            },
            {
                from: 'node_modules/@yandex-int/rum-counter/dist/bundle/onload.min.js',
                to: 'rum'
            },
            {
                from: 'node_modules/@yandex-int/rum-counter/debug/blockstat.js',
                to: 'rum'
            }
        ]),
        new ExtractTextPlugin('index.css'),
        new DefinePlugin({
            'process.env.BEM_LANG': JSON.stringify(BEM_LANG),
            'process.env.NODE_ENV': JSON.stringify(NODE_ENV)
        })
    ],
    resolve: {
        alias: {
            // Waiting for https://github.yandex-team.ru/lego/islands/pull/2304
            bem: require.resolve('./node_modules/islands/common.blocks/i-bem/i-bem.react.js')
        }
    }
};

module.exports.getStyleLoaders = getStyleLoaders;
