package ru.yandex.intranet.d.dao;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.yandex.ydb.table.query.DataQueryResult;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.ListValue;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.StructValue;
import com.yandex.ydb.table.values.Value;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.datasource.impl.YdbQuerySource;
import ru.yandex.intranet.d.datasource.model.WithTxId;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;

/**
 * Abstract DAO with common methods.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 02-04-2021
 */
public abstract class AbstractDao<ModelClass, IdentityClass> {
    protected final YdbQuerySource ydbQuerySource;

    protected AbstractDao(YdbQuerySource ydbQuerySource) {
        this.ydbQuerySource = ydbQuerySource;
    }

    public Mono<Optional<ModelClass>> getById(YdbTxSession session, IdentityClass id, TenantId tenantId) {
        return getByIdStartTx(session, id, tenantId).map(WithTxId::get);
    }

    public Mono<WithTxId<Optional<ModelClass>>> getByIdStartTx(
            YdbTxSession session, IdentityClass id, TenantId tenantId
    ) {
        String query = ydbQuerySource.getQuery(queryKeyPrefix() + ".getById");
        Params params = getIdentityWithTenantParams(id, tenantId);
        return session.executeDataQueryRetryable(query, params).map(r -> new WithTxId<>(toModel(r), r.getTxId()));
    }

    public Mono<List<ModelClass>> getByIds(YdbTxSession session, List<IdentityClass> ids, TenantId tenantId) {
        return getByIds(session,
                ids.stream().map(id -> new WithTenant<>(tenantId, id)).collect(Collectors.toList())
        );
    }

    public Mono<List<ModelClass>> getByIds(YdbTxSession session, List<WithTenant<IdentityClass>> ids) {
        return getByIds(session, toWithTenantsListValue(ids)).map(WithTxId::get);
    }

    public Mono<WithTxId<List<ModelClass>>> getByIdsStartTx(
            YdbTxSession session, List<IdentityClass> ids, TenantId tenantId
    ) {
        ListValue idsValue = toWithTenantsListValue(
                ids.stream().map(id -> new WithTenant<>(tenantId, id)).collect(Collectors.toList())
        );
        return getByIds(session, idsValue);
    }

    public Mono<ModelClass> upsertOneRetryable(YdbTxSession session, ModelClass model) {
        return upsertOneTxRetryable(session, model).map(WithTxId::get);
    }

    public Mono<WithTxId<ModelClass>> upsertOneTxRetryable(YdbTxSession session, ModelClass model) {
        String queryKey = queryKeyPrefix() + ".upsertOne";
        String query = ydbQuerySource.getQuery(queryKey);
        Params params = Params.of("$data", toStructValue(model));
        return session.executeDataQueryRetryable(query, params)
                .map(result -> new WithTxId<>(model, result.getTxId()))
                .onErrorMap(error -> new YQLException("Failed YQL " + queryKey + ":\n" + query, error));
    }

    public Mono<Void> upsertAllRetryable(YdbTxSession session, List<ModelClass> models) {
        if (models.isEmpty()) {
            return Mono.empty();
        }
        String query = ydbQuerySource.getQuery(queryKeyPrefix() + ".upsertAll");
        Params params = Params.of("$dataList", toListValue(models));
        return session.executeDataQueryRetryable(query, params).then();
    }

    protected Params getIdentityWithTenantParams(IdentityClass id, TenantId tenantId) {
        return getIdentityParams(id)
                .put("$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
    }

    protected StructValue getIdentityWithTenantStructValue(IdentityClass id, TenantId tenantId) {
        Map<String, Value<?>> params = getIdentityWithTenantParams(id, tenantId).values();
        Map<String, Value> members = new HashMap<>(params.size());
        params.forEach((name, value) -> members.put(name.substring(1), value));
        return StructValue.of(members);
    }

    protected ListValue toWithTenantsListValue(List<WithTenant<IdentityClass>> ids) {
        StructValue[] values = ids.stream().map(id ->
                getIdentityWithTenantStructValue(id.getIdentity(), id.getTenantId())
        ).toArray(StructValue[]::new);
        return ListValue.of(values);
    }

    protected Mono<WithTxId<List<ModelClass>>> getByIds(YdbTxSession session, ListValue ids) {
        if (ids.isEmpty()) {
            return Mono.just(new WithTxId<>(List.of(), null));
        }
        String query = ydbQuerySource.getQuery(queryKeyPrefix() + ".getByIds");
        Params params = Params.of("$ids", ids);
        return session.executeDataQueryRetryable(query, params)
                .map(result -> new WithTxId<>(toModels(result), result.getTxId()));
    }

    protected StructValue toStructValue(ModelClass model) {
        return StructValue.of(prepareFieldValues(model));
    }

    protected ListValue toListValue(List<ModelClass> models) {
        return ListValue.of(models.stream().map(this::toStructValue).toArray(StructValue[]::new));
    }

    protected Optional<ModelClass> toModel(DataQueryResult result) {
        if (result.isEmpty()) {
            return Optional.empty();
        }
        if (result.getResultSetCount() > 1) {
            throw new IllegalStateException("Too many result sets");
        }
        ResultSetReader reader = result.getResultSet(0);
        return toModel(reader);
    }

    protected Optional<ModelClass> toModel(DataQueryResult result, int resultSetIndex) {
        if (result.isEmpty() || resultSetIndex >= result.getResultSetCount()) {
            throw new IllegalStateException("Too few result sets");
        }
        ResultSetReader reader = result.getResultSet(resultSetIndex);
        return toModel(reader);
    }

    protected <T> Optional<T> toModel(DataQueryResult result, int resultSetIndex,
                                      Function<ResultSetReader, T> readOneRow) {
        if (result.isEmpty() || resultSetIndex >= result.getResultSetCount()) {
            throw new IllegalStateException("Too few result sets");
        }
        ResultSetReader reader = result.getResultSet(resultSetIndex);
        return toModel(reader, readOneRow);
    }

    protected Optional<ModelClass> toModel(ResultSetReader reader) {
        if (!reader.next()) {
            return Optional.empty();
        }
        if (reader.getRowCount() > 1) {
            throw new IllegalStateException("Non unique resource");
        }
        ModelClass model = readOneRow(reader, new HashMap<>());
        return Optional.of(model);
    }

    protected <T> Optional<T> toModel(ResultSetReader reader, Function<ResultSetReader, T> readOneRow) {
        if (!reader.next()) {
            return Optional.empty();
        }
        if (reader.getRowCount() > 1) {
            throw new IllegalStateException("Non unique resource");
        }
        T model = readOneRow.apply(reader);
        return Optional.of(model);
    }

    protected List<ModelClass> toModels(DataQueryResult result) {
        if (result.isEmpty()) {
            return List.of();
        }
        if (result.getResultSetCount() > 1) {
            throw new IllegalStateException("Too many result sets");
        }
        ResultSetReader reader = result.getResultSet(0);
        return toModels(reader);
    }

    protected List<ModelClass> toModels(DataQueryResult result, int resultSetIndex) {
        if (result.isEmpty() || resultSetIndex >= result.getResultSetCount()) {
            throw new IllegalStateException("Too few result sets");
        }
        ResultSetReader reader = result.getResultSet(resultSetIndex);
        return toModels(reader);
    }

    protected <T> List<T> toModels(DataQueryResult result, int resultSetIndex,
                                   Function<ResultSetReader, T> readOneRow) {
        if (result.isEmpty() || resultSetIndex >= result.getResultSetCount()) {
            throw new IllegalStateException("Too few result sets");
        }
        ResultSetReader reader = result.getResultSet(resultSetIndex);
        return toModels(reader, readOneRow);
    }

    protected List<ModelClass> toModels(ResultSetReader reader) {
        List<ModelClass> list = new ArrayList<>();
        Map<String, TenantId> tenantIdCache = new HashMap<>();
        while (reader.next()) {
            list.add(readOneRow(reader, tenantIdCache));
        }
        return list;
    }

    protected <T> List<T> toModels(ResultSetReader reader, Function<ResultSetReader, T> readOneRow) {
        List<T> list = new ArrayList<>();
        while (reader.next()) {
            list.add(readOneRow.apply(reader));
        }
        return list;
    }

    protected abstract WithTenant<IdentityClass> getIdentityWithTenant(ModelClass model);

    protected abstract Params getIdentityParams(IdentityClass id);

    @SuppressWarnings("rawtypes")
    protected abstract Map<String, Value> prepareFieldValues(ModelClass model);

    protected abstract ModelClass readOneRow(ResultSetReader reader, Map<String, TenantId> tenantIdCache);

    protected abstract String queryKeyPrefix();
}
