package ru.yandex.intranet.d.dao;

import java.util.List;
import java.util.stream.Collectors;

import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.values.ListValue;
import com.yandex.ydb.table.values.PrimitiveValue;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.datasource.impl.YdbQuerySource;
import ru.yandex.intranet.d.datasource.model.WithTxId;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;

/**
 * Abstract DAO with common methods and soft remove methods.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 20.10.2020
 */
public abstract class AbstractDaoWithSoftRemove<ModelClass, IdentityClass>
        extends AbstractDao<ModelClass, IdentityClass> {

    protected AbstractDaoWithSoftRemove(YdbQuerySource ydbQuerySource) {
        super(ydbQuerySource);
    }

    public Mono<Void> removeRetryable(YdbTxSession session, IdentityClass id, TenantId tenantId, long newVersion) {
        final Params params = getIdentityWithTenantParams(id, tenantId)
                .put("$new_version", PrimitiveValue.int64(newVersion));
        String query = ydbQuerySource.getQuery(queryKeyPrefix() + ".remove");
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<Void> removeRetryable(YdbTxSession session, WithTenant<IdentityClass> idWithTenant, long newVersion) {
        return removeRetryable(session, idWithTenant.getIdentity(), idWithTenant.getTenantId(), newVersion);
    }

    public Mono<Void> removeRetryable(YdbTxSession session, ModelClass model, long newVersion) {
        return removeRetryable(session, getIdentityWithTenant(model), newVersion);
    }

    public Mono<WithTxId<Void>> removeAllRetryable(YdbTxSession session, List<IdentityClass> ids, TenantId tenantId) {
        return removeAllRetryable(session,
                ids.stream().map(id -> new WithTenant<>(tenantId, id)).collect(Collectors.toList())
        );
    }

    public Mono<WithTxId<Void>> removeAllRetryable(YdbTxSession session, List<WithTenant<IdentityClass>> ids) {
        return removeAllRetryable(session, toWithTenantsListValue(ids));
    }

    public Mono<WithTxId<Void>> removeAllModelsRetryable(YdbTxSession session, List<ModelClass> models) {
        return removeAllRetryable(session,
                models.stream().map(this::getIdentityWithTenant).collect(Collectors.toList())
        );
    }

    protected Mono<WithTxId<Void>> removeAllRetryable(YdbTxSession session, ListValue ids) {
        if (ids.isEmpty()) {
            return Mono.empty();
        }
        String query = ydbQuerySource.getQuery(queryKeyPrefix() + ".removeAll");
        Params params = Params.of("$ids", ids);
        return session.executeDataQueryRetryable(query, params).map(v -> new WithTxId<>(null, v.getTxId()));
    }

}
