package ru.yandex.intranet.d.dao

import com.yandex.ydb.table.query.Params
import com.yandex.ydb.table.result.ResultSetReader
import kotlinx.coroutines.reactor.awaitSingle
import ru.yandex.intranet.d.datasource.model.YdbTxSession
import java.util.*
import java.util.concurrent.atomic.AtomicReference

/**
 * DAO pagination.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
class DaoPagination private constructor() {
    companion object {
        suspend fun <T> getAllPages(session: YdbTxSession,
                                    firstPageQuerySupplier: () -> String,
                                    nextPageQuerySupplier: () -> String,
                                    firstPageParamsSupplier: (limit: Long) -> Params,
                                    nextPageParamsSupplier: (limit: Long, lastOnPreviousPage: T) -> Params,
                                    rowReader: (reader: ResultSetReader) -> T,
                                    perPage: Long = 1000): List<T> {
            val result = Collections.synchronizedList(mutableListOf<T>())
            val firstPageQuery = firstPageQuerySupplier()
            val nextPageQuery = nextPageQuerySupplier()
            val firstPageParams = firstPageParamsSupplier(perPage)
            val firstPage = DaoReader.toModels(session.executeDataQueryRetryable(firstPageQuery, firstPageParams)
                .awaitSingle(), rowReader)
            result.addAll(firstPage)
            if (firstPage.size < perPage) {
                return result.toList()
            }
            val lastOnPage = AtomicReference(firstPage.lastOrNull())
            while (true) {
                val currentLastOnPage = lastOnPage.get() ?: break
                val nextPageParams = nextPageParamsSupplier(perPage, currentLastOnPage)
                val nextPage = DaoReader.toModels(session.executeDataQueryRetryable(nextPageQuery, nextPageParams)
                    .awaitSingle(), rowReader)
                result.addAll(nextPage)
                lastOnPage.set(nextPage.lastOrNull())
                if (nextPage.size < perPage) {
                    break
                }
            }
            return result.toList()
        }
        suspend fun <T> getAllPages(session: YdbTxSession,
                                    firstPageQuerySupplier: () -> String,
                                    nextPageQuerySupplier: () -> String,
                                    lastPageQuerySupplier: () -> String,
                                    firstPageParamsSupplier: (limit: Long) -> Params,
                                    nextPageParamsSupplier: (limit: Long, lastOnPreviousPage: T) -> Params,
                                    lastPageParamsSupplier: (limit: Long, lastOnPreviousPage: T) -> Params,
                                    lastPagePredicate: (lastOnPreviousPage: T) -> Boolean,
                                    rowReader: (reader: ResultSetReader) -> T,
                                    perPage: Long = 1000): List<T> {
            val result = Collections.synchronizedList(mutableListOf<T>())
            val firstPageQuery = firstPageQuerySupplier()
            val nextPageQuery = nextPageQuerySupplier()
            val lastPageQuery = lastPageQuerySupplier()
            val firstPageParams = firstPageParamsSupplier(perPage)
            val firstPage = DaoReader.toModels(session.executeDataQueryRetryable(firstPageQuery, firstPageParams)
                .awaitSingle(), rowReader)
            result.addAll(firstPage)
            if (firstPage.size < perPage) {
                return result.toList()
            }
            val lastOnPage = AtomicReference(firstPage.lastOrNull())
            while (true) {
                val currentLastOnPage = lastOnPage.get() ?: break
                val (currentParams, currentQuery) = if (lastPagePredicate(currentLastOnPage)) {
                    Pair(lastPageParamsSupplier(perPage, currentLastOnPage), lastPageQuery)
                } else {
                    Pair(nextPageParamsSupplier(perPage, currentLastOnPage), nextPageQuery)
                }
                val currentPage = DaoReader.toModels(session.executeDataQueryRetryable(currentQuery, currentParams)
                    .awaitSingle(), rowReader)
                result.addAll(currentPage)
                lastOnPage.set(currentPage.lastOrNull())
                if (currentPage.size < perPage) {
                    break
                }
            }
            return result.toList()
        }
    }
}
