package ru.yandex.intranet.d.dao;

import java.io.UncheckedIOException;

import javax.annotation.Nullable;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.yandex.ydb.table.result.ValueReader;
import com.yandex.ydb.table.values.OptionalType;
import com.yandex.ydb.table.values.OptionalValue;
import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.PrimitiveValue;

import ru.yandex.intranet.d.util.ObjectMapperHolder;

/**
 * Helper for JSON field.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 09.12.2020
 */
public class JsonFieldHelper<T> {
    private final ObjectReader objectReader;
    private final ObjectWriter objectWriter;

    public JsonFieldHelper(
            ObjectMapperHolder objectMapper,
            TypeReference<T> typeReference
    ) {
        ObjectMapper mapper = objectMapper.getObjectMapper();
        this.objectReader = mapper.readerFor(typeReference);
        this.objectWriter = mapper.writerFor(typeReference);
    }

    public JsonFieldHelper(
            ObjectMapperHolder objectMapper,
            Class<T> clazz
    ) {
        ObjectMapper mapper = objectMapper.getObjectMapper();
        this.objectReader = mapper.readerFor(clazz);
        this.objectWriter = mapper.writerFor(clazz);
    }

    @Nullable
    public T read(ValueReader reader) {
        if (!reader.isOptionalItemPresent()) {
            return null;
        }
        String json = reader.getJsonDocument();
        try {
            return objectReader.readValue(json);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }

    public OptionalValue writeOptional(@Nullable T value) {
        OptionalType type = OptionalType.of(PrimitiveType.jsonDocument());
        if (value == null) {
            return type.emptyValue();
        }
        return type.newValue(write(value));
    }

    public PrimitiveValue write(T value) {
        try {
            String json = objectWriter.writeValueAsString(value);
            return PrimitiveValue.jsonDocument(json);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }

    public OptionalValue writeOptionalJson(@Nullable T value) {
        OptionalType type = OptionalType.of(PrimitiveType.json());
        if (value == null) {
            return type.emptyValue();
        }
        return type.newValue(write(value));
    }
}
