package ru.yandex.intranet.d.dao;

import java.util.List;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.yandex.ydb.table.query.DataQueryResult;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.ListValue;
import com.yandex.ydb.table.values.PrimitiveValue;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.datasource.model.WithTruncatedFlag;
import ru.yandex.intranet.d.datasource.model.WithTxId;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;

/**
 * Common query behaviours
 *
 * @author Denis Blokhin <denblo@yandex-team.ru>
 */
public class QueryUtils {
    private QueryUtils() {
    }

    public static ListValue toDeletedParam(boolean withDeleted) {
        return withDeleted
                ? ListValue.of(PrimitiveValue.bool(true), PrimitiveValue.bool(false))
                : ListValue.of(PrimitiveValue.bool(false));
    }

    public static <T, K> Mono<WithTxId<List<T>>> getAllRows(
            YdbTxSession session,
            BiFunction<YdbTxSession, K, Mono<DataQueryResult>> query,
            Function<ResultSetReader, List<T>> convert,
            Function<T, K> keyExtractor
    ) {
        YdbTxSession startSession = session; // todo: don't commit first query (if read-write transaction)
        return getRowsAsManyAsPossible(startSession, null, query, convert)
                .expand(r -> {
                    if (r.isTruncated()) {
                        YdbTxSession txSession = session; // todo: use result.getTransactionId()
                        List<T> models = r.get().get();
                        T lastRow = models.get(models.size() - 1);
                        return getRowsAsManyAsPossible(txSession, keyExtractor.apply(lastRow), query, convert);
                    }
                    // todo: finally commit transaction, if (session.getTxControl().isCommitTx())
                    return Mono.empty();
                })
                .reduce((a, b) -> new WithTruncatedFlag<>(
                        new WithTxId<>(
                                Stream.concat(
                                        a.get().get().stream(), b.get().get().stream()
                                ).collect(Collectors.toList()),
                                b.get().getTransactionId()
                        ),
                        b.isTruncated()
                ))
                .map(WithTruncatedFlag::get);
    }

    private static <T, K> Mono<WithTruncatedFlag<WithTxId<List<T>>>> getRowsAsManyAsPossible(
            YdbTxSession session,
            K lastId,
            BiFunction<YdbTxSession, K, Mono<DataQueryResult>> query,
            Function<ResultSetReader, List<T>> convert
    ) {
        return query.apply(session, lastId).map(result -> {
            ResultSetReader reader = toReader(result);
            boolean isTruncated = reader != null && reader.isTruncated();
            List<T> models = convert.apply(reader);
            return new WithTruncatedFlag<>(new WithTxId<>(models, result.getTxId()), isTruncated);
        });
    }

    private static ResultSetReader toReader(DataQueryResult result) {
        if (result.isEmpty()) {
            return null;
        }
        if (result.getResultSetCount() > 1) {
            throw new IllegalStateException("Too many result sets");
        }
        return result.getResultSet(0);
    }
}
