package ru.yandex.intranet.d.dao.accounts;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import com.yandex.ydb.table.query.DataQueryResult;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.ListValue;
import com.yandex.ydb.table.values.OptionalValue;
import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.TupleType;
import com.yandex.ydb.table.values.TupleValue;
import com.yandex.ydb.table.values.Value;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuple3;
import reactor.util.function.Tuples;
import ru.yandex.intranet.d.dao.AbstractDaoWithStringId;
import ru.yandex.intranet.d.dao.QueryUtils;
import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.datasource.Ydb;
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource;
import ru.yandex.intranet.d.datasource.model.WithTxId;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.StringIdWithTenant;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountReserveType;
import ru.yandex.intranet.d.model.accounts.ServiceAccountKeys;

import static com.yandex.ydb.table.values.PrimitiveValue.bool;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static ru.yandex.intranet.d.dao.accounts.AccountsDao.Fields.ACCOUNTS_SPACES_ID;
import static ru.yandex.intranet.d.dao.accounts.AccountsDao.Fields.DELETED;
import static ru.yandex.intranet.d.dao.accounts.AccountsDao.Fields.DISPLAY_NAME;
import static ru.yandex.intranet.d.dao.accounts.AccountsDao.Fields.FOLDER_ID;
import static ru.yandex.intranet.d.dao.accounts.AccountsDao.Fields.FREE_TIER;
import static ru.yandex.intranet.d.dao.accounts.AccountsDao.Fields.ID;
import static ru.yandex.intranet.d.dao.accounts.AccountsDao.Fields.OUTER_ACCOUNT_ID_IN_PROVIDER;
import static ru.yandex.intranet.d.dao.accounts.AccountsDao.Fields.OUTER_ACCOUNT_KEY_IN_PROVIDER;
import static ru.yandex.intranet.d.dao.accounts.AccountsDao.Fields.PROVIDER_ID;
import static ru.yandex.intranet.d.dao.accounts.AccountsDao.Fields.RESERVE_TYPE;
import static ru.yandex.intranet.d.dao.accounts.AccountsDao.Fields.TENANT_ID;
import static ru.yandex.intranet.d.dao.accounts.AccountsDao.Fields.VERSION;
import static ru.yandex.intranet.d.datasource.Ydb.nullableUtf8;
import static ru.yandex.intranet.d.datasource.Ydb.nullableValue;


/**
 * Accounts DAO.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 15.10.2020
 */
@Component
public class AccountsDao extends AbstractDaoWithStringId<AccountModel> {

    protected AccountsDao(YdbQuerySource ydbQuerySource) {
        super(ydbQuerySource);
    }

    public Mono<List<AccountModel>> getAllByFolderIds(
            YdbTxSession session, List<WithTenant<String>> folderIds, boolean withDeleted
    ) {
        if (folderIds.isEmpty()) {
            return Mono.just(List.of());
        }
        String query = ydbQuerySource.getQuery("yql.queries.accounts.getAllByFolderIds");
        return QueryUtils.getAllRows(session,
                (nextSession, lastId) -> {
                    ListValue idsParam = ListValue.of(folderIds.stream().map(id ->
                            TupleValue.of(utf8(id.getTenantId().getId()),
                                    utf8(id.getIdentity())))
                            .toArray(TupleValue[]::new));
                    Params params = Params.of(
                            "$folderIds", idsParam,
                            "$include_deleted", bool(withDeleted),
                            "$from_id", nullableUtf8(lastId)
                    );
                    return session.executeDataQueryRetryable(query, params);
                },
                this::toModels,
                AccountModel::getId
        ).map(WithTxId::get);
    }

    public Mono<Optional<AccountModel>> getByIdWithDeleted(YdbTxSession session, String id, TenantId tenantId) {
        String query = ydbQuerySource.getQuery("yql.queries.accounts.getByIdWithDeleted");
        Params params = Params.of("$id", PrimitiveValue.utf8(id),
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId())
        );
        return session.executeDataQueryRetryable(query, params).map(this::toModel);
    }

    public Mono<List<AccountModel>> getByIdsWithDeleted(YdbTxSession session, List<WithTenant<String>> ids) {
        if (ids.isEmpty()) {
            return Mono.just(List.of());
        }
        List<StringIdWithTenant> stringIds = ids.stream()
                .map(id -> new StringIdWithTenant(id.getTenantId(), id.getIdentity()))
                .collect(Collectors.toList());

        String query = ydbQuerySource.getQuery(queryKeyPrefix() + ".getByIdsWithDeleted");
        Params params = Params.of("$ids", Ydb.toIdWithTenantsListValue(stringIds));
        return session.executeDataQueryRetryable(query, params).map(this::toModels);
    }

    public Mono<List<AccountModel>> getAllByIdsWithDeleted(
            YdbTxSession session, List<String> ids, TenantId tenantId) {
        if (ids.isEmpty()) {
            return Mono.just(List.of());
        }
        PrimitiveValue tenantIdValue = utf8(tenantId.getId());
        ListValue idsValue = ListValue.of(
                ids.stream().map(id -> TupleValue.of(utf8(id), tenantIdValue)).toArray(TupleValue[]::new)
        );
        TupleType fromKeyType = TupleType.of(List.of(PrimitiveType.utf8(), PrimitiveType.utf8()));
        String query = ydbQuerySource.getQuery(queryKeyPrefix() + ".getAllByIdsWithDeleted");
        return QueryUtils.getAllRows(session, (nextSession, lastId) -> {
                    OptionalValue fromId = nullableValue(fromKeyType,
                    lastId == null ? null : TupleValue.of(List.of(tenantIdValue, utf8(lastId))));
                    Params params = Params.of(
                            "$ids", idsValue,
                            "$from_key", fromId);
                    return nextSession.executeDataQueryRetryable(query, params);
                },
                this::toModels,
                AccountModel::getId).map(WithTxId::get);
    }

    public Mono<Optional<AccountModel>> getAllByExternalId(YdbTxSession session,
                                                           WithTenant<AccountModel.ExternalId> id) {
        String query = ydbQuerySource.getQuery("yql.queries.accounts.getAllByExternalId");
        Params params = Params.of("$provider_id", PrimitiveValue.utf8(id.getIdentity().getProviderId()),
                "$outer_account_id_in_provider", PrimitiveValue.utf8(id.getIdentity().getOuterAccountIdInProvider()),
                "$tenant_id", PrimitiveValue.utf8(id.getTenantId().getId()),
                "$accounts_spaces_id", Ydb.nullToEmptyUtf8(id.getIdentity().getAccountsSpacesId())
        );
        return session.executeDataQueryRetryable(query, params).map(this::toModel);
    }

    public Mono<List<AccountModel>> getAllByExternalIds(YdbTxSession session,
                                                        List<WithTenant<AccountModel.ExternalId>> ids) {
        if (ids.isEmpty()) {
            return Mono.just(List.of());
        }
        String query = ydbQuerySource.getQuery("yql.queries.accounts.getAllByExternalIds");
        ListValue idsParam = ListValue.of(ids.stream().map(id -> TupleValue.of(
                PrimitiveValue.utf8(id.getTenantId().getId()),
                PrimitiveValue.utf8(id.getIdentity().getProviderId()),
                PrimitiveValue.utf8(id.getIdentity().getOuterAccountIdInProvider()),
                Ydb.nullToEmptyUtf8(id.getIdentity().getAccountsSpacesId())
        )).toArray(TupleValue[]::new));
        Params params = Params.of("$ids", idsParam);
        return session.executeDataQueryRetryable(query, params).map(this::toModels);
    }

    public Mono<Map<FolderIdProviderId, Long>> countByFolderIdProviderId(YdbTxSession session,
                                                                         List<WithTenant<FolderIdProviderId>> ids) {
        if (ids.isEmpty()) {
            return Mono.just(Map.of());
        }
        String query = ydbQuerySource.getQuery("yql.queries.accounts.countByFoldersAndProviders");
        ListValue idsParam = ListValue.of(ids.stream().map(id ->
                TupleValue.of(PrimitiveValue.utf8(id.getTenantId().getId()),
                        PrimitiveValue.utf8(id.getIdentity().getFolderId()),
                        PrimitiveValue.utf8(id.getIdentity().getProviderId())))
                .toArray(TupleValue[]::new));
        Params params = Params.of("$ids", idsParam);
        return session.executeDataQueryRetryable(query, params).map(this::toCounts);
    }

    public Mono<List<AccountModel>> getByFoldersForProvider(YdbTxSession session, TenantId tenantId,
                                                            String providerId, Set<String> folderIds,
                                                            String accountsSpaceId) {
       return getByFoldersForProvider(session, tenantId, providerId, folderIds, accountsSpaceId, true);
    }

    public Mono<List<AccountModel>> getByFoldersForProvider(YdbTxSession session, TenantId tenantId,
                                                            String providerId, Set<String> folderIds,
                                                            String accountsSpaceId, boolean withDeleted) {
        if (folderIds.isEmpty()) {
            return Mono.just(List.of());
        }
        List<String> sortedFolderIds = folderIds.stream().sorted().collect(Collectors.toList());
        String firstPageQuery = ydbQuerySource.getQuery("yql.queries.accounts.getByFoldersForProviderFirstPage");
        var tenantIdValue = PrimitiveValue.utf8(tenantId.getId());
        var providerIdValue = PrimitiveValue.utf8(providerId);
        var accountsSpacesIdValue = Ydb.nullToEmptyUtf8(Optional.ofNullable(accountsSpaceId));
        ListValue complexIds = ListValue.of(folderIds.stream().sorted().map(folderId -> TupleValue.of(
                tenantIdValue, utf8(folderId), providerIdValue, accountsSpacesIdValue
        )).toArray(TupleValue[]::new));
        HashMap<String, Value<?>> paramsMap = new HashMap<>();
        paramsMap.put("$complex_ids", complexIds);
        paramsMap.put("$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS));
        paramsMap.put("$include_deleted", PrimitiveValue.bool(withDeleted));
        Params firstPageParams = Params.copyOf(paramsMap);
        return session.executeDataQueryRetryable(firstPageQuery, firstPageParams).flatMap(firstPageResult -> {
            List<AccountModel> accounts = toModels(firstPageResult);
            if (!firstPageResult.getResultSet(0).isTruncated() && accounts.size() < Ydb.MAX_RESPONSE_ROWS) {
                return Mono.just(accounts);
            }
            return getNextPage(session, tenantId, sortedFolderIds, providerId, accountsSpaceId,
                    accounts.get(accounts.size() - 1), withDeleted)
                    .expand(tuple -> {
                        if (!tuple.getT2() && tuple.getT1().size() < Ydb.MAX_RESPONSE_ROWS) {
                            return Mono.empty();
                        } else {
                            return getNextPage(session, tenantId, sortedFolderIds, providerId, accountsSpaceId,
                                    tuple.getT1().get(tuple.getT1().size() - 1), withDeleted);
                        }
            }).map(Tuple2::getT1).reduce(accounts, (l, r) -> Stream.concat(l.stream(), r.stream())
                    .collect(Collectors.toList()));
        });
    }

    public Mono<List<AccountModel>> getByFoldersForProvider(YdbTxSession session, TenantId tenantId,
                                                            String providerId, String folderId, boolean withDeleted) {
        String firstPageQuery = ydbQuerySource.getQuery("yql.queries.accounts.getByFolderAndProviderFirstPage");
        HashMap<String, Value<?>> paramsMap = new HashMap<>();
        paramsMap.put("$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        paramsMap.put("$folder_id", PrimitiveValue.utf8(folderId));
        paramsMap.put("$provider_id", PrimitiveValue.utf8(providerId));
        paramsMap.put("$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS));
        paramsMap.put("$include_deleted", PrimitiveValue.bool(withDeleted));
        Params firstPageParams = Params.copyOf(paramsMap);
        return session.executeDataQueryRetryable(firstPageQuery, firstPageParams).flatMap(firstPageResult -> {
            List<AccountModel> accounts = toModels(firstPageResult);
            if (!firstPageResult.getResultSet(0).isTruncated() && accounts.size() < Ydb.MAX_RESPONSE_ROWS) {
                return Mono.just(accounts);
            }
            return getNextPage(session, tenantId, folderId, providerId, accounts.get(accounts.size() - 1), withDeleted)
                    .expand(tuple -> {
                        if (!tuple.getT2() && tuple.getT1().size() < Ydb.MAX_RESPONSE_ROWS) {
                            return Mono.empty();
                        } else {
                            return getNextPage(session, tenantId, folderId, providerId,
                                    tuple.getT1().get(tuple.getT1().size() - 1), withDeleted);
                        }
                    }).map(Tuple2::getT1).reduce(accounts, (l, r) -> Stream.concat(l.stream(), r.stream())
                            .collect(Collectors.toList()));
        });
    }

    public Mono<List<AccountModel>> getAllNonDeletedByAccountsSpaceExcluding(YdbTxSession session, TenantId tenantId,
                                                                             String providerId, String accountsSpaceId,
                                                                             Set<String> externalAccountIdsToExclude) {
        String firstPageQuery = ydbQuerySource
                .getQuery("yql.queries.accounts.getNonDeletedByProviderAccountsSpaceFirstPage");
        HashMap<String, Value<?>> paramsMap = new HashMap<>();
        paramsMap.put("$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        paramsMap.put("$provider_id", PrimitiveValue.utf8(providerId));
        paramsMap.put("$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS));
        paramsMap.put("$accounts_spaces_id", Ydb.nullToEmptyUtf8(Optional.ofNullable(accountsSpaceId)));
        Params firstPageParams = Params.copyOf(paramsMap);
        return session.executeDataQueryRetryable(firstPageQuery, firstPageParams).flatMap(firstPageResult -> {
            List<AccountModel> accounts = toModels(firstPageResult, 0);
            long size = toModel(firstPageResult, 1, this::toSize).orElse(0L);
            Optional<AccountModel> lastAccount = toModel(firstPageResult, 2);
            List<AccountModel> filteredAccounts = accounts.stream()
                    .filter(a -> !externalAccountIdsToExclude.contains(a.getOuterAccountIdInProvider()))
                    .collect(Collectors.toList());
            if (lastAccount.isEmpty() || size < Ydb.MAX_RESPONSE_ROWS) {
                return Mono.just(filteredAccounts);
            }
            return getNextPage(session, tenantId, providerId, accountsSpaceId, lastAccount.get(),
                    externalAccountIdsToExclude)
                    .expand(tuple -> {
                        if (tuple.getT2().isEmpty() || tuple.getT3() < Ydb.MAX_RESPONSE_ROWS) {
                            return Mono.empty();
                        } else {
                            return getNextPage(session, tenantId, providerId, accountsSpaceId, tuple.getT2().get(),
                                    externalAccountIdsToExclude);
                        }
                    }).map(Tuple2::getT1).reduce(filteredAccounts, (l, r) -> Stream.concat(l.stream(), r.stream())
                            .collect(Collectors.toList()));
        });
    }

    public Mono<List<AccountModel>> getAllByTenant(YdbTxSession session, TenantId tenantId) {
        String firstPageQuery = ydbQuerySource.getQuery(queryKeyPrefix() + ".getByTenantFirstPage");
        Params firstPageParams = Params.of("$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS));
        return session.executeDataQueryRetryable(firstPageQuery, firstPageParams).flatMap(firstPageResult -> {
            List<AccountModel> models = toModels(firstPageResult);
            if (!firstPageResult.getResultSet(0).isTruncated() && models.size() < Ydb.MAX_RESPONSE_ROWS) {
                return Mono.just(models);
            }
            return getNextPageByTenant(session, tenantId, models.get(models.size() - 1)).expand(tuple -> {
                if (!tuple.getT2() && tuple.getT1().size() < Ydb.MAX_RESPONSE_ROWS) {
                    return Mono.empty();
                } else {
                    return getNextPageByTenant(session, tenantId, tuple.getT1().get(tuple.getT1().size() - 1));
                }
            }).map(Tuple2::getT1).reduce(models, (l, r) -> Stream.concat(l.stream(), r.stream())
                    .collect(Collectors.toList()));
        });
    }

    @SuppressWarnings("ParameterNumber")
    public Mono<List<AccountModel>> getByFolderAndProvider(YdbTxSession session, TenantId tenantId,
                                                           String folderId, String providerId,
                                                           AccountByFolderAndProviderPagingFrom from, int limit,
                                                           boolean withDeleted) {
        String query;
        Params params;
        if (from == null) {
            query = ydbQuerySource.getQuery("yql.queries.accounts.getByFolderAndProviderFirstPage");
            HashMap<String, Value<?>> paramsMap = new HashMap<>();
            paramsMap.put("$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
            paramsMap.put("$folder_id", PrimitiveValue.utf8(folderId));
            paramsMap.put("$provider_id", PrimitiveValue.utf8(providerId));
            paramsMap.put("$limit", PrimitiveValue.uint64(limit));
            paramsMap.put("$include_deleted", PrimitiveValue.bool(withDeleted));
            params = Params.copyOf(paramsMap);
        } else {
            query = ydbQuerySource.getQuery("yql.queries.accounts.getByFolderAndProviderNextPage");
            Map<String, Value<?>> paramsMap = new HashMap<>();
            paramsMap.put("$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
            paramsMap.put("$folder_id", PrimitiveValue.utf8(folderId));
            paramsMap.put("$provider_id", PrimitiveValue.utf8(providerId));
            paramsMap.put("$from_accounts_spaces_id", Ydb.nullToEmptyUtf8(from.getAccountsSpaceId()));
            paramsMap.put("$from_id", PrimitiveValue.utf8(from.getAccountId()));
            paramsMap.put("$limit", PrimitiveValue.uint64(limit));
            paramsMap.put("$include_deleted", PrimitiveValue.bool(withDeleted));
            params = Params.copyOf(paramsMap);
        }
        return session.executeDataQueryRetryable(query, params).map(this::toModels);
    }

    @SuppressWarnings("ParameterNumber")
    public Mono<List<AccountModel>> getByFolder(YdbTxSession session, TenantId tenantId, String folderId,
                                                AccountByFolderPagingFrom from, int limit, boolean withDeleted) {
        String query;
        Params params;
        if (from == null) {
            query = ydbQuerySource.getQuery("yql.queries.accounts.getByFolderFirstPage");
            HashMap<String, Value<?>> paramsMap = new HashMap<>();
            paramsMap.put("$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
            paramsMap.put("$folder_id", PrimitiveValue.utf8(folderId));
            paramsMap.put("$limit", PrimitiveValue.uint64(limit));
            paramsMap.put("$include_deleted", PrimitiveValue.bool(withDeleted));
            params = Params.copyOf(paramsMap);
        } else {
            query = ydbQuerySource.getQuery("yql.queries.accounts.getByFolderNextPage");
            Map<String, Value<?>> paramsMap = new HashMap<>();
            paramsMap.put("$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
            paramsMap.put("$folder_id", PrimitiveValue.utf8(folderId));
            paramsMap.put("$from_provider_id", PrimitiveValue.utf8(from.getProviderId()));
            paramsMap.put("$from_accounts_spaces_id", Ydb.nullToEmptyUtf8(from.getAccountsSpaceId()));
            paramsMap.put("$from_id", PrimitiveValue.utf8(from.getAccountId()));
            paramsMap.put("$limit", PrimitiveValue.uint64(limit));
            paramsMap.put("$include_deleted", PrimitiveValue.bool(withDeleted));
            params = Params.copyOf(paramsMap);
        }
        return session.executeDataQueryRetryable(query, params).map(this::toModels);
    }

    public Mono<List<AccountModel>> getAllByFoldersProvidersAccountsSpaces(YdbTxSession session,
                                                                           Set<FolderProviderAccountsSpace> ids,
                                                                           boolean withDeleted) {
        if (ids.isEmpty()) {
            return Mono.just(List.of());
        }
        String firstPageQuery = ydbQuerySource.getQuery("yql.queries.accounts.getByFolderProviderSpaceFirstPage");
        HashMap<String, Value<?>> paramsMap = new HashMap<>();
        List<FolderProviderAccountsSpace> sortedIds = ids.stream().sorted(FolderProviderAccountsSpace.COMPARATOR)
                .collect(Collectors.toList());
        ListValue idsParam = ListValue.of(sortedIds.stream().map(id -> TupleValue.of(
                PrimitiveValue.utf8(id.getTenantId().getId()),
                PrimitiveValue.utf8(id.getFolderId()),
                PrimitiveValue.utf8(id.getProviderId()),
                Ydb.nullToEmptyUtf8(id.getAccountsSpaceId())
        )).toArray(TupleValue[]::new));
        paramsMap.put("$ids", idsParam);
        paramsMap.put("$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS));
        paramsMap.put("$include_deleted", PrimitiveValue.bool(withDeleted));
        Params firstPageParams = Params.copyOf(paramsMap);
        return session.executeDataQueryRetryable(firstPageQuery, firstPageParams).flatMap(firstPageResult -> {
            List<AccountModel> accounts = toModels(firstPageResult);
            if (!firstPageResult.getResultSet(0).isTruncated() && accounts.size() < Ydb.MAX_RESPONSE_ROWS) {
                return Mono.just(accounts);
            }
            return getNextPage(session, sortedIds, accounts.get(accounts.size() - 1), withDeleted)
                    .expand(tuple -> {
                        if (!tuple.getT2() && tuple.getT1().size() < Ydb.MAX_RESPONSE_ROWS) {
                            return Mono.empty();
                        } else {
                            return getNextPage(session, sortedIds,
                                    tuple.getT1().get(tuple.getT1().size() - 1), withDeleted);
                        }
                    }).map(Tuple2::getT1).reduce(accounts, (l, r) -> Stream.concat(l.stream(), r.stream())
                            .collect(Collectors.toList()));
        });
    }

    public Mono<List<ServiceAccountKeys>> getAllNonDeletedServiceAccountKeysByProviderAccountsSpaces(
            YdbTxSession session, TenantId tenantId, String providerId, Set<String> accountsSpaceIds) {
        Map<String, TenantId> tenantIdCache = new ConcurrentHashMap<>();
        List<String> sortedAccountsSpaceIds = accountsSpaceIds.stream().map(v -> v == null ? "" : v).sorted().toList();
        String firstPageQuery = ydbQuerySource
                .getQuery("yql.queries.accounts.getByProviderSpacesFirstPage");
        HashMap<String, Value<?>> paramsMap = new HashMap<>();
        paramsMap.put("$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        paramsMap.put("$provider_id", PrimitiveValue.utf8(providerId));
        paramsMap.put("$accounts_spaces_ids", ListValue.of(sortedAccountsSpaceIds.stream()
                .map(PrimitiveValue::utf8).toArray(PrimitiveValue[]::new)));
        paramsMap.put("$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS));
        Params firstPageParams = Params.copyOf(paramsMap);
        return session.executeDataQueryRetryable(firstPageQuery, firstPageParams).flatMap(firstPageResult -> {
            List<ServiceAccountKeys> accounts = toModels(firstPageResult, 0,
                    v -> toServiceAccountKeys(v, tenantIdCache));
            long size = toModel(firstPageResult, 1, this::toSize).orElse(0L);
            Optional<ServiceAccountKeys> lastAccount = toModel(firstPageResult, 2,
                    v -> toServiceAccountKeys(v, tenantIdCache));
            if (lastAccount.isEmpty() || size < Ydb.MAX_RESPONSE_ROWS) {
                return Mono.just(accounts);
            }
            return getServiceAccountKeysNextPage(session, tenantId, providerId, sortedAccountsSpaceIds,
                    lastAccount.get(), tenantIdCache).expand(tuple -> {
                        if (tuple.getT2().isEmpty() || tuple.getT3() < Ydb.MAX_RESPONSE_ROWS) {
                            return Mono.empty();
                        } else {
                            return getServiceAccountKeysNextPage(session, tenantId, providerId, sortedAccountsSpaceIds,
                                    tuple.getT2().get(), tenantIdCache);
                        }
                    }).map(Tuple2::getT1).reduce(accounts, (l, r) -> Stream.concat(l.stream(), r.stream())
                            .collect(Collectors.toList()));
        });
    }

    private Mono<Tuple2<List<AccountModel>, Boolean>> getNextPageByTenant(
            YdbTxSession session, TenantId tenantId, AccountModel from) {
        String nextPageQuery = ydbQuerySource
                .getQuery(queryKeyPrefix() + ".getByTenantNextPage");
        Map<String, Value<?>> paramsMap = Map.of("$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS),
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$from_id", PrimitiveValue.utf8(from.getId()));
        Params nextPageParams = Params.copyOf(paramsMap);
        return session.executeDataQueryRetryable(nextPageQuery, nextPageParams).map(nextPageResult -> {
            List<AccountModel> models = toModels(nextPageResult);
            return Tuples.of(models, nextPageResult.getResultSet(0).isTruncated());
        });
    }

    @Override
    protected StringIdWithTenant getIdWithTenant(AccountModel model) {
        return new StringIdWithTenant(model.getTenantId(), model.getId());
    }

    @SuppressWarnings("rawtypes")
    @Override
    protected Map<String, Value> prepareFieldValues(AccountModel model) {
        HashMap<String, Value> fields = new HashMap<>();
        fields.put(TENANT_ID.field(), PrimitiveValue.utf8(model.getTenantId().getId()));
        fields.put(ID.field(), PrimitiveValue.utf8(model.getId()));
        fields.put(VERSION.field(), PrimitiveValue.int64(model.getVersion()));
        fields.put(DELETED.field(), bool(model.isDeleted()));
        fields.put(DISPLAY_NAME.field(), Ydb.nullableUtf8(model.getDisplayName().orElse(null)));

        fields.put(PROVIDER_ID.field(), PrimitiveValue.utf8(model.getProviderId()));
        fields.put(OUTER_ACCOUNT_ID_IN_PROVIDER.field(), PrimitiveValue.utf8(model.getOuterAccountIdInProvider()));
        fields.put(OUTER_ACCOUNT_KEY_IN_PROVIDER.field(),
                Ydb.nullableUtf8(model.getOuterAccountKeyInProvider().orElse(null)));
        fields.put(FOLDER_ID.field(), PrimitiveValue.utf8(model.getFolderId()));
        fields.put(Fields.LAST_ACCOUNT_UPDATE.field(), PrimitiveValue.timestamp(model.getLastAccountUpdate()));
        fields.put(Fields.LAST_RECEIVED_VERSION.field(),
                Ydb.nullableInt64(model.getLastReceivedVersion().orElse(null)));
        fields.put(Fields.LATEST_SUCCESSFUL_ACCOUNT_OPERATION_ID.field(),
                Ydb.nullableUtf8(model.getLatestSuccessfulAccountOperationId().orElse(null)));
        fields.put(ACCOUNTS_SPACES_ID.field(), Ydb.nullToEmptyUtf8(model.getAccountsSpacesId()));
        fields.put(FREE_TIER.field(), bool(model.isFreeTier()));
        fields.put(RESERVE_TYPE.field(), Ydb.nullableUtf8(model.getReserveType().map(Enum::name).orElse(null)));
        return fields;
    }

    @Override
    protected AccountModel readOneRow(ResultSetReader reader, Map<String, TenantId> tenantIdCache) {
        return new AccountModel.Builder()
                .setTenantId(Tenants.getInstance(reader.getColumn(TENANT_ID.field()).getUtf8()))
                .setId(reader.getColumn(ID.field()).getUtf8())
                .setVersion(reader.getColumn(VERSION.field()).getInt64())
                .setDeleted(reader.getColumn(DELETED.field()).getBool())
                .setDisplayName(Ydb.utf8OrNull(reader.getColumn(DISPLAY_NAME.field())))
                .setProviderId(reader.getColumn(PROVIDER_ID.field()).getUtf8())
                .setOuterAccountIdInProvider(reader.getColumn(OUTER_ACCOUNT_ID_IN_PROVIDER.field()).getUtf8())
                .setOuterAccountKeyInProvider(Ydb.utf8OrNull(reader.getColumn(OUTER_ACCOUNT_KEY_IN_PROVIDER.field())))
                .setFolderId(reader.getColumn(FOLDER_ID.field()).getUtf8())
                .setLastAccountUpdate(reader.getColumn(Fields.LAST_ACCOUNT_UPDATE.field()).getTimestamp())
                .setLastReceivedVersion(Ydb.int64OrNull(reader.getColumn(Fields.LAST_RECEIVED_VERSION.field())))
                .setLatestSuccessfulAccountOperationId(Ydb.utf8OrNull(reader
                        .getColumn(Fields.LATEST_SUCCESSFUL_ACCOUNT_OPERATION_ID.field())))
                .setAccountsSpacesId(Ydb.utf8EmptyToNull(reader.getColumn(ACCOUNTS_SPACES_ID.field())))
                .setFreeTier(Ydb.boolOrDefault(reader.getColumn(FREE_TIER.field()), false))
                .setReserveType(toReserveType(Ydb.utf8OrNull(reader.getColumn(RESERVE_TYPE.field()))))
                .build();
    }

    @Override
    protected String queryKeyPrefix() {
        return "yql.queries.accounts";
    }

    @Nullable
    private AccountReserveType toReserveType(@Nullable String value) {
        if (value == null) {
            return null;
        }
        return AccountReserveType.valueOf(value);
    }

    private long toSize(ResultSetReader reader) {
        return reader.getColumn("size").getUint64();
    }

    private Map<FolderIdProviderId, Long> toCounts(DataQueryResult result) {
        if (result.isEmpty()) {
            return Map.of();
        }
        if (result.getResultSetCount() > 1) {
            throw new IllegalStateException("Too many result sets");
        }
        ResultSetReader reader = result.getResultSet(0);
        Map<FolderIdProviderId, Long> counts = new HashMap<>();
        while (reader.next()) {
            long count = reader.getColumn("cnt").getUint64();
            String folderId = reader.getColumn("folder_id").getUtf8();
            String providerId = reader.getColumn("provider_id").getUtf8();
            counts.put(new FolderIdProviderId(folderId, providerId), count);
        }
        return counts;
    }

    private Mono<Tuple2<List<AccountModel>, Boolean>> getNextPage(YdbTxSession session, TenantId tenantId,
                                                                  List<String> sortedFolderIds, String providerId,
                                                                  String accountsSpaceId, AccountModel from,
                                                                  boolean withDeleted) {
        List<String> remainingFolderIds = sortedFolderIds.stream().filter(id -> id.compareTo(from.getFolderId()) > 0)
                .collect(Collectors.toList());
        String nextPageQuery;
        Map<String, Value<?>> paramsMap = new HashMap<>();
        paramsMap.put("$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        paramsMap.put("$provider_id", PrimitiveValue.utf8(providerId));
        paramsMap.put("$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS));
        paramsMap.put("$from_id", PrimitiveValue.utf8(from.getId()));
        paramsMap.put("$from_folder_id", PrimitiveValue.utf8(from.getFolderId()));
        paramsMap.put("$accounts_spaces_id", Ydb.nullToEmptyUtf8(Optional.ofNullable(accountsSpaceId)));
        paramsMap.put("$include_deleted", PrimitiveValue.bool(withDeleted));
        if (remainingFolderIds.isEmpty()) {
            nextPageQuery = ydbQuerySource.getQuery("yql.queries.accounts.getByFoldersForProviderLastPages");
        } else {
            nextPageQuery = ydbQuerySource.getQuery("yql.queries.accounts.getByFoldersForProviderNextPages");
            ListValue nextPageFolderIds = ListValue.of(remainingFolderIds.stream().map(PrimitiveValue::utf8)
                    .toArray(PrimitiveValue[]::new));
            paramsMap.put("$from_folder_ids", nextPageFolderIds);
        }
        Params nextPageParams = Params.copyOf(paramsMap);
        return session.executeDataQueryRetryable(nextPageQuery, nextPageParams).map(nextPageResult -> {
            List<AccountModel> accounts = toModels(nextPageResult);
            return Tuples.of(accounts, nextPageResult.getResultSet(0).isTruncated());
        });
    }

    private Mono<Tuple2<List<AccountModel>, Boolean>> getNextPage(YdbTxSession session, TenantId tenantId,
                                                                  String folderId, String providerId,
                                                                  AccountModel from, boolean withDeleted) {
        Map<String, Value<?>> paramsMap = new HashMap<>();
        paramsMap.put("$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        paramsMap.put("$folder_id", PrimitiveValue.utf8(folderId));
        paramsMap.put("$provider_id", PrimitiveValue.utf8(providerId));
        paramsMap.put("$from_accounts_spaces_id", Ydb.nullToEmptyUtf8(from.getAccountsSpacesId()));
        paramsMap.put("$from_id", PrimitiveValue.utf8(from.getId()));
        paramsMap.put("$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS));
        paramsMap.put("$include_deleted", PrimitiveValue.bool(withDeleted));
        Params nextPageParams = Params.copyOf(paramsMap);
        String nextPageQuery = ydbQuerySource.getQuery("yql.queries.accounts.getByFolderAndProviderNextPage");
        return session.executeDataQueryRetryable(nextPageQuery, nextPageParams).map(nextPageResult -> {
            List<AccountModel> accounts = toModels(nextPageResult);
            return Tuples.of(accounts, nextPageResult.getResultSet(0).isTruncated());
        });
    }

    private Mono<Tuple3<List<AccountModel>, Optional<AccountModel>, Long>> getNextPage(
            YdbTxSession session, TenantId tenantId, String providerId, String accountsSpaceId, AccountModel from,
            Set<String> externalAccountIdsToExclude) {
        Map<String, Value<?>> paramsMap = new HashMap<>();
        paramsMap.put("$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        paramsMap.put("$provider_id", PrimitiveValue.utf8(providerId));
        paramsMap.put("$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS));
        paramsMap.put("$from_outer_account_id_in_provider", PrimitiveValue.utf8(from.getOuterAccountIdInProvider()));
        paramsMap.put("$accounts_spaces_id", Ydb.nullToEmptyUtf8(Optional.ofNullable(accountsSpaceId)));
        String nextPageQuery = ydbQuerySource
                .getQuery("yql.queries.accounts.getNonDeletedByProviderAccountsSpaceNextPages");
        Params nextPageParams = Params.copyOf(paramsMap);
        return session.executeDataQueryRetryable(nextPageQuery, nextPageParams).map(nextPageResult -> {
            List<AccountModel> accounts = toModels(nextPageResult, 0);
            long size = toModel(nextPageResult, 1, this::toSize).orElse(0L);
            Optional<AccountModel> nextFrom = toModel(nextPageResult, 2);
            List<AccountModel> filteredAccounts = accounts.stream()
                    .filter(a -> !externalAccountIdsToExclude.contains(a.getOuterAccountIdInProvider()))
                    .collect(Collectors.toList());
            return Tuples.of(filteredAccounts, nextFrom, size);
        });
    }

    private Mono<Tuple2<List<AccountModel>, Boolean>> getNextPage(YdbTxSession session,
                                                                  List<FolderProviderAccountsSpace> ids,
                                                                  AccountModel from, boolean withDeleted) {
        Map<String, Value<?>> paramsMap = new HashMap<>();
        FolderProviderAccountsSpace fromId = FolderProviderAccountsSpace.fromAccount(from);
        List<FolderProviderAccountsSpace> remainingIds = ids.stream()
                .filter(v -> FolderProviderAccountsSpace.COMPARATOR.compare(v, fromId) > 0)
                .collect(Collectors.toList());
        if (!remainingIds.isEmpty()) {
            ListValue fromIdsParam = ListValue.of(remainingIds.stream().map(id -> TupleValue.of(
                    PrimitiveValue.utf8(id.getTenantId().getId()),
                    PrimitiveValue.utf8(id.getFolderId()),
                    PrimitiveValue.utf8(id.getProviderId()),
                    Ydb.nullToEmptyUtf8(id.getAccountsSpaceId())
            )).toArray(TupleValue[]::new));
            paramsMap.put("$from_ids", fromIdsParam);
        }
        paramsMap.put("$from_id", PrimitiveValue.utf8(from.getId()));
        paramsMap.put("$from_tenant_id", PrimitiveValue.utf8(from.getTenantId().getId()));
        paramsMap.put("$from_folder_id", PrimitiveValue.utf8(from.getFolderId()));
        paramsMap.put("$from_provider_id", PrimitiveValue.utf8(from.getProviderId()));
        paramsMap.put("$from_accounts_spaces_id", Ydb.nullToEmptyUtf8(from.getAccountsSpacesId()));
        paramsMap.put("$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS));
        paramsMap.put("$include_deleted", PrimitiveValue.bool(withDeleted));
        Params nextPageParams = Params.copyOf(paramsMap);
        String nextPageQuery = remainingIds.isEmpty()
                ? ydbQuerySource.getQuery("yql.queries.accounts.getByFolderProviderSpaceLastPage")
                : ydbQuerySource.getQuery("yql.queries.accounts.getByFolderProviderSpaceNextPage");
        return session.executeDataQueryRetryable(nextPageQuery, nextPageParams).map(nextPageResult -> {
            List<AccountModel> accounts = toModels(nextPageResult);
            return Tuples.of(accounts, nextPageResult.getResultSet(0).isTruncated());
        });
    }

    private Mono<Tuple3<List<ServiceAccountKeys>, Optional<ServiceAccountKeys>, Long>> getServiceAccountKeysNextPage(
            YdbTxSession session, TenantId tenantId, String providerId, List<String> accountsSpaceIds,
            ServiceAccountKeys from, Map<String, TenantId> tenantIdCache) {
        List<String> remainingAccountsSpaceIds = accountsSpaceIds.stream()
                .filter(v -> v.compareTo(from.getAccountsSpaceId() == null ? "" : from.getAccountsSpaceId()) > 0)
                .toList();
        Map<String, Value<?>> paramsMap = new HashMap<>();
        paramsMap.put("$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        paramsMap.put("$provider_id", PrimitiveValue.utf8(providerId));
        if (!remainingAccountsSpaceIds.isEmpty()) {
            paramsMap.put("$accounts_spaces_ids", ListValue.of(remainingAccountsSpaceIds.stream()
                    .map(PrimitiveValue::utf8).toArray(PrimitiveValue[]::new)));
        }
        paramsMap.put("$from_accounts_space_id",
                PrimitiveValue.utf8(from.getAccountsSpaceId() == null ? "" : from.getAccountsSpaceId()));
        paramsMap.put("$from_outer_account_id_in_provider", PrimitiveValue.utf8(from.getExternalAccountId()));
        paramsMap.put("$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS));
        String nextPageQuery = remainingAccountsSpaceIds.isEmpty()
                ? ydbQuerySource.getQuery("yql.queries.accounts.getByProviderSpacesLastPage")
                : ydbQuerySource.getQuery("yql.queries.accounts.getByProviderSpacesNextPage");
        Params nextPageParams = Params.copyOf(paramsMap);
        return session.executeDataQueryRetryable(nextPageQuery, nextPageParams).map(nextPageResult -> {
            List<ServiceAccountKeys> accounts = toModels(nextPageResult, 0,
                    v -> toServiceAccountKeys(v, tenantIdCache));
            long size = toModel(nextPageResult, 1, this::toSize).orElse(0L);
            Optional<ServiceAccountKeys> nextFrom = toModel(nextPageResult, 2,
                    v -> toServiceAccountKeys(v, tenantIdCache));
            return Tuples.of(accounts, nextFrom, size);
        });
    }

    private ServiceAccountKeys toServiceAccountKeys(ResultSetReader reader, Map<String, TenantId> tenantIdCache) {
        String accountsSpaceId = reader.getColumn("accounts_spaces_id").getUtf8();
        return new ServiceAccountKeys(
                TenantId.getInstance(reader.getColumn("tenant_id").getUtf8(), tenantIdCache),
                reader.getColumn("id").getUtf8(),
                reader.getColumn("provider_id").getUtf8(),
                reader.getColumn("outer_account_id_in_provider").getUtf8(),
                Ydb.utf8OrNull(reader.getColumn("outer_account_key_in_provider")),
                "".equals(accountsSpaceId) ? null : accountsSpaceId,
                reader.getColumn("service_id").getInt64(),
                reader.getColumn("folder_id").getUtf8()
        );
    }

    @SuppressWarnings({"unused", "RedundantSuppression"})
    public enum Fields {
        TENANT_ID,
        ID,
        VERSION,
        DELETED,
        DISPLAY_NAME,
        PROVIDER_ID,
        OUTER_ACCOUNT_ID_IN_PROVIDER,
        OUTER_ACCOUNT_KEY_IN_PROVIDER,
        FOLDER_ID,
        LAST_ACCOUNT_UPDATE,
        LAST_RECEIVED_VERSION,
        LATEST_SUCCESSFUL_ACCOUNT_OPERATION_ID,
        ACCOUNTS_SPACES_ID,
        FREE_TIER,
        RESERVE_TYPE;

        public String field() {
            return name().toLowerCase();
        }
    }

    public static final class FolderIdProviderId {

        private final String folderId;
        private final String providerId;

        public FolderIdProviderId(String folderId, String providerId) {
            this.folderId = folderId;
            this.providerId = providerId;
        }

        public String getFolderId() {
            return folderId;
        }

        public String getProviderId() {
            return providerId;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            FolderIdProviderId that = (FolderIdProviderId) o;
            return Objects.equals(folderId, that.folderId) &&
                    Objects.equals(providerId, that.providerId);
        }

        @Override
        public int hashCode() {
            return Objects.hash(folderId, providerId);
        }

        @Override
        public String toString() {
            return "FolderIdProviderId{" +
                    "folderId='" + folderId + '\'' +
                    ", providerId='" + providerId + '\'' +
                    '}';
        }

    }

}
