package ru.yandex.intranet.d.dao.accounts;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import com.yandex.ydb.table.query.DataQueryResult;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.ListValue;
import com.yandex.ydb.table.values.OptionalValue;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.StructValue;
import com.yandex.ydb.table.values.TupleType;
import com.yandex.ydb.table.values.TupleValue;
import com.yandex.ydb.table.values.Value;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.AbstractDaoWithSoftRemove;
import ru.yandex.intranet.d.dao.QueryUtils;
import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.datasource.Ydb;
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource;
import ru.yandex.intranet.d.datasource.model.WithTxId;
import ru.yandex.intranet.d.datasource.model.YdbExecuteScanQuerySettings;
import ru.yandex.intranet.d.datasource.model.YdbSession;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.folders.FolderType;
import ru.yandex.intranet.d.model.quotas.QuotaAggregationModel;

import static com.yandex.ydb.table.values.PrimitiveType.utf8;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static ru.yandex.intranet.d.dao.accounts.AccountsQuotasDao.Fields.ACCOUNT_ID;
import static ru.yandex.intranet.d.dao.accounts.AccountsQuotasDao.Fields.ALLOCATED_QUOTA;
import static ru.yandex.intranet.d.dao.accounts.AccountsQuotasDao.Fields.FOLDER_ID;
import static ru.yandex.intranet.d.dao.accounts.AccountsQuotasDao.Fields.FROZEN_PROVIDED_QUOTA;
import static ru.yandex.intranet.d.dao.accounts.AccountsQuotasDao.Fields.PROVIDED_QUOTA;
import static ru.yandex.intranet.d.dao.accounts.AccountsQuotasDao.Fields.PROVIDER_ID;
import static ru.yandex.intranet.d.dao.accounts.AccountsQuotasDao.Fields.RESOURCE_ID;
import static ru.yandex.intranet.d.dao.accounts.AccountsQuotasDao.Fields.TENANT_ID;
import static ru.yandex.intranet.d.datasource.Ydb.nullableUtf8;
import static ru.yandex.intranet.d.datasource.Ydb.nullableValue;

/**
 * AccountsQuotasDao.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 20.10.2020
 */
@Component
public class AccountsQuotasDao extends AbstractDaoWithSoftRemove<AccountsQuotasModel, AccountsQuotasModel.Identity> {


    public AccountsQuotasDao(YdbQuerySource ydbQuerySource) {
        super(ydbQuerySource);
    }

    public Mono<List<AccountsQuotasModel>> getByTenantFolderProviderResource(
            YdbTxSession session, List<WithTenant<FolderProviderAccountsSpaceResource>> params) {
        if (params.isEmpty()) {
            return Mono.just(List.of());
        }
        String firstPageQuery = ydbQuerySource
                .getQuery("yql.queries.accounts.quotas.getByTenantFolderProviderAccountsSpaceResourceFirstPage");
        ListValue firstPageListParam = ListValue.of(params.stream().map(id ->
                StructValue.of("tenant_id_param", PrimitiveValue.utf8(id.getTenantId().getId()),
                        "folder_id_param", PrimitiveValue.utf8(id.getIdentity().getFolderId()),
                        "provider_id_param", PrimitiveValue.utf8(id.getIdentity().getProviderId()),
                        "resource_id_param", PrimitiveValue.utf8(id.getIdentity().getResourceId()),
                        "accounts_spaces_id_param", Ydb.nullToEmptyUtf8(id.getIdentity()
                                .getAccountSpaceId())))
                .toArray(StructValue[]::new));
        Params firstPageParams = Params.of("$params", firstPageListParam,
                "$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS));
        return session.executeDataQueryRetryable(firstPageQuery, firstPageParams).flatMap(firstPageResult -> {
            List<AccountsQuotasModel> quotas = toModels(firstPageResult);
            if (!firstPageResult.getResultSet(0).isTruncated() && quotas.size() < Ydb.MAX_RESPONSE_ROWS) {
                return Mono.just(quotas);
            }
            return getNextPage(session, params, quotas.get(quotas.size() - 1)).expand(tuple -> {
                if (!tuple.getT2() && tuple.getT1().size() < Ydb.MAX_RESPONSE_ROWS) {
                    return Mono.empty();
                } else {
                    return getNextPage(session, params, tuple.getT1().get(tuple.getT1().size() - 1));
                }
            }).map(Tuple2::getT1).reduce(quotas, (l, r) -> Stream.concat(l.stream(), r.stream())
                    .collect(Collectors.toList()));
        });
    }

    public Mono<List<WithTenant<AccountResource>>> getResourceIdsByAccountIds(YdbTxSession session,
                                                                              List<WithTenant<String>> accountIds) {
        if (accountIds.isEmpty()) {
            return Mono.just(List.of());
        }
        String firstPageQuery = ydbQuerySource
                .getQuery(queryKeyPrefix() + ".getResourceIdsByAccountIdsFirstPage");
        ListValue firstPageIds = ListValue.of(accountIds.stream().map(id ->
                TupleValue.of(PrimitiveValue.utf8(id.getTenantId().getId()),
                        PrimitiveValue.utf8(id.getIdentity())))
                .toArray(TupleValue[]::new));
        Params firstPageParams = Params.of("$ids", firstPageIds,
                "$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS));
        return session.executeDataQueryRetryable(firstPageQuery, firstPageParams).flatMap(firstPageResult -> {
            List<WithTenant<AccountResource>> quotas = toResourceIds(firstPageResult);
            if (!firstPageResult.getResultSet(0).isTruncated() && quotas.size() < Ydb.MAX_RESPONSE_ROWS) {
                return Mono.just(quotas);
            }
            return getResourceIdsNextPage(session, accountIds, quotas.get(quotas.size() - 1)).expand(tuple -> {
                if (!tuple.getT2() && tuple.getT1().size() < Ydb.MAX_RESPONSE_ROWS) {
                    return Mono.empty();
                } else {
                    return getResourceIdsNextPage(session, accountIds, tuple.getT1().get(tuple.getT1().size() - 1));
                }
            }).map(Tuple2::getT1).reduce(quotas, (l, r) -> Stream.concat(l.stream(), r.stream())
                    .collect(Collectors.toList()));
        });
    }

    public Mono<List<AccountsQuotasModel>> getAllByAccountIds(
            YdbTxSession session,
            TenantId tenantId,
            Set<String> accountIds
    ) {
        if (accountIds.isEmpty()) {
            return Mono.just(List.of());
        }
        ListValue accountIdsValue =
                ListValue.of(accountIds.stream().map(PrimitiveValue::utf8).toArray(PrimitiveValue[]::new));
        PrimitiveValue tenantIdValue = utf8(tenantId.getId());
        String query = ydbQuerySource.getQuery("yql.queries.accounts.quotas.getAllByAccountIds");
        return QueryUtils.getAllRows(session,
                (nextSession, lastId) -> {
                    String lastAccountId = lastId != null ? lastId.getAccountId() : null;
                    String lastResourceId = lastId != null ? lastId.getResourceId() : null;
                    Params params = Params.of(
                            "$tenant_id", tenantIdValue,
                            "$accountIds", accountIdsValue,
                            "$from_account_id", nullableUtf8(lastAccountId),
                            "$from_resource_id", nullableUtf8(lastResourceId)
                    );
                    return session.executeDataQueryRetryable(query, params);
                },
                this::toModels,
                AccountsQuotasModel::getIdentity
        ).map(WithTxId::get);
    }

    public Mono<List<AccountsQuotasModel>> getAllByTenant(YdbTxSession session, TenantId tenantId) {
        String firstPageQuery = ydbQuerySource
                .getQuery(queryKeyPrefix() + ".getByTenantFirstPage");
        Params firstPageParams = Params.of("$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS));
        return session.executeDataQueryRetryable(firstPageQuery, firstPageParams).flatMap(firstPageResult -> {
            List<AccountsQuotasModel> models = toModels(firstPageResult);
            if (!firstPageResult.getResultSet(0).isTruncated() && models.size() < Ydb.MAX_RESPONSE_ROWS) {
                return Mono.just(models);
            }
            return getNextPageByTenant(session, tenantId, models.get(models.size() - 1)).expand(tuple -> {
                if (!tuple.getT2() && tuple.getT1().size() < Ydb.MAX_RESPONSE_ROWS) {
                    return Mono.empty();
                } else {
                    return getNextPageByTenant(session, tenantId, tuple.getT1().get(tuple.getT1().size() - 1));
                }
            }).map(Tuple2::getT1).reduce(models, (l, r) -> Stream.concat(l.stream(), r.stream())
                    .collect(Collectors.toList()));
        });
    }

    public Mono<List<QuotaAggregationModel>> getForAggregationByProviderAndResources(
            YdbTxSession session, TenantId tenantId, String providerId, Collection<String> resourceIds) {
        List<String> sortedResourceIds = resourceIds.stream().sorted().toList();
        String firstPageQuery = ydbQuerySource
                .getQuery("yql.queries.accounts.quotas.getQuotasAggregationFirstPage");
        Params firstPageParams = Params.of("$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$provider_id", PrimitiveValue.utf8(providerId),
                "$resource_ids", ListValue.of(sortedResourceIds.stream().map(PrimitiveValue::utf8)
                        .toArray(PrimitiveValue[]::new)),
                "$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS));
        return session.executeDataQueryRetryable(firstPageQuery, firstPageParams).flatMap(firstPageResult -> {
            QuotaAggregationPage page = toQuotaAggregationModels(firstPageResult);
            if ((!firstPageResult.getResultSet(0).isTruncated() && page.models.size() < Ydb.MAX_RESPONSE_ROWS)
                    || page.cursor == null) {
                return Mono.just(page.models);
            }
            return getNextPageForAggregationByProviderAndResources(session, tenantId, providerId, sortedResourceIds,
                    page.cursor).expand(tuple -> {
                if ((!tuple.getT2() && tuple.getT1().models.size() < Ydb.MAX_RESPONSE_ROWS)
                        || tuple.getT1().cursor == null) {
                    return Mono.empty();
                } else {
                    return getNextPageForAggregationByProviderAndResources(session, tenantId, providerId,
                            sortedResourceIds, tuple.getT1().cursor);
                }
            }).map(p -> p.getT1().models).reduce(page.models, (l, r) -> Stream.concat(l.stream(), r.stream())
                    .collect(Collectors.toList()));
        });
    }

    public Mono<List<QuotaAggregationModel>> getForAggregationByProvider(
            YdbTxSession session, TenantId tenantId, String providerId) {
        String firstPageQuery = ydbQuerySource
                .getQuery("yql.queries.accounts.quotas.getProviderQuotasAggregationFirstPage");
        Params firstPageParams = Params.of("$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$provider_id", PrimitiveValue.utf8(providerId),
                "$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS));
        return session.executeDataQueryRetryable(firstPageQuery, firstPageParams).flatMap(firstPageResult -> {
            QuotaAggregationPage page = toQuotaAggregationModels(firstPageResult);
            if ((!firstPageResult.getResultSet(0).isTruncated() && page.models.size() < Ydb.MAX_RESPONSE_ROWS)
                    || page.cursor == null) {
                return Mono.just(page.models);
            }
            return getNextPageForAggregationByProvider(session, tenantId, providerId, page.cursor).expand(tuple -> {
                if ((!tuple.getT2() && tuple.getT1().models.size() < Ydb.MAX_RESPONSE_ROWS)
                        || tuple.getT1().cursor == null) {
                    return Mono.empty();
                } else {
                    return getNextPageForAggregationByProvider(session, tenantId, providerId, tuple.getT1().cursor);
                }
            }).map(p -> p.getT1().models).reduce(page.models, (l, r) -> Stream.concat(l.stream(), r.stream())
                    .collect(Collectors.toList()));
        });
    }

    public Mono<WithTxId<List<AccountsQuotasModel>>> getAllByResourceIds(
            YdbTxSession session, TenantId tenantId, List<String> resourceIds
    ) {
        PrimitiveValue tenantIdValue = PrimitiveValue.utf8(tenantId.getId());
        ListValue resourceIdsValue = ListValue.of(
                resourceIds.stream().map(PrimitiveValue::utf8).toArray(PrimitiveValue[]::new)
        );
        TupleType fromKeyType = TupleType.of(List.of(utf8(), utf8(), utf8(), utf8(), utf8()));
        String query = ydbQuerySource.getQuery("yql.queries.accounts.quotas.getAllByResourceIds");
        return QueryUtils.getAllRows(session,
                (nextSession, lastPrimaryKey) -> {
                    OptionalValue fromKey = nullableValue(fromKeyType,
                            lastPrimaryKey == null ? null : TupleValue.of(
                                    // tenant_id, folder_id, provider_id, account_id, resource_id
                                    tenantIdValue,
                                    PrimitiveValue.utf8(lastPrimaryKey.getT1()),
                                    PrimitiveValue.utf8(lastPrimaryKey.getT2()),
                                    PrimitiveValue.utf8(lastPrimaryKey.getT3()),
                                    PrimitiveValue.utf8(lastPrimaryKey.getT4())
                            )
                    );
                    Params params = Params.of(
                            "$tenant_id", tenantIdValue,
                            "$resource_ids", resourceIdsValue,
                            "$from_key", fromKey
                    );
                    return nextSession.executeDataQueryRetryable(query, params);
                },
                this::toModels,
                accountQuota -> Tuples.of(// folder_id, provider_id, account_id, resource_id
                        accountQuota.getFolderId(),
                        accountQuota.getProviderId(),
                        accountQuota.getAccountId(),
                        accountQuota.getResourceId()
                )
        );
    }

    private Mono<Tuple2<List<AccountsQuotasModel>, Boolean>> getNextPageByTenant(
            YdbTxSession session, TenantId tenantId, AccountsQuotasModel from) {
        String nextPageQuery = ydbQuerySource
                .getQuery(queryKeyPrefix() + ".getByTenantNextPage");
        Map<String, Value<?>> paramsMap = Map.of("$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS),
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$from_account_id", PrimitiveValue.utf8(from.getAccountId()),
                "$from_resource_id", PrimitiveValue.utf8(from.getResourceId()));
        Params nextPageParams = Params.copyOf(paramsMap);
        return session.executeDataQueryRetryable(nextPageQuery, nextPageParams).map(nextPageResult -> {
            List<AccountsQuotasModel> models = toModels(nextPageResult);
            return Tuples.of(models, nextPageResult.getResultSet(0).isTruncated());
        });
    }

    private Mono<Tuple2<QuotaAggregationPage, Boolean>> getNextPageForAggregationByProviderAndResources(
            YdbTxSession session, TenantId tenantId, String providerId, List<String> resourceIds,
            QuotaAggregationCursor from) {
        String fromResourceId = from.model.getResourceId();
        String fromAccountId = from.accountId;
        List<String> remainingResourceIds = resourceIds.stream()
                .filter(id -> id.compareTo(fromResourceId) > 0).toList();
        String nextPageQuery = remainingResourceIds.isEmpty()
                ? ydbQuerySource.getQuery("yql.queries.accounts.quotas.getQuotasAggregationLastPage")
                : ydbQuerySource.getQuery("yql.queries.accounts.quotas.getQuotasAggregationNextPage");
        Map<String, Value<?>> paramsMap = new HashMap<>();
        paramsMap.put("$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS));
        paramsMap.put("$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        paramsMap.put("$provider_id", PrimitiveValue.utf8(providerId));
        paramsMap.put("$from_account_id", PrimitiveValue.utf8(fromAccountId));
        paramsMap.put("$from_resource_id", PrimitiveValue.utf8(fromResourceId));
        if (!remainingResourceIds.isEmpty()) {
            paramsMap.put("$resource_ids", ListValue.of(remainingResourceIds.stream().map(PrimitiveValue::utf8)
                    .toArray(PrimitiveValue[]::new)));
        }
        Params nextPageParams = Params.copyOf(paramsMap);
        return session.executeDataQueryRetryable(nextPageQuery, nextPageParams).map(nextPageResult -> {
            QuotaAggregationPage page = toQuotaAggregationModels(nextPageResult);
            return Tuples.of(page, nextPageResult.getResultSet(0).isTruncated());
        });
    }

    private Mono<Tuple2<QuotaAggregationPage, Boolean>> getNextPageForAggregationByProvider(
            YdbTxSession session, TenantId tenantId, String providerId, QuotaAggregationCursor from) {
        String fromResourceId = from.model.getResourceId();
        String fromAccountId = from.accountId;
        String nextPageQuery = ydbQuerySource
                .getQuery("yql.queries.accounts.quotas.getProviderQuotasAggregationNextPage");
        Map<String, Value<?>> paramsMap = new HashMap<>();
        paramsMap.put("$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS));
        paramsMap.put("$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        paramsMap.put("$provider_id", PrimitiveValue.utf8(providerId));
        paramsMap.put("$from_account_id", PrimitiveValue.utf8(fromAccountId));
        paramsMap.put("$from_resource_id", PrimitiveValue.utf8(fromResourceId));
        Params nextPageParams = Params.copyOf(paramsMap);
        return session.executeDataQueryRetryable(nextPageQuery, nextPageParams).map(nextPageResult -> {
            QuotaAggregationPage page = toQuotaAggregationModels(nextPageResult);
            return Tuples.of(page, nextPageResult.getResultSet(0).isTruncated());
        });
    }

    private QuotaAggregationPage toQuotaAggregationModels(DataQueryResult result) {
        if (result.isEmpty()) {
            return new QuotaAggregationPage(List.of(), null);
        }
        if (result.getResultSetCount() > 1) {
            throw new IllegalStateException("Too many result sets");
        }
        ResultSetReader reader = result.getResultSet(0);
        List<QuotaAggregationModel> list = new ArrayList<>();
        String accountId = null;
        while (reader.next()) {
            String resourceId = reader.getColumn("resource_id").getUtf8();
            long provided = reader.getColumn("provided_quota").getInt64();
            long allocated = reader.getColumn("allocated_quota").getInt64();
            long serviceId = reader.getColumn("service_id").getInt64();
            accountId = reader.getColumn("account_id").getUtf8();
            String folderId = reader.getColumn("folder_id").getUtf8();
            FolderType folderType = FolderType.valueOf(reader.getColumn("folder_type").getUtf8());
            list.add(new QuotaAggregationModel(resourceId, null, null, provided, allocated, serviceId,
                    folderId, accountId, folderType));
        }
        QuotaAggregationCursor cursor = !list.isEmpty() && accountId != null
                ? new QuotaAggregationCursor(list.get(list.size() - 1), accountId) : null;
        return new QuotaAggregationPage(list, cursor);
    }

    public Mono<WithTxId<List<AccountsQuotasModel>>> getAllByFolderIds(YdbTxSession session, TenantId tenantId,
                                                                       Set<String> folderIds) {
        String query = ydbQuerySource.getQuery(queryKeyPrefix() + ".getAllByFolderIds");
        return QueryUtils.getAllRows(session,
                (nextSession, quotaLastPrimaryKey) -> {
                    ListValue idsParam = ListValue.of(folderIds.stream()
                            .map(PrimitiveValue::utf8).toArray(PrimitiveValue[]::new));
                    TupleType fromKeyType = TupleType.of(List.of(utf8(), utf8()));
                    TupleValue fromKey = getFromKey(quotaLastPrimaryKey);
                    Params params = Params.of("$folder_ids", idsParam,
                            "$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                            "$from_key", nullableValue(fromKeyType, fromKey));
                    return nextSession.executeDataQueryRetryable(query, params);
                },
                this::toModels,
                AccountsQuotasModel::getIdentity);
    }

    public Mono<WithTxId<List<AccountsQuotasModel>>> getAllByFoldersAndProvider(
            YdbTxSession session, TenantId tenantId, Set<String> folderIds, String providerId,
            boolean includeWithZeroProvided) {
        String query = ydbQuerySource.getQuery("yql.queries.accounts.quotas.getAllByFoldersAndProvider");
        return QueryUtils.getAllRows(session,
                (nextSession, quotaLastPrimaryKey) -> {
                    ListValue idsParam = ListValue.of(folderIds.stream()
                            .map(PrimitiveValue::utf8).toArray(PrimitiveValue[]::new));
                    TupleType fromKeyType = TupleType.of(List.of(utf8(), utf8()));
                    TupleValue fromKey = getFromKey(quotaLastPrimaryKey);
                    Params params = Params.of("$folder_ids", idsParam,
                            "$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                            "$from_key", nullableValue(fromKeyType, fromKey),
                            "$provider_id", PrimitiveValue.utf8(providerId),
                            "$include_with_zero_provided", PrimitiveValue.bool(includeWithZeroProvided));
                    return nextSession.executeDataQueryRetryable(query, params);
                },
                this::toModels,
                AccountsQuotasModel::getIdentity);
    }

    public Flux<WithTenant<AccountsQuotasModel.Identity>> scanZeroAccountsQuotas(
            YdbSession session, Instant lastProvisionUpdateThreshold, long limit, Duration timeout) {
        String query = ydbQuerySource.getQuery("yql.queries.accounts.quotas.scanZeroAccountsQuotas");
        Params params = Params.of("$limit", PrimitiveValue.uint64(limit),
                "$last_provision_update_threshold", PrimitiveValue.timestamp(lastProvisionUpdateThreshold));
        return session.executeScanQuery(query, params,
                        YdbExecuteScanQuerySettings.builder().timeout(timeout).build())
                .flatMapIterable(reader -> {
                    List<WithTenant<AccountsQuotasModel.Identity>> page = new ArrayList<>();
                    Map<String, TenantId> tenantIdCache = new HashMap<>();
                    while (reader.next()) {
                        String tenantIdString = reader.getColumn(TENANT_ID.field()).getUtf8();
                        TenantId tenantId = tenantIdCache.computeIfAbsent(tenantIdString, TenantId::new);
                        String accountId = reader.getColumn(ACCOUNT_ID.field()).getUtf8();
                        String resourceId = reader.getColumn(RESOURCE_ID.field()).getUtf8();
                        page.add(new WithTenant<>(tenantId, new AccountsQuotasModel.Identity(accountId, resourceId)));
                    }
                    return page;
                });
    }

    public Mono<List<AccountsQuotasModel>> getPageByAccount(YdbTxSession session, String accountId,
                                                            String fromResourceId, TenantId tenantId, long limit) {
        String query;
        Params params;
        if (fromResourceId == null) {
            query = ydbQuerySource.getQuery("yql.queries.accounts.quotas.getByAccountFirstPage");
            HashMap<String, Value<?>> paramsMap = new HashMap<>();
            paramsMap.put("$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
            paramsMap.put("$account_id", PrimitiveValue.utf8(accountId));
            paramsMap.put("$limit", PrimitiveValue.uint64(limit));
            params = Params.copyOf(paramsMap);
        } else {
            query = ydbQuerySource.getQuery("yql.queries.accounts.quotas.getByAccountNextPage");
            Map<String, Value<?>> paramsMap = new HashMap<>();
            paramsMap.put("$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
            paramsMap.put("$account_id", PrimitiveValue.utf8(accountId));
            paramsMap.put("$from_resource_id", PrimitiveValue.utf8(fromResourceId));
            paramsMap.put("$limit", PrimitiveValue.uint64(limit));
            params = Params.copyOf(paramsMap);
        }
        return session.executeDataQueryRetryable(query, params).map(this::toModels);
    }

    private TupleValue getFromKey(AccountsQuotasModel.Identity quotaLastPrimaryKey) {
        return Optional.ofNullable(quotaLastPrimaryKey)
                .map(key -> TupleValue.of(PrimitiveValue.utf8(key.getAccountId()),
                        PrimitiveValue.utf8(key.getResourceId())))
                .orElse(null);
    }

    @Override
    protected WithTenant<AccountsQuotasModel.Identity> getIdentityWithTenant(AccountsQuotasModel model) {
        return new WithTenant<>(model.getTenantId(), model.getIdentity());
    }

    @Override
    protected Params getIdentityParams(AccountsQuotasModel.Identity id) {
        return Params.create()
                .put("$account_id", PrimitiveValue.utf8(id.getAccountId()))
                .put("$resource_id", PrimitiveValue.utf8(id.getResourceId()));
    }

    @SuppressWarnings("rawtypes")
    @Override
    protected Map<String, Value> prepareFieldValues(AccountsQuotasModel model) {
        HashMap<String, Value> fields = new HashMap<>();

        fields.put(TENANT_ID.field(), PrimitiveValue.utf8(model.getTenantId().getId()));
        fields.put(ACCOUNT_ID.field(), PrimitiveValue.utf8(model.getAccountId()));
        fields.put(RESOURCE_ID.field(), PrimitiveValue.utf8(model.getResourceId()));
        fields.put(PROVIDED_QUOTA.field(), PrimitiveValue.int64(model.getProvidedQuota()));
        fields.put(ALLOCATED_QUOTA.field(), PrimitiveValue.int64(model.getAllocatedQuota()));
        fields.put(FOLDER_ID.field(), PrimitiveValue.utf8(model.getFolderId()));
        fields.put(PROVIDER_ID.field(), PrimitiveValue.utf8(model.getProviderId()));
        fields.put(FOLDER_ID.field(), PrimitiveValue.utf8(model.getFolderId()));
        fields.put(Fields.LAST_PROVISION_UPDATE.field(), PrimitiveValue.timestamp(model.getLastProvisionUpdate()));
        fields.put(Fields.LAST_RECEIVED_PROVISION_VERSION.field(),
                Ydb.nullableInt64(model.getLastReceivedProvisionVersion().orElse(null)));
        fields.put(Fields.LATEST_SUCCESSFUL_PROVISION_OPERATION_ID.field(),
                Ydb.nullableUtf8(model.getLatestSuccessfulProvisionOperationId().orElse(null)));
        fields.put(FROZEN_PROVIDED_QUOTA.field(), PrimitiveValue.int64(model.getFrozenProvidedQuota()));

        return fields;
    }

    @Override
    protected AccountsQuotasModel readOneRow(ResultSetReader reader, Map<String, TenantId> tenantIdCache) {
        return new AccountsQuotasModel.Builder()
                .setTenantId(Tenants.getInstance(reader.getColumn(TENANT_ID.field()).getUtf8()))
                .setAccountId(reader.getColumn(ACCOUNT_ID.field()).getUtf8())
                .setResourceId(reader.getColumn(RESOURCE_ID.field()).getUtf8())
                .setProvidedQuota(reader.getColumn(PROVIDED_QUOTA.field()).getInt64())
                .setAllocatedQuota(reader.getColumn(ALLOCATED_QUOTA.field()).getInt64())
                .setFolderId(reader.getColumn(FOLDER_ID.field()).getUtf8())
                .setProviderId(reader.getColumn(PROVIDER_ID.field()).getUtf8())
                .setLastProvisionUpdate(reader.getColumn(Fields.LAST_PROVISION_UPDATE.field()).getTimestamp())
                .setLastReceivedProvisionVersion(Ydb.int64OrNull(reader
                        .getColumn(Fields.LAST_RECEIVED_PROVISION_VERSION.field())))
                .setLatestSuccessfulProvisionOperationId(Ydb.utf8OrNull(reader
                        .getColumn(Fields.LATEST_SUCCESSFUL_PROVISION_OPERATION_ID.field())))
                .setFrozenProvidedQuota(Ydb.int64OrDefault(reader.getColumn(FROZEN_PROVIDED_QUOTA.field()), 0))
                .build();
    }

    @Override
    protected String queryKeyPrefix() {
        return "yql.queries.accounts.quotas";
    }

    private Mono<Tuple2<List<AccountsQuotasModel>, Boolean>> getNextPage(
            YdbTxSession session, List<WithTenant<FolderProviderAccountsSpaceResource>> params,
            AccountsQuotasModel from) {
        String nextPageQuery = ydbQuerySource
                .getQuery("yql.queries.accounts.quotas.getByTenantFolderProviderAccountsSpaceResourceNextPage");
        ListValue nextPageListParam = ListValue.of(params.stream().map(id ->
                        StructValue.of("tenant_id_param", PrimitiveValue.utf8(id.getTenantId().getId()),
                                "folder_id_param", PrimitiveValue.utf8(id.getIdentity().getFolderId()),
                                "provider_id_param", PrimitiveValue.utf8(id.getIdentity().getProviderId()),
                                "resource_id_param", PrimitiveValue.utf8(id.getIdentity().getResourceId()),
                                "accounts_spaces_id_param", Ydb.nullToEmptyUtf8(id.getIdentity()
                                        .getAccountSpaceId())))
                .toArray(StructValue[]::new));
        Map<String, Value<?>> paramsMap = Map.of("$params", nextPageListParam,
                "$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS),
                "$from_tenant_id", PrimitiveValue.utf8(from.getTenantId().getId()),
                "$from_account_id", PrimitiveValue.utf8(from.getAccountId()),
                "$from_resource_id", PrimitiveValue.utf8(from.getResourceId()));
        Params nextPageParams = Params.copyOf(paramsMap);
        return session.executeDataQueryRetryable(nextPageQuery, nextPageParams).map(nextPageResult -> {
            List<AccountsQuotasModel> quotas = toModels(nextPageResult);
            return Tuples.of(quotas, nextPageResult.getResultSet(0).isTruncated());
        });
    }

    private Mono<Tuple2<List<WithTenant<AccountResource>>, Boolean>> getResourceIdsNextPage(
            YdbTxSession session, List<WithTenant<String>> accountIds, WithTenant<AccountResource> from) {
        String nextPageQuery = ydbQuerySource
                .getQuery(queryKeyPrefix() + ".getResourceIdsByAccountIdsNextPage");
        ListValue nextPageIds = ListValue.of(accountIds.stream().map(id ->
                TupleValue.of(PrimitiveValue.utf8(id.getTenantId().getId()),
                        PrimitiveValue.utf8(id.getIdentity())))
                .toArray(TupleValue[]::new));
        Map<String, Value<?>> paramsMap = Map.of("$ids", nextPageIds,
                "$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS),
                "$from_tenant_id", PrimitiveValue.utf8(from.getTenantId().getId()),
                "$from_account_id", PrimitiveValue.utf8(from.getIdentity().getAccountId()),
                "$from_resource_id", PrimitiveValue.utf8(from.getIdentity().getResourceId()));
        Params nextPageParams = Params.copyOf(paramsMap);
        return session.executeDataQueryRetryable(nextPageQuery, nextPageParams).map(nextPageResult -> {
            List<WithTenant<AccountResource>> resourceIds = toResourceIds(nextPageResult);
            return Tuples.of(resourceIds, nextPageResult.getResultSet(0).isTruncated());
        });
    }

    private List<WithTenant<AccountResource>> toResourceIds(DataQueryResult result) {
        if (result.isEmpty()) {
            return List.of();
        }
        if (result.getResultSetCount() > 1) {
            throw new IllegalStateException("Too many result sets");
        }
        ResultSetReader reader = result.getResultSet(0);
        List<WithTenant<AccountResource>> list = new ArrayList<>();
        while (reader.next()) {
            list.add(readOneResourceIdRow(reader));
        }
        return list;
    }

    private WithTenant<AccountResource> readOneResourceIdRow(ResultSetReader reader) {
        TenantId tenantId = Tenants.getInstance(reader.getColumn(TENANT_ID.field()).getUtf8());
        String resourceId = reader.getColumn(RESOURCE_ID.field()).getUtf8();
        String accountId = reader.getColumn(ACCOUNT_ID.field()).getUtf8();
        return new WithTenant<>(tenantId, new AccountResource(accountId, resourceId));
    }

    @SuppressWarnings({"unused", "RedundantSuppression"})
    public enum Fields {
        TENANT_ID,
        ACCOUNT_ID,
        RESOURCE_ID,
        PROVIDED_QUOTA,
        ALLOCATED_QUOTA,
        FOLDER_ID,
        PROVIDER_ID,
        LAST_PROVISION_UPDATE,
        LAST_RECEIVED_PROVISION_VERSION,
        LATEST_SUCCESSFUL_PROVISION_OPERATION_ID,
        FROZEN_PROVIDED_QUOTA;

        public String field() {
            return name().toLowerCase();
        }
    }

    public static final class FolderProviderAccountsSpaceResource {

        private final String folderId;
        private final String providerId;
        private final String accountSpaceId;
        private final String resourceId;

        public FolderProviderAccountsSpaceResource(String folderId, String providerId, String accountSpaceId,
                                                   String resourceId) {
            this.folderId = folderId;
            this.providerId = providerId;
            this.accountSpaceId = accountSpaceId;
            this.resourceId = resourceId;
        }

        public String getFolderId() {
            return folderId;
        }

        public String getProviderId() {
            return providerId;
        }

        public Optional<String> getAccountSpaceId() {
            return Optional.ofNullable(accountSpaceId);
        }

        public String getResourceId() {
            return resourceId;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            FolderProviderAccountsSpaceResource that = (FolderProviderAccountsSpaceResource) o;
            return Objects.equals(folderId, that.folderId) &&
                    Objects.equals(providerId, that.providerId) &&
                    Objects.equals(accountSpaceId, that.accountSpaceId) &&
                    Objects.equals(resourceId, that.resourceId);
        }

        @Override
        public int hashCode() {
            return Objects.hash(folderId, providerId, accountSpaceId, resourceId);
        }

        @Override
        public String toString() {
            return "FolderProviderAccountsSpaceResource{" +
                    "folderId='" + folderId + '\'' +
                    ", providerId='" + providerId + '\'' +
                    ", accountSpaceId='" + accountSpaceId + '\'' +
                    ", resourceId='" + resourceId + '\'' +
                    '}';
        }

    }

    public static final class AccountResource {

        private final String accountId;
        private final String resourceId;

        public AccountResource(String accountId, String resourceId) {
            this.accountId = accountId;
            this.resourceId = resourceId;
        }

        public String getAccountId() {
            return accountId;
        }

        public String getResourceId() {
            return resourceId;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            AccountResource that = (AccountResource) o;
            return Objects.equals(accountId, that.accountId) &&
                    Objects.equals(resourceId, that.resourceId);
        }

        @Override
        public int hashCode() {
            return Objects.hash(accountId, resourceId);
        }

        @Override
        public String toString() {
            return "AccountResource{" +
                    "accountId='" + accountId + '\'' +
                    ", resourceId='" + resourceId + '\'' +
                    '}';
        }

    }

    private record QuotaAggregationPage(List<QuotaAggregationModel> models,
                                        @Nullable QuotaAggregationCursor cursor) {
    }

    private record QuotaAggregationCursor(QuotaAggregationModel model, String accountId) {
    }
}
