package ru.yandex.intranet.d.dao.accounts;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.core.type.TypeReference;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.result.ValueReader;
import com.yandex.ydb.table.values.ListValue;
import com.yandex.ydb.table.values.OptionalValue;
import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.TupleType;
import com.yandex.ydb.table.values.TupleValue;
import com.yandex.ydb.table.values.Value;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import ru.yandex.intranet.d.dao.AbstractDaoWithSoftRemove;
import ru.yandex.intranet.d.dao.JsonFieldHelper;
import ru.yandex.intranet.d.dao.QueryUtils;
import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource;
import ru.yandex.intranet.d.datasource.model.WithTxId;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel.OperationType;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel.RequestStatus;
import ru.yandex.intranet.d.model.accounts.OperationChangesModel;
import ru.yandex.intranet.d.model.accounts.OperationErrorCollections;
import ru.yandex.intranet.d.model.accounts.OperationErrorKind;
import ru.yandex.intranet.d.model.accounts.OperationOrdersModel;
import ru.yandex.intranet.d.model.accounts.OperationSource;
import ru.yandex.intranet.d.util.ObjectMapperHolder;

import static com.yandex.ydb.table.values.PrimitiveValue.timestamp;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static ru.yandex.intranet.d.dao.accounts.AccountsQuotasOperationsDao.Fields.ACCOUNTS_SPACE_ID;
import static ru.yandex.intranet.d.dao.accounts.AccountsQuotasOperationsDao.Fields.AUTHOR_USER_ID;
import static ru.yandex.intranet.d.dao.accounts.AccountsQuotasOperationsDao.Fields.AUTHOR_USER_UID;
import static ru.yandex.intranet.d.dao.accounts.AccountsQuotasOperationsDao.Fields.CREATE_DATE_TIME;
import static ru.yandex.intranet.d.dao.accounts.AccountsQuotasOperationsDao.Fields.ERROR_KIND;
import static ru.yandex.intranet.d.dao.accounts.AccountsQuotasOperationsDao.Fields.ERROR_MESSAGE;
import static ru.yandex.intranet.d.dao.accounts.AccountsQuotasOperationsDao.Fields.FULL_ERROR_MESSAGE;
import static ru.yandex.intranet.d.dao.accounts.AccountsQuotasOperationsDao.Fields.LAST_REQUEST_ID;
import static ru.yandex.intranet.d.dao.accounts.AccountsQuotasOperationsDao.Fields.OPERATION_ID;
import static ru.yandex.intranet.d.dao.accounts.AccountsQuotasOperationsDao.Fields.OPERATION_SOURCE;
import static ru.yandex.intranet.d.dao.accounts.AccountsQuotasOperationsDao.Fields.OPERATION_TYPE;
import static ru.yandex.intranet.d.dao.accounts.AccountsQuotasOperationsDao.Fields.ORDERS;
import static ru.yandex.intranet.d.dao.accounts.AccountsQuotasOperationsDao.Fields.PROVIDER_ID;
import static ru.yandex.intranet.d.dao.accounts.AccountsQuotasOperationsDao.Fields.REQUESTED_CHANGES;
import static ru.yandex.intranet.d.dao.accounts.AccountsQuotasOperationsDao.Fields.REQUEST_STATUS;
import static ru.yandex.intranet.d.dao.accounts.AccountsQuotasOperationsDao.Fields.TENANT_ID;
import static ru.yandex.intranet.d.dao.accounts.AccountsQuotasOperationsDao.Fields.UPDATE_DATE_TIME;
import static ru.yandex.intranet.d.datasource.Ydb.nullableTimestamp;
import static ru.yandex.intranet.d.datasource.Ydb.nullableUtf8;
import static ru.yandex.intranet.d.datasource.Ydb.nullableValue;
import static ru.yandex.intranet.d.datasource.Ydb.timestampOrNull;
import static ru.yandex.intranet.d.datasource.Ydb.utf8OrNull;

/**
 * AccountsQuotasOperationsDao.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 22.10.2020
 */
@Component
public class AccountsQuotasOperationsDao extends AbstractDaoWithSoftRemove<AccountsQuotasOperationsModel, String> {

    private final JsonFieldHelper<OperationChangesModel> requestedChangesHelper;
    private final JsonFieldHelper<OperationOrdersModel> ordersHelper;
    private final JsonFieldHelper<OperationErrorCollections> fullErrorMessageHelper;

    public AccountsQuotasOperationsDao(YdbQuerySource ydbQuerySource,
                                       @Qualifier("ydbJsonObjectMapper") ObjectMapperHolder objectMapper) {
        super(ydbQuerySource);
        this.requestedChangesHelper = new JsonFieldHelper<>(objectMapper,
                new TypeReference<OperationChangesModel>() { });
        this.ordersHelper = new JsonFieldHelper<>(objectMapper, new TypeReference<OperationOrdersModel>() { });
        this.fullErrorMessageHelper = new JsonFieldHelper<>(objectMapper,
                new TypeReference<OperationErrorCollections>() { });
    }

    public Mono<List<AccountsQuotasOperationsModel>> getAllByIds(
            YdbTxSession session, TenantId tenantId, List<String> ids
    ) {
        if (ids.isEmpty()) {
            return Mono.just(List.of());
        }
        PrimitiveValue tenantIdValue = utf8(tenantId.getId());
        ListValue idsValue = ListValue.of(
                ids.stream().map(id -> TupleValue.of(tenantIdValue, utf8(id))).toArray(TupleValue[]::new)
        );
        String query = ydbQuerySource.getQuery("yql.queries.accounts.quotas.operations.getAllByIds");
        TupleType fromKeyType = TupleType.of(List.of(PrimitiveType.utf8(), PrimitiveType.utf8()));
        return QueryUtils.getAllRows(session, (nextSession, lastId) -> {
                    OptionalValue fromKey = nullableValue(fromKeyType,
                            lastId == null ? null : TupleValue.of(tenantIdValue, utf8(lastId)));
                    Params params = Params.of(
                            "$ids", idsValue,
                            "$from_key", fromKey);
                    return nextSession.executeDataQueryRetryable(query, params);
                },
                this::toModels,
                AccountsQuotasOperationsModel::getOperationId).map(WithTxId::get);
    }

    @Override
    protected WithTenant<String> getIdentityWithTenant(AccountsQuotasOperationsModel model) {
        return new WithTenant<>(model.getTenantId(), model.getOperationId());
    }

    @Override
    protected Params getIdentityParams(String operationId) {
        return Params.create()
                .put("$operation_id", utf8(operationId));
    }

    @SuppressWarnings("rawtypes")
    @Override
    protected Map<String, Value> prepareFieldValues(AccountsQuotasOperationsModel model) {
        HashMap<String, Value> fields = new HashMap<>();

        fields.put(TENANT_ID.field(), utf8(model.getTenantId().getId()));
        fields.put(OPERATION_ID.field(), utf8(model.getOperationId()));
        fields.put(LAST_REQUEST_ID.field(), nullableUtf8(model.getLastRequestId().orElse(null)));
        fields.put(CREATE_DATE_TIME.field(), timestamp(model.getCreateDateTime()));
        fields.put(OPERATION_SOURCE.field(), utf8(model.getOperationSource().name()));
        fields.put(OPERATION_TYPE.field(), utf8(model.getOperationType().name()));
        fields.put(AUTHOR_USER_ID.field(), utf8(model.getAuthorUserId()));
        fields.put(AUTHOR_USER_UID.field(), nullableUtf8(model.getAuthorUserUid().orElse(null)));
        fields.put(PROVIDER_ID.field(), utf8(model.getProviderId()));
        fields.put(ACCOUNTS_SPACE_ID.field(), nullableUtf8(model.getAccountsSpaceId().orElse(null)));
        fields.put(UPDATE_DATE_TIME.field(), nullableTimestamp(model.getUpdateDateTime().orElse(null)));
        fields.put(REQUEST_STATUS.field(), nullableUtf8(model.getRequestStatus().map(Enum::name).orElse(null)));
        fields.put(ERROR_MESSAGE.field(), nullableUtf8(model.getErrorMessage().orElse(null)));
        fields.put(FULL_ERROR_MESSAGE.field(),
                fullErrorMessageHelper.writeOptional(model.getFullErrorMessage().orElse(null)));
        fields.put(REQUESTED_CHANGES.field(), requestedChangesHelper.write(model.getRequestedChanges()));
        fields.put(ORDERS.field(), ordersHelper.write(model.getOrders()));
        fields.put(ERROR_KIND.field(), nullableUtf8(model.getErrorKind().map(Enum::name).orElse(null)));

        return fields;
    }

    @Override
    protected AccountsQuotasOperationsModel readOneRow(ResultSetReader reader, Map<String, TenantId> tenantIdCache) {
        return new AccountsQuotasOperationsModel.Builder()
                .setTenantId(Tenants.getInstance(reader.getColumn(TENANT_ID.field()).getUtf8()))
                .setOperationId(reader.getColumn(OPERATION_ID.field()).getUtf8())
                .setLastRequestId(utf8OrNull(reader.getColumn(LAST_REQUEST_ID.field())))
                .setCreateDateTime(reader.getColumn(CREATE_DATE_TIME.field()).getTimestamp())
                .setOperationSource(OperationSource.valueOf(reader.getColumn(OPERATION_SOURCE.field()).getUtf8()))
                .setOperationType(OperationType.valueOf(reader.getColumn(OPERATION_TYPE.field()).getUtf8()))
                .setAuthorUserId(reader.getColumn(AUTHOR_USER_ID.field()).getUtf8())
                .setAuthorUserUid(utf8OrNull(reader.getColumn(AUTHOR_USER_UID.field())))
                .setProviderId(reader.getColumn(PROVIDER_ID.field()).getUtf8())
                .setAccountsSpaceId(utf8OrNull(reader.getColumn(ACCOUNTS_SPACE_ID.field())))
                .setUpdateDateTime(timestampOrNull(reader.getColumn(UPDATE_DATE_TIME.field())))
                .setRequestStatus(requestStatusOrNull(reader.getColumn(REQUEST_STATUS.field())))
                .setErrorMessage(utf8OrNull(reader.getColumn(ERROR_MESSAGE.field())))
                .setFullErrorMessage(fullErrorMessageHelper.read(reader.getColumn(FULL_ERROR_MESSAGE.field())))
                .setRequestedChanges(requestedChangesHelper.read(reader.getColumn(REQUESTED_CHANGES.field())))
                .setOrders(ordersHelper.read(reader.getColumn(ORDERS.field())))
                .setErrorKind(errorKindOrNull(reader.getColumn(ERROR_KIND.field())))
                .build();
    }

    private RequestStatus requestStatusOrNull(ValueReader value) {
        if (value.isOptionalItemPresent()) {
            return RequestStatus.valueOf(value.getUtf8());
        }
        return null;
    }

    private OperationErrorKind errorKindOrNull(ValueReader value) {
        if (value.isOptionalItemPresent()) {
            return OperationErrorKind.valueOf(value.getUtf8());
        }
        return null;
    }

    @Override
    protected String queryKeyPrefix() {
        return "yql.queries.accounts.quotas.operations";
    }

    @SuppressWarnings({"unused", "RedundantSuppression"})
    public enum Fields {
        TENANT_ID,
        OPERATION_ID,
        LAST_REQUEST_ID,
        CREATE_DATE_TIME,
        OPERATION_SOURCE,
        OPERATION_TYPE,
        AUTHOR_USER_ID,
        AUTHOR_USER_UID,
        PROVIDER_ID,
        ACCOUNTS_SPACE_ID,
        UPDATE_DATE_TIME,
        REQUEST_STATUS,
        ERROR_MESSAGE,
        FULL_ERROR_MESSAGE,
        REQUESTED_CHANGES,
        ORDERS,
        ERROR_KIND;

        public String field() {
            return name().toLowerCase();
        }
    }
}
