package ru.yandex.intranet.d.dao.accounts

import com.yandex.ydb.table.query.Params
import com.yandex.ydb.table.result.ResultSetReader
import com.yandex.ydb.table.values.PrimitiveValue
import com.yandex.ydb.table.values.StructValue
import kotlinx.coroutines.reactor.awaitSingle
import kotlinx.coroutines.reactor.awaitSingleOrNull
import org.springframework.stereotype.Component
import ru.yandex.intranet.d.dao.DaoReader
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource
import ru.yandex.intranet.d.datasource.model.YdbTxSession
import ru.yandex.intranet.d.model.TenantId
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsRequestLogModel

/**
 *
 * @author Mustakayev Marat <mmarat248@yandex-team.ru>
 */
@Component
class AccountsQuotasOperationsRequestLogDao(
    private val ydbQuerySource: YdbQuerySource
) {

    suspend fun getAllByOperationId(session: YdbTxSession, operationId: String,
                                    tenantId: TenantId, limit: Long): List<AccountsQuotasOperationsRequestLogModel> {
        if (limit > 1000) {
            throw IllegalArgumentException("Limit is too large")
        }

        val query = ydbQuerySource.getQuery(
            "yql.queries.accounts.quotas.operations.requestLog.getAllByOperationId")
        val params = Params.of(
            "\$operation_id", PrimitiveValue.utf8(operationId),
            "\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$limit", PrimitiveValue.uint64(limit)
        )

        return DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun upsertOneRetryable(session: YdbTxSession, value: AccountsQuotasOperationsRequestLogModel) {
        val query = ydbQuerySource.getQuery("yql.queries.accounts.quotas.operations.requestLog.upsertOne")
        val params = toUpsertOneParams(value)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    private fun toUpsertOneParams(value: AccountsQuotasOperationsRequestLogModel) = Params.of(
        "\$value", toUpsertStruct(value)
    )

    private fun toUpsertStruct(value: AccountsQuotasOperationsRequestLogModel) = StructValue.of(
        mapOf(
            "tenant_id" to PrimitiveValue.utf8(value.tenantId.id),
            "operation_id" to PrimitiveValue.utf8(value.operationId),
            "request_id" to PrimitiveValue.utf8(value.requestId),
            "create_date_time" to PrimitiveValue.timestamp(value.createDateTime),
            "request_data" to PrimitiveValue.jsonDocument(value.requestData),
            "response_data" to PrimitiveValue.jsonDocument(value.responseData),
            "expire_at" to PrimitiveValue.timestamp(value.expireAt)
        )
    )

    private fun toModel(reader: ResultSetReader) = AccountsQuotasOperationsRequestLogModel(
        tenantId = TenantId(reader.getColumn("tenant_id").utf8),
        operationId = reader.getColumn("operation_id").utf8,
        requestId = reader.getColumn("request_id").utf8,
        createDateTime = reader.getColumn("create_date_time").timestamp,
        requestData = reader.getColumn("request_data").jsonDocument,
        responseData = reader.getColumn("response_data").jsonDocument,
        expireAt = reader.getColumn("expire_at").timestamp,
    )

}
