package ru.yandex.intranet.d.dao.accounts;

import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.type.TypeReference;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.ListValue;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.TupleValue;
import com.yandex.ydb.table.values.Value;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.dao.AbstractDaoWithSoftRemove;
import ru.yandex.intranet.d.dao.JsonFieldHelper;
import ru.yandex.intranet.d.dao.QueryUtils;
import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource;
import ru.yandex.intranet.d.datasource.model.WithTxId;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.providers.ProviderId;
import ru.yandex.intranet.d.model.providers.ProviderUISettings;
import ru.yandex.intranet.d.model.resources.ResourceSegmentSettingsModel;
import ru.yandex.intranet.d.util.ObjectMapperHolder;

import static ru.yandex.intranet.d.dao.accounts.AccountsSpacesDao.Fields.DELETED;
import static ru.yandex.intranet.d.dao.accounts.AccountsSpacesDao.Fields.DESCRIPTION_EN;
import static ru.yandex.intranet.d.dao.accounts.AccountsSpacesDao.Fields.DESCRIPTION_RU;
import static ru.yandex.intranet.d.dao.accounts.AccountsSpacesDao.Fields.ID;
import static ru.yandex.intranet.d.dao.accounts.AccountsSpacesDao.Fields.NAME_EN;
import static ru.yandex.intranet.d.dao.accounts.AccountsSpacesDao.Fields.NAME_RU;
import static ru.yandex.intranet.d.dao.accounts.AccountsSpacesDao.Fields.OUTER_KEY_IN_PROVIDER;
import static ru.yandex.intranet.d.dao.accounts.AccountsSpacesDao.Fields.PROVIDER_ID;
import static ru.yandex.intranet.d.dao.accounts.AccountsSpacesDao.Fields.READ_ONLY;
import static ru.yandex.intranet.d.dao.accounts.AccountsSpacesDao.Fields.SEGMENTS;
import static ru.yandex.intranet.d.dao.accounts.AccountsSpacesDao.Fields.SEGMENTS_CONCAT;
import static ru.yandex.intranet.d.dao.accounts.AccountsSpacesDao.Fields.SYNC_ENABLED;
import static ru.yandex.intranet.d.dao.accounts.AccountsSpacesDao.Fields.TENANT_ID;
import static ru.yandex.intranet.d.dao.accounts.AccountsSpacesDao.Fields.UI_SETTINGS;
import static ru.yandex.intranet.d.dao.accounts.AccountsSpacesDao.Fields.VERSION;
import static ru.yandex.intranet.d.datasource.Ydb.bool;
import static ru.yandex.intranet.d.datasource.Ydb.int64;
import static ru.yandex.intranet.d.datasource.Ydb.nullableUtf8;
import static ru.yandex.intranet.d.datasource.Ydb.uint64;
import static ru.yandex.intranet.d.datasource.Ydb.utf8;
import static ru.yandex.intranet.d.datasource.Ydb.utf8OrNull;

/**
 * AccountsSpacesDao.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 09.12.2020
 */
@Component
public class AccountsSpacesDao extends AbstractDaoWithSoftRemove<AccountSpaceModel, String> {
    private final JsonFieldHelper<Set<ResourceSegmentSettingsModel>> segments;
    private final JsonFieldHelper<ProviderUISettings> uiSettings;

    protected AccountsSpacesDao(
            YdbQuerySource ydbQuerySource,
            @Qualifier("ydbJsonObjectMapper") ObjectMapperHolder objectMapper
    ) {
        super(ydbQuerySource);
        this.segments = new JsonFieldHelper<>(objectMapper, new TypeReference<>() {
        });
        this.uiSettings = new JsonFieldHelper<>(objectMapper, new TypeReference<>() {
        });
    }

    public static String toIndexConcat(Set<ResourceSegmentSettingsModel> data) {
        return data.stream()
                .sorted(Comparator.comparing(ResourceSegmentSettingsModel::getSegmentationId))
                .map(ResourceSegmentSettingsModel::getSegmentId)
                .collect(Collectors.joining(";"));
    }

    public Mono<WithTxId<Optional<AccountSpaceModel>>> getBySegments(
            YdbTxSession session,
            TenantId tenantId,
            String providerId,
            Set<ResourceSegmentSettingsModel> segments
    ) {
        String query = ydbQuerySource.getQuery(queryKeyPrefix() + ".getBySegments");
        Params params = Params.create()
                .put("$tenant_id", utf8(tenantId.getId()))
                .put("$provider_id", utf8(providerId))
                .put("$segments_concat", utf8(toIndexConcat(segments)));
        return session.executeDataQueryRetryable(query, params).map(r -> new WithTxId<>(toModel(r), r.getTxId()));
    }

    public Mono<WithTxId<List<AccountSpaceModel>>> getAllByProvider(
            YdbTxSession session,
            TenantId tenantId,
            String providerId
    ) {
        String query = ydbQuerySource.getQuery(queryKeyPrefix() + ".getAllByProvider");
        PrimitiveValue tenantIdValue = utf8(tenantId.getId());
        return QueryUtils.getAllRows(session,
                (nextSession, lastId) -> {
                    Params params = Params.of(
                            "$tenant_id", tenantIdValue,
                            "$provider_id", utf8(providerId),
                            "$from_id", nullableUtf8(lastId)
                    );
                    return session.executeDataQueryRetryable(query, params);
                },
                this::toModels,
                AccountSpaceModel::getId
        );
    }

    public Mono<List<AccountSpaceModel>> getByProvider(
            YdbTxSession session,
            String providerId,
            TenantId tenantId,
            String accountSpaceModelIdFrom,
            int limit,
            boolean withDeleted
    ) {
        String query = ydbQuerySource.getQuery(queryKeyPrefix() + ".getByProvider");
        Params params = Params.of(
                "$tenant_id", utf8(tenantId.getId()),
                "$provider_id", utf8(providerId),
                "$from_id", nullableUtf8(accountSpaceModelIdFrom),
                "$limit", uint64(limit),
                "$include_deleted", PrimitiveValue.bool(withDeleted)
        );
        return session.executeDataQueryRetryable(query, params)
                .map(this::toModels);
    }

    public Mono<WithTxId<List<AccountSpaceModel>>> getAllByProviderIds(
            YdbTxSession session,
            List<WithTenant<ProviderId>> providerIds
    ) {
        String query = ydbQuerySource.getQuery("yql.queries.accounts.spaces.getAllByProviderIds");
        ListValue providerIdsValue = ListValue.of(providerIds.stream().map(id -> TupleValue.of(
                PrimitiveValue.utf8(id.getTenantId().getId()),
                PrimitiveValue.utf8(id.getIdentity().getValue())
        )).toArray(TupleValue[]::new));
        return QueryUtils.getAllRows(session,
                (nextSession, lastId) -> {
                    Params params = Params.of(
                            "$provider_ids", providerIdsValue,
                            "$from_id", nullableUtf8(lastId)
                    );
                    return session.executeDataQueryRetryable(query, params);
                },
                this::toModels,
                AccountSpaceModel::getId
        );
    }

    public Mono<Void> setReadOnly(YdbTxSession session, String id, TenantId tenantId, boolean readOnly) {
        String query = ydbQuerySource.getQuery(queryKeyPrefix() + ".setReadOnly");
        final Params params = getIdentityWithTenantParams(id, tenantId)
                .put("$read_only", PrimitiveValue.bool(readOnly));
        return session.executeDataQuery(query, params).then();
    }

    public Mono<WithTxId<Optional<AccountSpaceModel>>> getByKey(
            YdbTxSession session,
            TenantId tenantId,
            String providerId,
            String outerKeyInProvider
    ) {
        String query = ydbQuerySource.getQuery(queryKeyPrefix() + ".getByKey");
        Params params = Params.create()
                .put("$tenant_id", utf8(tenantId.getId()))
                .put("$provider_id", utf8(providerId))
                .put("$outer_key_in_provider", utf8(outerKeyInProvider));
        return session.executeDataQueryRetryable(query, params).map(r -> new WithTxId<>(toModel(r), r.getTxId()));
    }

    @Override
    protected WithTenant<String> getIdentityWithTenant(AccountSpaceModel model) {
        return new WithTenant<>(model.getTenantId(), model.getId());
    }

    @Override
    protected Params getIdentityParams(String id) {
        return Params.create().put("$id", utf8(id));
    }

    @SuppressWarnings("rawtypes")
    @Override
    protected Map<String, Value> prepareFieldValues(AccountSpaceModel model) {
        HashMap<String, Value> fields = new HashMap<>();

        fields.put(TENANT_ID.field(), utf8(model.getTenantId().getId()));
        fields.put(ID.field(), utf8(model.getId()));
        fields.put(DELETED.field(), bool(model.isDeleted()));
        fields.put(NAME_EN.field(), utf8(model.getNameEn()));
        fields.put(NAME_RU.field(), utf8(model.getNameRu()));
        fields.put(DESCRIPTION_EN.field(), utf8(model.getDescriptionEn()));
        fields.put(DESCRIPTION_RU.field(), utf8(model.getDescriptionRu()));
        fields.put(PROVIDER_ID.field(), utf8(model.getProviderId()));
        fields.put(OUTER_KEY_IN_PROVIDER.field(), nullableUtf8(model.getOuterKeyInProvider()));
        fields.put(VERSION.field(), int64(model.getVersion()));
        fields.put(SEGMENTS.field(), segments.write(model.getSegments()));
        fields.put(SEGMENTS_CONCAT.field(), utf8(toIndexConcat(model.getSegments())));
        fields.put(READ_ONLY.field(), bool(model.isReadOnly()));
        fields.put(UI_SETTINGS.field(), uiSettings.writeOptional(model.getUiSettings().orElse(null)));
        fields.put(SYNC_ENABLED.field(), bool(model.isSyncEnabled()));

        return fields;

    }

    @Override
    protected AccountSpaceModel readOneRow(ResultSetReader reader, Map<String, TenantId> tenantIdCache) {
        return new AccountSpaceModel.Builder()
                .setTenantId(Tenants.getInstance(utf8(reader.getColumn(TENANT_ID.field()))))
                .setId(utf8(reader.getColumn(ID.field())))
                .setDeleted(bool(reader.getColumn(DELETED.field())))
                .setNameEn(utf8(reader.getColumn(NAME_EN.field())))
                .setNameRu(utf8(reader.getColumn(NAME_RU.field())))
                .setDescriptionEn(utf8(reader.getColumn(DESCRIPTION_EN.field())))
                .setDescriptionRu(utf8(reader.getColumn(DESCRIPTION_RU.field())))
                .setProviderId(utf8(reader.getColumn(PROVIDER_ID.field())))
                .setOuterKeyInProvider(utf8OrNull(reader.getColumn(OUTER_KEY_IN_PROVIDER.field())))
                .setVersion(int64(reader.getColumn(VERSION.field())))
                .setSegments(segments.read(reader.getColumn(SEGMENTS.field())))
                .setReadOnly(bool(reader.getColumn(READ_ONLY.field())))
                .setUiSettings(uiSettings.read(reader.getColumn(UI_SETTINGS.field())))
                .setSyncEnabled(bool(reader.getColumn(SYNC_ENABLED.field())))
                .build();
    }

    @Override
    protected String queryKeyPrefix() {
        return "yql.queries.accounts.spaces";
    }

    @SuppressWarnings({"unused", "RedundantSuppression"})
    public enum Fields {
        TENANT_ID,
        ID,
        DELETED,
        NAME_EN,
        NAME_RU,
        DESCRIPTION_EN,
        DESCRIPTION_RU,
        PROVIDER_ID,
        OUTER_KEY_IN_PROVIDER,
        VERSION,
        SEGMENTS,
        SEGMENTS_CONCAT,
        READ_ONLY,
        UI_SETTINGS,
        SYNC_ENABLED;

        public String field() {
            return name().toLowerCase();
        }
    }
}
