package ru.yandex.intranet.d.dao.accounts;

import java.util.Comparator;
import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nullable;

import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.accounts.AccountModel;

/**
 * Tenant id, folder id, provider id, accounts space id.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class FolderProviderAccountsSpace {

    public static final Comparator<FolderProviderAccountsSpace> COMPARATOR = Comparator
            .<FolderProviderAccountsSpace, String>comparing(v -> v.getTenantId().getId())
            .thenComparing(FolderProviderAccountsSpace::getFolderId)
            .thenComparing(FolderProviderAccountsSpace::getProviderId)
            .thenComparing(v -> v.getAccountsSpaceId().orElse(""));

    private final TenantId tenantId;
    private final String folderId;
    private final String providerId;
    private final String accountsSpaceId;

    public FolderProviderAccountsSpace(TenantId tenantId,
                                       String folderId,
                                       String providerId,
                                       @Nullable
                                       String accountsSpaceId) {
        this.tenantId = tenantId;
        this.folderId = folderId;
        this.providerId = providerId;
        this.accountsSpaceId = accountsSpaceId;
    }

    public static FolderProviderAccountsSpace fromAccount(AccountModel account) {
        return new FolderProviderAccountsSpace(account.getTenantId(), account.getFolderId(), account.getProviderId(),
                account.getAccountsSpacesId().orElse(null));
    }

    public TenantId getTenantId() {
        return tenantId;
    }

    public String getFolderId() {
        return folderId;
    }

    public String getProviderId() {
        return providerId;
    }

    public Optional<String> getAccountsSpaceId() {
        return Optional.ofNullable(accountsSpaceId);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        FolderProviderAccountsSpace that = (FolderProviderAccountsSpace) o;
        return Objects.equals(tenantId, that.tenantId) &&
                Objects.equals(folderId, that.folderId) &&
                Objects.equals(providerId, that.providerId) &&
                Objects.equals(accountsSpaceId, that.accountsSpaceId);
    }

    @Override
    public int hashCode() {
        return Objects.hash(tenantId, folderId, providerId, accountsSpaceId);
    }

    @Override
    public String toString() {
        return "FolderProviderAccountsSpace{" +
                "tenantId=" + tenantId +
                ", folderId='" + folderId + '\'' +
                ", providerId='" + providerId + '\'' +
                ", accountsSpaceId='" + accountsSpaceId + '\'' +
                '}';
    }

}
