package ru.yandex.intranet.d.dao.accounts;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.ListValue;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.Value;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.AbstractDao;
import ru.yandex.intranet.d.datasource.Ydb;
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.accounts.OperationInProgressModel;

/**
 * Operations in progress DAO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class OperationsInProgressDao extends AbstractDao<OperationInProgressModel, OperationInProgressModel.Key> {

    public OperationsInProgressDao(YdbQuerySource ydbQuerySource) {
        super(ydbQuerySource);
    }

    public Mono<List<OperationInProgressModel>> getAllByTenant(YdbTxSession session, TenantId tenantId) {
        String firstPageQuery = ydbQuerySource
                .getQuery("yql.queries.operationsInProgress.getByTenantFirstPage");
        Params firstPageParams = Params.of("$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS));
        return session.executeDataQueryRetryable(firstPageQuery, firstPageParams).flatMap(firstPageResult -> {
            List<OperationInProgressModel> models = toModels(firstPageResult);
            if (!firstPageResult.getResultSet(0).isTruncated() && models.size() < Ydb.MAX_RESPONSE_ROWS) {
                return Mono.just(models);
            }
            return getNextPageByTenant(session, tenantId, models.get(models.size() - 1)).expand(tuple -> {
                if (!tuple.getT2() && tuple.getT1().size() < Ydb.MAX_RESPONSE_ROWS) {
                    return Mono.empty();
                } else {
                    return getNextPageByTenant(session, tenantId, tuple.getT1().get(tuple.getT1().size() - 1));
                }
            }).map(Tuple2::getT1).reduce(models, (l, r) -> Stream.concat(l.stream(), r.stream())
                    .collect(Collectors.toList()));
        });
    }

    public Mono<List<OperationInProgressModel>> getAllByTenantAccounts(YdbTxSession session, TenantId tenantId,
                                                                       Set<String> accountIds) {
        if (accountIds.isEmpty()) {
            return Mono.just(List.of());
        }
        List<String> sortedAccountIds = accountIds.stream().sorted().collect(Collectors.toList());
        String firstPageQuery = ydbQuerySource
                .getQuery("yql.queries.operationsInProgress.getByTenantAccountsFirstPage");
        ListValue accountIdsParam = ListValue.of(sortedAccountIds.stream().map(PrimitiveValue::utf8)
                .toArray(PrimitiveValue[]::new));
        Params firstPageParams = Params.of("$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$account_ids", accountIdsParam,
                "$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS));
        return session.executeDataQueryRetryable(firstPageQuery, firstPageParams).flatMap(firstPageResult -> {
            List<OperationInProgressModel> models = toModels(firstPageResult);
            if (!firstPageResult.getResultSet(0).isTruncated() && models.size() < Ydb.MAX_RESPONSE_ROWS) {
                return Mono.just(models);
            }
            return getNextPageByTenantAccounts(session, tenantId, sortedAccountIds, models.get(models.size() - 1))
                    .expand(tuple -> {
                        if (!tuple.getT2() && tuple.getT1().size() < Ydb.MAX_RESPONSE_ROWS) {
                            return Mono.empty();
                        } else {
                            return getNextPageByTenantAccounts(session, tenantId, sortedAccountIds,
                                    tuple.getT1().get(tuple.getT1().size() - 1));
                        }
                    }).map(Tuple2::getT1).reduce(models, (l, r) -> Stream.concat(l.stream(), r.stream())
                            .collect(Collectors.toList()));
        });
    }

    public Mono<List<OperationInProgressModel>> getAllByTenantFolders(YdbTxSession session, TenantId tenantId,
                                                                      Set<String> folderIds) {
        if (folderIds.isEmpty()) {
            return Mono.just(List.of());
        }
        List<String> sortedFolderIds = folderIds.stream().sorted().collect(Collectors.toList());
        String firstPageQuery = ydbQuerySource
                .getQuery("yql.queries.operationsInProgress.getByTenantFoldersFirstPage");
        ListValue folderIdsParam = ListValue.of(sortedFolderIds.stream().map(PrimitiveValue::utf8)
                .toArray(PrimitiveValue[]::new));
        Params firstPageParams = Params.of("$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$folder_ids", folderIdsParam,
                "$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS));
        return session.executeDataQueryRetryable(firstPageQuery, firstPageParams).flatMap(firstPageResult -> {
            List<OperationInProgressModel> models = toModels(firstPageResult);
            if (!firstPageResult.getResultSet(0).isTruncated() && models.size() < Ydb.MAX_RESPONSE_ROWS) {
                return Mono.just(models);
            }
            return getNextPageByTenantFolders(session, tenantId, sortedFolderIds, models.get(models.size() - 1))
                    .expand(tuple -> {
                        if (!tuple.getT2() && tuple.getT1().size() < Ydb.MAX_RESPONSE_ROWS) {
                            return Mono.empty();
                        } else {
                            return getNextPageByTenantFolders(session, tenantId, sortedFolderIds,
                                    tuple.getT1().get(tuple.getT1().size() - 1));
                        }
                    }).map(Tuple2::getT1).reduce(models, (l, r) -> Stream.concat(l.stream(), r.stream())
                            .collect(Collectors.toList()));
        });
    }

    // WARNING Query without index, use with caution (table should be small though, so it is likely harmless)
    public Mono<Void> deleteByOperationIdRetryable(YdbTxSession session, TenantId tenantId, String operationId) {
        String query = ydbQuerySource.getQuery("yql.queries.operationsInProgress.deleteByOperationId");
        Params params = Params.of("$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$operation_id", PrimitiveValue.utf8(operationId));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<Void> deleteOneRetryable(YdbTxSession session, WithTenant<OperationInProgressModel.Key> idWithTenant) {
        String query = ydbQuerySource.getQuery("yql.queries.operationsInProgress.deleteOne");
        Params params = getIdentityWithTenantParams(idWithTenant.getIdentity(), idWithTenant.getTenantId());
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<Void> deleteManyRetryable(YdbTxSession session, List<WithTenant<OperationInProgressModel.Key>> ids) {
        if (ids.isEmpty()) {
            return Mono.empty();
        }
        String query = ydbQuerySource.getQuery("yql.queries.operationsInProgress.deleteMany");
        Params params = Params.of("$ids", toWithTenantsListValue(ids));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<Void> incrementRetryCounterRetryable(
            YdbTxSession session, List<WithTenant<OperationInProgressModel.Key>> ids,
            long retryCounter
    ) {
        String query = ydbQuerySource.getQuery("yql.queries.operationsInProgress.incrementRetryCounter");
        Params params = Params.of(
                "$ids", toWithTenantsListValue(ids),
                "$newRetryCounter", PrimitiveValue.int64(retryCounter + 1)
        );
        return session.executeDataQueryRetryable(query, params).then();
    }

    @Override
    protected WithTenant<OperationInProgressModel.Key> getIdentityWithTenant(OperationInProgressModel model) {
        return new WithTenant<>(model.getTenantId(), model.getKey());
    }

    @Override
    protected Params getIdentityParams(OperationInProgressModel.Key id) {
        return Params.create()
                .put("$operation_id", PrimitiveValue.utf8(id.getOperationId()))
                .put("$folder_id", PrimitiveValue.utf8(id.getFolderId()));
    }

    @SuppressWarnings("rawtypes")
    @Override
    protected Map<String, Value> prepareFieldValues(OperationInProgressModel model) {
        HashMap<String, Value> fields = new HashMap<>();
        fields.put(Fields.TENANT_ID.field(), PrimitiveValue.utf8(model.getTenantId().getId()));
        fields.put(Fields.OPERATION_ID.field(), PrimitiveValue.utf8(model.getOperationId()));
        fields.put(Fields.FOLDER_ID.field(), PrimitiveValue.utf8(model.getFolderId()));
        fields.put(Fields.ACCOUNT_ID.field(), Ydb.nullableUtf8(model.getAccountId().orElse(null)));
        fields.put(Fields.RETRY_COUNTER.field(), Ydb.int64(model.getRetryCounter()));
        return fields;
    }

    @Override
    protected OperationInProgressModel readOneRow(ResultSetReader reader, Map<String, TenantId> tenantIdCache) {
        return OperationInProgressModel.builder()
                .tenantId(tenantIdCache.computeIfAbsent(reader
                        .getColumn(Fields.TENANT_ID.field()).getUtf8(), TenantId::new))
                .operationId(reader.getColumn(Fields.OPERATION_ID.field()).getUtf8())
                .folderId(reader.getColumn(Fields.FOLDER_ID.field()).getUtf8())
                .accountId(Ydb.utf8OrNull(reader.getColumn(Fields.ACCOUNT_ID.field())))
                .retryCounter(Ydb.int64OrDefault(reader.getColumn(Fields.RETRY_COUNTER.field()), 0L))
                .build();
    }

    @Override
    protected String queryKeyPrefix() {
        return "yql.queries.operationsInProgress";
    }

    private Mono<Tuple2<List<OperationInProgressModel>, Boolean>> getNextPageByTenant(
            YdbTxSession session, TenantId tenantId, OperationInProgressModel from) {
        String nextPageQuery = ydbQuerySource
                .getQuery("yql.queries.operationsInProgress.getByTenantNextPage");
        Map<String, Value<?>> paramsMap = Map.of("$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS),
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$from_folder_id", PrimitiveValue.utf8(from.getFolderId()),
                "$from_operation_id", PrimitiveValue.utf8(from.getOperationId()));
        Params nextPageParams = Params.copyOf(paramsMap);
        return session.executeDataQueryRetryable(nextPageQuery, nextPageParams).map(nextPageResult -> {
            List<OperationInProgressModel> models = toModels(nextPageResult);
            return Tuples.of(models, nextPageResult.getResultSet(0).isTruncated());
        });
    }

    private Mono<Tuple2<List<OperationInProgressModel>, Boolean>> getNextPageByTenantAccounts(
            YdbTxSession session, TenantId tenantId, List<String> accountIds, OperationInProgressModel from) {
        List<String> remainingAccountIds = accountIds.stream()
                .filter(id -> id.compareTo(from.getAccountId().orElseThrow()) > 0)
                .collect(Collectors.toList());
        String nextPageQuery;
        Map<String, Value<?>> paramsMap;
        if (!remainingAccountIds.isEmpty()) {
            nextPageQuery = ydbQuerySource
                    .getQuery("yql.queries.operationsInProgress.getByTenantAccountsNextPages");
            ListValue accountIdsParam = ListValue.of(remainingAccountIds.stream().map(PrimitiveValue::utf8)
                    .toArray(PrimitiveValue[]::new));
            paramsMap = Map.of("$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS),
                    "$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                    "$from_account_ids", accountIdsParam,
                    "$from_account_id", PrimitiveValue.utf8(from.getAccountId().orElseThrow()),
                    "$from_operation_id", PrimitiveValue.utf8(from.getOperationId()));
        } else {
            nextPageQuery = ydbQuerySource
                    .getQuery("yql.queries.operationsInProgress.getByTenantAccountsLastPages");
            paramsMap = Map.of("$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS),
                    "$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                    "$from_account_id", PrimitiveValue.utf8(from.getAccountId().orElseThrow()),
                    "$from_operation_id", PrimitiveValue.utf8(from.getOperationId()));
        }
        Params nextPageParams = Params.copyOf(paramsMap);
        return session.executeDataQueryRetryable(nextPageQuery, nextPageParams).map(nextPageResult -> {
            List<OperationInProgressModel> models = toModels(nextPageResult);
            return Tuples.of(models, nextPageResult.getResultSet(0).isTruncated());
        });
    }

    private Mono<Tuple2<List<OperationInProgressModel>, Boolean>> getNextPageByTenantFolders(
            YdbTxSession session, TenantId tenantId, List<String> folderIds, OperationInProgressModel from) {
        List<String> remainingFolderIds = folderIds.stream()
                .filter(id -> id.compareTo(from.getFolderId()) > 0)
                .collect(Collectors.toList());
        String nextPageQuery;
        Map<String, Value<?>> paramsMap;
        if (!remainingFolderIds.isEmpty()) {
            nextPageQuery = ydbQuerySource
                    .getQuery("yql.queries.operationsInProgress.getByTenantFoldersNextPages");
            ListValue folderIdsParam = ListValue.of(remainingFolderIds.stream().map(PrimitiveValue::utf8)
                    .toArray(PrimitiveValue[]::new));
            paramsMap = Map.of("$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS),
                    "$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                    "$from_folder_ids", folderIdsParam,
                    "$from_folder_id", PrimitiveValue.utf8(from.getFolderId()),
                    "$from_operation_id", PrimitiveValue.utf8(from.getOperationId()));
        } else {
            nextPageQuery = ydbQuerySource
                    .getQuery("yql.queries.operationsInProgress.getByTenantFoldersLastPages");
            paramsMap = Map.of("$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS),
                    "$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                    "$from_folder_id", PrimitiveValue.utf8(from.getFolderId()),
                    "$from_operation_id", PrimitiveValue.utf8(from.getOperationId()));
        }
        Params nextPageParams = Params.copyOf(paramsMap);
        return session.executeDataQueryRetryable(nextPageQuery, nextPageParams).map(nextPageResult -> {
            List<OperationInProgressModel> models = toModels(nextPageResult);
            return Tuples.of(models, nextPageResult.getResultSet(0).isTruncated());
        });
    }

    public enum Fields {
        TENANT_ID,
        OPERATION_ID,
        FOLDER_ID,
        ACCOUNT_ID,
        RETRY_COUNTER;

        public String field() {
            return name().toLowerCase();
        }
    }

}
