package ru.yandex.intranet.d.dao.accounts

import com.yandex.ydb.table.query.Params
import com.yandex.ydb.table.result.ResultSetReader
import com.yandex.ydb.table.values.ListValue
import com.yandex.ydb.table.values.PrimitiveValue
import com.yandex.ydb.table.values.StructValue
import kotlinx.coroutines.reactor.awaitSingle
import kotlinx.coroutines.reactor.awaitSingleOrNull
import org.springframework.stereotype.Component
import reactor.core.publisher.Mono
import ru.yandex.intranet.d.dao.DaoPagination
import ru.yandex.intranet.d.dao.DaoReader
import ru.yandex.intranet.d.datasource.Ydb
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource
import ru.yandex.intranet.d.datasource.model.YdbTxSession
import ru.yandex.intranet.d.kotlin.AccountsSpacesId
import ru.yandex.intranet.d.kotlin.ProviderId
import ru.yandex.intranet.d.kotlin.mono
import ru.yandex.intranet.d.model.TenantId
import ru.yandex.intranet.d.model.accounts.ProviderReserveAccountKey
import ru.yandex.intranet.d.model.accounts.ProviderReserveAccountModel

/**
 * Provider reserve accounts DAO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
class ProviderReserveAccountsDao(private val ydbQuerySource: YdbQuerySource) {

    suspend fun getById(session: YdbTxSession, id: ProviderReserveAccountKey): ProviderReserveAccountModel? {
        val query = ydbQuerySource.getQuery("yql.queries.providerReserveAccounts.getById")
        val params = toKeyParams(id)
        return DaoReader.toModel(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getByIds(session: YdbTxSession,
                         ids: Collection<ProviderReserveAccountKey>): List<ProviderReserveAccountModel> {
        if (ids.isEmpty()) {
            return listOf()
        }
        return ids.distinct().chunked(1000).map {
            val query = ydbQuerySource.getQuery("yql.queries.providerReserveAccounts.getByIds")
            val params = toKeyListParams(it)
            DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
        }.flatten()
    }

    fun upsertOneRetryableMono(session: YdbTxSession, value: ProviderReserveAccountModel) : Mono<Unit> {
        return mono {
            upsertOneRetryable(session, value)
        }
    }

    suspend fun upsertOneRetryable(session: YdbTxSession, value: ProviderReserveAccountModel) {
        val query = ydbQuerySource.getQuery("yql.queries.providerReserveAccounts.upsertOne")
        val params = toUpsertOneParams(value)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun upsertManyRetryable(session: YdbTxSession, values: Collection<ProviderReserveAccountModel>) {
        if (values.isEmpty()) {
            return
        }
        val query = ydbQuerySource.getQuery("yql.queries.providerReserveAccounts.upsertMany")
        val params = toUpsertManyParams(values)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun deleteByIdRetryable(session: YdbTxSession, id: ProviderReserveAccountKey) {
        val query = ydbQuerySource.getQuery("yql.queries.providerReserveAccounts.deleteById")
        val params = toKeyParams(id)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun deleteByIdsRetryable(session: YdbTxSession, ids: Collection<ProviderReserveAccountKey>) {
        if (ids.isEmpty()) {
            return
        }
        val query = ydbQuerySource.getQuery("yql.queries.providerReserveAccounts.deleteByIds")
        val params = toKeyListParams(ids)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun existsByProvider(session: YdbTxSession, tenantId: TenantId, providerId: ProviderId): Boolean {
        val query = ydbQuerySource.getQuery("yql.queries.providerReserveAccounts.existsByProvider")
        val params = toExistsByProviderParams(tenantId, providerId)
        return DaoReader.toModel(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toExists)!!
    }

    suspend fun existsByProviderAccountSpace(session: YdbTxSession, tenantId: TenantId, providerId: ProviderId,
                                             accountsSpaceId: AccountsSpacesId?): Boolean {
        val query = ydbQuerySource.getQuery("yql.queries.providerReserveAccounts.existsByProviderAccountSpace")
        val params = toExistsByProviderAccountSpaceParams(tenantId, providerId, accountsSpaceId)
        return DaoReader.toModel(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toExists)!!
    }

    suspend fun getAllByTenant(session: YdbTxSession, tenantId: TenantId): List<ProviderReserveAccountModel> {
        return DaoPagination.getAllPages(session,
            { ydbQuerySource.getQuery("yql.queries.providerReserveAccounts.getByTenantFirstPage") },
            { ydbQuerySource.getQuery("yql.queries.providerReserveAccounts.getByTenantNextPage") },
            { limit -> toGetByTenantFirstPageParams(tenantId, limit) },
            { limit, lastOnPreviousPage -> toGetByTenantNextPageParams(tenantId, limit,
                lastOnPreviousPage.key.providerId, lastOnPreviousPage.key.accountsSpaceId,
                lastOnPreviousPage.key.accountId)},
            this::toModel
        )
    }

    fun getAllByProviderMono(session: YdbTxSession, tenantId: TenantId,
                             providerId: ProviderId): Mono<List<ProviderReserveAccountModel>> {
        return mono {
            getAllByProvider(session, tenantId, providerId)
        }
    }

    suspend fun getAllByProvider(session: YdbTxSession, tenantId: TenantId,
                                 providerId: ProviderId): List<ProviderReserveAccountModel> {
        return DaoPagination.getAllPages(session,
            { ydbQuerySource.getQuery("yql.queries.providerReserveAccounts.getByProviderFirstPage") },
            { ydbQuerySource.getQuery("yql.queries.providerReserveAccounts.getByProviderNextPage") },
            { limit -> toGetByProviderFirstPageParams(tenantId, providerId, limit) },
            { limit, lastOnPreviousPage -> toGetByProviderNextPageParams(tenantId, providerId, limit,
                lastOnPreviousPage.key.accountsSpaceId, lastOnPreviousPage.key.accountId)},
            this::toModel
        )
    }

    suspend fun getAllByProviderAccountsSpace(session: YdbTxSession, tenantId: TenantId, providerId: ProviderId,
                                              accountsSpaceId: AccountsSpacesId?): List<ProviderReserveAccountModel> {
        return DaoPagination.getAllPages(session,
            { ydbQuerySource.getQuery("yql.queries.providerReserveAccounts.getByProviderAccountSpaceFirstPage") },
            { ydbQuerySource.getQuery("yql.queries.providerReserveAccounts.getByProviderAccountSpaceNextPage") },
            { limit -> toGetByProviderAccountSpaceFirstPageParams(tenantId, providerId, accountsSpaceId, limit) },
            { limit, lastOnPreviousPage -> toGetByProviderAccountSpaceNextPageParams(tenantId, providerId,
                accountsSpaceId, limit, lastOnPreviousPage.key.accountId)},
            this::toModel
        )
    }

    private fun toModel(reader: ResultSetReader) = ProviderReserveAccountModel(
        key = toKey(reader)
    )

    private fun toKey(reader: ResultSetReader) = ProviderReserveAccountKey(
        tenantId = TenantId(reader.getColumn("tenant_id").utf8),
        providerId = reader.getColumn("provider_id").utf8,
        accountsSpaceId = Ydb.utf8EmptyToNull(reader.getColumn("accounts_space_id")),
        accountId = reader.getColumn("account_id").utf8
    )

    private fun toExists(reader: ResultSetReader) = reader.getColumn("provider_reserve_accounts_exists").bool

    private fun toKeyParams(id: ProviderReserveAccountKey) = Params.of(
        "\$id", toKeyStruct(id)
    )

    private fun toKeyListParams(ids: Collection<ProviderReserveAccountKey>) = Params.of(
        "\$ids", ListValue.of(*ids.map { toKeyStruct(it) }.toTypedArray())
    )

    private fun toKeyStruct(id: ProviderReserveAccountKey) = StructValue.of(mapOf(
        "tenant_id" to PrimitiveValue.utf8(id.tenantId.id),
        "provider_id" to PrimitiveValue.utf8(id.providerId),
        "accounts_space_id" to Ydb.nullToEmptyUtf8(id.accountsSpaceId),
        "account_id" to PrimitiveValue.utf8(id.accountId)
    ))

    private fun toUpsertOneParams(value: ProviderReserveAccountModel) = Params.of(
        "\$value", toUpsertStruct(value)
    )

    private fun toUpsertManyParams(values: Collection<ProviderReserveAccountModel>) = Params.of(
        "\$values", ListValue.of(*values.map { toUpsertStruct(it) }.toTypedArray())
    )

    private fun toUpsertStruct(value: ProviderReserveAccountModel) = StructValue.of(
        mapOf(
            "tenant_id" to PrimitiveValue.utf8(value.key.tenantId.id),
            "provider_id" to PrimitiveValue.utf8(value.key.providerId),
            "accounts_space_id" to Ydb.nullToEmptyUtf8(value.key.accountsSpaceId),
            "account_id" to PrimitiveValue.utf8(value.key.accountId)
        )
    )

    private fun toExistsByProviderParams(tenantId: TenantId, providerId: ProviderId) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$provider_id", PrimitiveValue.utf8(providerId))

    private fun toExistsByProviderAccountSpaceParams(tenantId: TenantId, providerId: ProviderId,
                                                     accountsSpaceId: AccountsSpacesId?) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$provider_id", PrimitiveValue.utf8(providerId),
            "\$accounts_space_id", Ydb.nullToEmptyUtf8(accountsSpaceId))

    private fun toGetByTenantFirstPageParams(tenantId: TenantId, limit: Long) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$limit", PrimitiveValue.uint64(limit))

    private fun toGetByTenantNextPageParams(tenantId: TenantId, limit: Long, fromProviderId: ProviderId,
                                            fromAccountsSpaceId: AccountsSpacesId?, fromAccountId: String) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$limit", PrimitiveValue.uint64(limit),
            "\$from_provider_id", PrimitiveValue.utf8(fromProviderId),
            "\$from_accounts_space_id", Ydb.nullToEmptyUtf8(fromAccountsSpaceId),
            "\$from_account_id", PrimitiveValue.utf8(fromAccountId))

    private fun toGetByProviderFirstPageParams(tenantId: TenantId, providerId: ProviderId, limit: Long) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$provider_id", PrimitiveValue.utf8(providerId),
            "\$limit", PrimitiveValue.uint64(limit))

    private fun toGetByProviderNextPageParams(tenantId: TenantId, providerId: ProviderId, limit: Long,
                                              fromAccountsSpaceId: AccountsSpacesId?, fromAccountId: String) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$provider_id", PrimitiveValue.utf8(providerId),
            "\$limit", PrimitiveValue.uint64(limit),
            "\$from_accounts_space_id", Ydb.nullToEmptyUtf8(fromAccountsSpaceId),
            "\$from_account_id", PrimitiveValue.utf8(fromAccountId))

    private fun toGetByProviderAccountSpaceFirstPageParams(tenantId: TenantId, providerId: ProviderId,
                                                           accountsSpaceId: AccountsSpacesId?, limit: Long) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$provider_id", PrimitiveValue.utf8(providerId),
            "\$accounts_space_id", Ydb.nullToEmptyUtf8(accountsSpaceId),
            "\$limit", PrimitiveValue.uint64(limit))

    private fun toGetByProviderAccountSpaceNextPageParams(tenantId: TenantId, providerId: ProviderId,
                                                          accountsSpaceId: AccountsSpacesId?, limit: Long,
                                                          fromAccountId: String) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$provider_id", PrimitiveValue.utf8(providerId),
            "\$accounts_space_id", Ydb.nullToEmptyUtf8(accountsSpaceId),
            "\$limit", PrimitiveValue.uint64(limit),
            "\$from_account_id", PrimitiveValue.utf8(fromAccountId))

}
