package ru.yandex.intranet.d.dao.aggregates

import com.fasterxml.jackson.core.type.TypeReference
import com.yandex.ydb.table.query.Params
import com.yandex.ydb.table.result.ResultSetReader
import com.yandex.ydb.table.values.ListValue
import com.yandex.ydb.table.values.PrimitiveValue
import com.yandex.ydb.table.values.StructValue
import kotlinx.coroutines.reactor.awaitSingle
import kotlinx.coroutines.reactor.awaitSingleOrNull
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.stereotype.Component
import ru.yandex.intranet.d.dao.DaoReader
import ru.yandex.intranet.d.dao.JsonFieldHelper
import ru.yandex.intranet.d.dao.WithTx
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource
import ru.yandex.intranet.d.datasource.model.YdbTxSession
import ru.yandex.intranet.d.model.TenantId
import ru.yandex.intranet.d.model.aggregates.ServiceDenormalizedAggregateAmounts
import ru.yandex.intranet.d.model.aggregates.ServiceDenormalizedAggregateKey
import ru.yandex.intranet.d.model.aggregates.ServiceDenormalizedAggregateKeyWithEpoch
import ru.yandex.intranet.d.model.aggregates.ServiceDenormalizedAggregateKeyWithEpochPage
import ru.yandex.intranet.d.model.aggregates.ServiceDenormalizedAggregateModel
import ru.yandex.intranet.d.util.ObjectMapperHolder
import ru.yandex.intranet.d.web.model.aggregation.RankSubtreeSortingOrder.DESC
import ru.yandex.intranet.d.web.model.aggregation.RankSubtreeSortingParamsDto

/**
 * Denormalized service aggregates DAO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
class ServiceDenormalizedAggregatesDao(private val ydbQuerySource: YdbQuerySource,
                                       @Qualifier("ydbJsonObjectMapper") private val objectMapper: ObjectMapperHolder) {

    private val exactFieldHelper: JsonFieldHelper<ServiceDenormalizedAggregateAmounts> = JsonFieldHelper(objectMapper,
        object : TypeReference<ServiceDenormalizedAggregateAmounts>() {})

    suspend fun getById(session: YdbTxSession,
                        id: ServiceDenormalizedAggregateKey): ServiceDenormalizedAggregateModel? {
        val query = ydbQuerySource.getQuery("yql.queries.serviceDenormalizedAggregates.getById")
        val params = toKeyParams(id)
        return DaoReader.toModel(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getByIds(session: YdbTxSession,
                         ids: Collection<ServiceDenormalizedAggregateKey>): List<ServiceDenormalizedAggregateModel> {
        if (ids.isEmpty()) {
            return listOf()
        }
        return ids.distinct().chunked(1000).map {
            val query = ydbQuerySource.getQuery("yql.queries.serviceDenormalizedAggregates.getByIds")
            val params = toKeyListParams(it)
            DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
        }.flatten()
    }

    suspend fun upsertOneRetryable(session: YdbTxSession, value: ServiceDenormalizedAggregateModel) {
        val query = ydbQuerySource.getQuery("yql.queries.serviceDenormalizedAggregates.upsertOne")
        val params = toUpsertOneParams(value)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun upsertManyRetryable(session: YdbTxSession, values: Collection<ServiceDenormalizedAggregateModel>) {
        if (values.isEmpty()) {
            return
        }
        val query = ydbQuerySource.getQuery("yql.queries.serviceDenormalizedAggregates.upsertMany")
        val params = toUpsertManyParams(values)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun deleteByIdRetryable(session: YdbTxSession, id: ServiceDenormalizedAggregateKey) {
        val query = ydbQuerySource.getQuery("yql.queries.serviceDenormalizedAggregates.deleteById")
        val params = toKeyParams(id)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun deleteByIdsRetryable(session: YdbTxSession, ids: Collection<ServiceDenormalizedAggregateKey>) {
        if (ids.isEmpty()) {
            return
        }
        val query = ydbQuerySource.getQuery("yql.queries.serviceDenormalizedAggregates.deleteByIds")
        val params = toKeyListParams(ids)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun getSubtreeDescendingTransferableFirstPage(session: YdbTxSession,
                                                          tenantId: TenantId,
                                                          subtreeRootServiceId: Long,
                                                          resourceId: String,
                                                          limit: Long): List<ServiceDenormalizedAggregateModel> {
        if (limit > 1000) {
            throw IllegalArgumentException("Limit is too large")
        }
        val query = ydbQuerySource
            .getQuery("yql.queries.serviceDenormalizedAggregates.getSubtreeDescendingTransferableFirstPage")
        val params = Params.of(
            "\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$subtree_root_service_id", PrimitiveValue.int64(subtreeRootServiceId),
            "\$resource_id", PrimitiveValue.utf8(resourceId),
            "\$limit", PrimitiveValue.uint64(limit)
        )
        return DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getSubtreeAscendingTransferableFirstPage(session: YdbTxSession,
                                                          tenantId: TenantId,
                                                          subtreeRootServiceId: Long,
                                                          resourceId: String,
                                                          limit: Long): List<ServiceDenormalizedAggregateModel> {
        if (limit > 1000) {
            throw IllegalArgumentException("Limit is too large")
        }
        val query = ydbQuerySource
            .getQuery("yql.queries.serviceDenormalizedAggregates.getSubtreeAscendingTransferableFirstPage")
        val params = Params.of(
            "\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$subtree_root_service_id", PrimitiveValue.int64(subtreeRootServiceId),
            "\$resource_id", PrimitiveValue.utf8(resourceId),
            "\$limit", PrimitiveValue.uint64(limit)
        )
        return DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getSubtreeDescendingTransferableNextPage(session: YdbTxSession,
                                                         tenantId: TenantId,
                                                         subtreeRootServiceId: Long,
                                                         resourceId: String,
                                                         fromTransferable: Long,
                                                         fromServiceId: Long,
                                                         limit: Long): List<ServiceDenormalizedAggregateModel> {
        if (limit > 1000) {
            throw IllegalArgumentException("Limit is too large")
        }
        val query = ydbQuerySource
            .getQuery("yql.queries.serviceDenormalizedAggregates.getSubtreeDescendingTransferableNextPage")
        val params = Params.of(
            "\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$subtree_root_service_id", PrimitiveValue.int64(subtreeRootServiceId),
            "\$resource_id", PrimitiveValue.utf8(resourceId),
            "\$from_transferable", PrimitiveValue.int64(fromTransferable),
            "\$from_service_id", PrimitiveValue.int64(fromServiceId),
            "\$limit", PrimitiveValue.uint64(limit)
        )
        return DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getSubtreeAscendingTransferableNextPage(session: YdbTxSession,
                                                         tenantId: TenantId,
                                                         subtreeRootServiceId: Long,
                                                         resourceId: String,
                                                         fromTransferable: Long,
                                                         fromServiceId: Long,
                                                         limit: Long): List<ServiceDenormalizedAggregateModel> {
        if (limit > 1000) {
            throw IllegalArgumentException("Limit is too large")
        }
        val query = ydbQuerySource
            .getQuery("yql.queries.serviceDenormalizedAggregates.getSubtreeAscendingTransferableNextPage")
        val params = Params.of(
            "\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$subtree_root_service_id", PrimitiveValue.int64(subtreeRootServiceId),
            "\$resource_id", PrimitiveValue.utf8(resourceId),
            "\$from_transferable", PrimitiveValue.int64(fromTransferable),
            "\$from_service_id", PrimitiveValue.int64(fromServiceId),
            "\$limit", PrimitiveValue.uint64(limit)
        )
        return DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getSubtreeDescendingDeallocatableFirstPage(session: YdbTxSession,
                                                           tenantId: TenantId,
                                                           subtreeRootServiceId: Long,
                                                           resourceId: String,
                                                           limit: Long): List<ServiceDenormalizedAggregateModel> {
        if (limit > 1000) {
            throw IllegalArgumentException("Limit is too large")
        }
        val query = ydbQuerySource
            .getQuery("yql.queries.serviceDenormalizedAggregates.getSubtreeDescendingDeallocatableFirstPage")
        val params = Params.of(
            "\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$subtree_root_service_id", PrimitiveValue.int64(subtreeRootServiceId),
            "\$resource_id", PrimitiveValue.utf8(resourceId),
            "\$limit", PrimitiveValue.uint64(limit)
        )
        return DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getSubtreeDescendingDeallocatableNextPage(session: YdbTxSession,
                                                          tenantId: TenantId,
                                                          subtreeRootServiceId: Long,
                                                          resourceId: String,
                                                          fromDeallocatable: Long,
                                                          fromServiceId: Long,
                                                          limit: Long): List<ServiceDenormalizedAggregateModel> {
        if (limit > 1000) {
            throw IllegalArgumentException("Limit is too large")
        }
        val query = ydbQuerySource
            .getQuery("yql.queries.serviceDenormalizedAggregates.getSubtreeDescendingDeallocatableNextPage")
        val params = Params.of(
            "\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$subtree_root_service_id", PrimitiveValue.int64(subtreeRootServiceId),
            "\$resource_id", PrimitiveValue.utf8(resourceId),
            "\$from_deallocatable", PrimitiveValue.int64(fromDeallocatable),
            "\$from_service_id", PrimitiveValue.int64(fromServiceId),
            "\$limit", PrimitiveValue.uint64(limit)
        )
        return DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getSubtreeWithCustomSortFirstPage(
        session: YdbTxSession,
        tenantId: TenantId,
        subtreeRootServiceId: Long,
        resourceId: String,
        limit: Long,
        sortingParams: RankSubtreeSortingParamsDto
    ): List<ServiceDenormalizedAggregateModel> {
        if (limit > 1000) {
            throw IllegalArgumentException("Limit is too large")
        }
        val query = if (sortingParams.order == DESC) {
            ydbQuerySource
                .getQuery("yql.queries.serviceDenormalizedAggregates.getSubtreeSortedByCustomFieldDescFirstPage")
        } else {
            ydbQuerySource
                .getQuery("yql.queries.serviceDenormalizedAggregates.getSubtreeSortedByCustomFieldAscFirstPage")
        }
            .replaceFirst("%%sorting_field%%", sortingParams.field.sortingField)

        val params = Params.of(
            "\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$subtree_root_service_id", PrimitiveValue.int64(subtreeRootServiceId),
            "\$resource_id", PrimitiveValue.utf8(resourceId),
            "\$limit", PrimitiveValue.uint64(limit)
        )
        return DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getSubtreeWithCustomSortNextPage(
        session: YdbTxSession,
        tenantId: TenantId,
        subtreeRootServiceId: Long,
        resourceId: String,
        fromSortingField: String,
        fromServiceId: Long,
        limit: Long,
        sortingParams: RankSubtreeSortingParamsDto
    ): List<ServiceDenormalizedAggregateModel> {
        if (limit > 1000) {
            throw IllegalArgumentException("Limit is too large")
        }
        val query = if (sortingParams.order == DESC) {
            ydbQuerySource
                .getQuery("yql.queries.serviceDenormalizedAggregates.getSubtreeSortedByCustomFieldDescNextPage")
        } else {
            ydbQuerySource
                .getQuery("yql.queries.serviceDenormalizedAggregates.getSubtreeSortedByCustomFieldAscNextPage")
        }
            .replace("%%sorting_field%%", sortingParams.field.sortingField)
        val params = Params.of(
            "\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$subtree_root_service_id", PrimitiveValue.int64(subtreeRootServiceId),
            "\$resource_id", PrimitiveValue.utf8(resourceId),
            "\$from_sorting_field", PrimitiveValue.utf8(fromSortingField),
            "\$from_service_id", PrimitiveValue.int64(fromServiceId),
            "\$limit", PrimitiveValue.uint64(limit)
        )
        return DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getSubtreeDescendingTransferableMultiResourceFirstPage(
        session: YdbTxSession, tenantId: TenantId, subtreeRootServiceId: Long, resourceIds: Collection<String>,
        limit: Long
    ): List<ServiceDenormalizedAggregateModel> {
        if (limit > 1000) {
            throw IllegalArgumentException("Limit is too large")
        }
        val query = ydbQuerySource.getQuery(
            "yql.queries.serviceDenormalizedAggregates.getSubtreeDescendingTransferableMultiResourceFirstPage")
        val params = Params.of(
            "\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$subtree_root_service_id", PrimitiveValue.int64(subtreeRootServiceId),
            "\$resource_ids", ListValue.of(*resourceIds.map { PrimitiveValue.utf8(it) }.toTypedArray()),
            "\$limit", PrimitiveValue.uint64(limit)
        )
        return DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getSubtreeDescendingTransferableMultiResourceNextPage(
        session: YdbTxSession, tenantId: TenantId, subtreeRootServiceId: Long, resourceIds: Collection<String>,
        fromResourceId: String, fromTransferable: Long, fromServiceId: Long, limit: Long
    ): List<ServiceDenormalizedAggregateModel> {
        if (limit > 1000) {
            throw IllegalArgumentException("Limit is too large")
        }
        val resourceIdsTail = resourceIds.distinct().sortedDescending().filter { it < fromResourceId }
        val (query, params) = if (resourceIdsTail.isNotEmpty()) {
            val nextQuery = ydbQuerySource.getQuery(
                "yql.queries.serviceDenormalizedAggregates.getSubtreeDescendingTransferableMultiResourceNextPage")
            val nextParams = Params.copyOf(mapOf(
                "\$tenant_id" to PrimitiveValue.utf8(tenantId.id),
                "\$subtree_root_service_id" to PrimitiveValue.int64(subtreeRootServiceId),
                "\$resource_ids" to ListValue.of(*resourceIds.map { PrimitiveValue.utf8(it) }.toTypedArray()),
                "\$from_resource_ids" to ListValue.of(*resourceIdsTail.map { PrimitiveValue.utf8(it) }.toTypedArray()),
                "\$from_resource_id" to PrimitiveValue.utf8(fromResourceId),
                "\$from_transferable" to PrimitiveValue.int64(fromTransferable),
                "\$from_service_id" to PrimitiveValue.int64(fromServiceId),
                "\$limit" to PrimitiveValue.uint64(limit)
            ))
            Pair(nextQuery, nextParams)
        } else {
            val lastQuery = ydbQuerySource.getQuery(
                "yql.queries.serviceDenormalizedAggregates.getSubtreeDescendingTransferableMultiResourceLastPage")
            val lastParams = Params.copyOf(mapOf(
                "\$tenant_id" to PrimitiveValue.utf8(tenantId.id),
                "\$subtree_root_service_id" to PrimitiveValue.int64(subtreeRootServiceId),
                "\$resource_ids" to ListValue.of(*resourceIds.map { PrimitiveValue.utf8(it) }.toTypedArray()),
                "\$from_resource_id" to PrimitiveValue.utf8(fromResourceId),
                "\$from_transferable" to PrimitiveValue.int64(fromTransferable),
                "\$from_service_id" to PrimitiveValue.int64(fromServiceId),
                "\$limit" to PrimitiveValue.uint64(limit)
            ))
            Pair(lastQuery, lastParams)
        }
        return DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getSubtreeDescendingDeallocatableMultiResourceFirstPage(
        session: YdbTxSession, tenantId: TenantId, subtreeRootServiceId: Long, resourceIds: Collection<String>,
        limit: Long
    ): List<ServiceDenormalizedAggregateModel> {
        if (limit > 1000) {
            throw IllegalArgumentException("Limit is too large")
        }
        val query = ydbQuerySource.getQuery(
            "yql.queries.serviceDenormalizedAggregates.getSubtreeDescendingDeallocatableMultiResourceFirstPage")
        val params = Params.of(
            "\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$subtree_root_service_id", PrimitiveValue.int64(subtreeRootServiceId),
            "\$resource_ids", ListValue.of(*resourceIds.map { PrimitiveValue.utf8(it) }.toTypedArray()),
            "\$limit", PrimitiveValue.uint64(limit)
        )
        return DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getSubtreeDescendingDeallocatableMultiResourceNextPage(
        session: YdbTxSession, tenantId: TenantId, subtreeRootServiceId: Long, resourceIds: Collection<String>,
        fromResourceId: String, fromDeallocatable: Long, fromServiceId: Long, limit: Long
    ): List<ServiceDenormalizedAggregateModel> {
        if (limit > 1000) {
            throw IllegalArgumentException("Limit is too large")
        }
        val resourceIdsTail = resourceIds.distinct().sortedDescending().filter { it < fromResourceId }
        val (query, params) = if (resourceIdsTail.isNotEmpty()) {
            val nextQuery = ydbQuerySource.getQuery(
                "yql.queries.serviceDenormalizedAggregates.getSubtreeDescendingDeallocatableMultiResourceNextPage")
            val nextParams = Params.copyOf(mapOf(
                "\$tenant_id" to PrimitiveValue.utf8(tenantId.id),
                "\$subtree_root_service_id" to PrimitiveValue.int64(subtreeRootServiceId),
                "\$resource_ids" to ListValue.of(*resourceIds.map { PrimitiveValue.utf8(it) }.toTypedArray()),
                "\$from_resource_ids" to ListValue.of(*resourceIdsTail.map { PrimitiveValue.utf8(it) }.toTypedArray()),
                "\$from_resource_id" to PrimitiveValue.utf8(fromResourceId),
                "\$from_deallocatable" to PrimitiveValue.int64(fromDeallocatable),
                "\$from_service_id" to PrimitiveValue.int64(fromServiceId),
                "\$limit" to PrimitiveValue.uint64(limit)
            ))
            Pair(nextQuery, nextParams)
        } else {
            val lastQuery = ydbQuerySource.getQuery(
                "yql.queries.serviceDenormalizedAggregates.getSubtreeDescendingDeallocatableMultiResourceLastPage")
            val lastParams = Params.copyOf(mapOf(
                "\$tenant_id" to PrimitiveValue.utf8(tenantId.id),
                "\$subtree_root_service_id" to PrimitiveValue.int64(subtreeRootServiceId),
                "\$resource_ids" to ListValue.of(*resourceIds.map { PrimitiveValue.utf8(it) }.toTypedArray()),
                "\$from_resource_id" to PrimitiveValue.utf8(fromResourceId),
                "\$from_deallocatable" to PrimitiveValue.int64(fromDeallocatable),
                "\$from_service_id" to PrimitiveValue.int64(fromServiceId),
                "\$limit" to PrimitiveValue.uint64(limit)
            ))
            Pair(lastQuery, lastParams)
        }
        return DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getKeysForOlderEpochsFirstPage(session: YdbTxSession,
                                               tenantId: TenantId,
                                               providerId: String,
                                               resourceId: String,
                                               currentEpoch: Long): WithTx<ServiceDenormalizedAggregateKeyWithEpochPage> {
        val query = ydbQuerySource.getQuery("yql.queries.serviceDenormalizedAggregates.getKeysForOlderEpochsFirstPage")
        val params = toGetKeysForOlderEpochsFirstPageParams(tenantId, providerId, resourceId, currentEpoch, 1000)
        val page = DaoReader.toModelsWithTx(session.executeDataQueryRetryable(query, params).awaitSingle(),
            this::toKeyWithEpoch)
        return WithTx(ServiceDenormalizedAggregateKeyWithEpochPage(keys = page.value,
            nextFrom = if (page.value.size >= 1000) { page.value.last() } else { null }), page.txId)
    }

    suspend fun getKeysForOlderEpochsNextPage(session: YdbTxSession,
                                              from: ServiceDenormalizedAggregateKeyWithEpoch
    ): WithTx<ServiceDenormalizedAggregateKeyWithEpochPage> {
        val query = ydbQuerySource.getQuery("yql.queries.serviceDenormalizedAggregates.getKeysForOlderEpochsNextPage")
        val params = toGetKeysForOlderEpochsNextPageParams(from, 1000)
        val page = DaoReader.toModelsWithTx(session.executeDataQueryRetryable(query, params).awaitSingle(),
            this::toKeyWithEpoch)
        return WithTx(ServiceDenormalizedAggregateKeyWithEpochPage(keys = page.value,
            nextFrom = if (page.value.size >= 1000) { page.value.last() } else { null }), page.txId)
    }

    suspend fun getKeysForOlderEpochsMultiResourceFirstPage(
        session: YdbTxSession, tenantId: TenantId, providerId: String, resourceIds: Collection<String>,
        currentEpoch: Long): WithTx<ServiceDenormalizedAggregateKeyWithEpochPage> {
        val query = ydbQuerySource
            .getQuery("yql.queries.serviceDenormalizedAggregates.getKeysForOlderEpochsMultiResourceFirstPage")
        val params = toGetKeysForOlderEpochsMultiResourceFirstPageParams(tenantId, providerId, resourceIds,
            currentEpoch, 1000)
        val page = DaoReader.toModelsWithTx(session.executeDataQueryRetryable(query, params).awaitSingle(),
            this::toKeyWithEpoch)
        return WithTx(ServiceDenormalizedAggregateKeyWithEpochPage(keys = page.value,
            nextFrom = if (page.value.size >= 1000) { page.value.last() } else { null }), page.txId)
    }

    suspend fun getKeysForOlderEpochsMultiResourceNextPage(
        session: YdbTxSession,
        from: ServiceDenormalizedAggregateKeyWithEpoch,
        resourceIds: Collection<String>,
        currentEpoch: Long
    ): WithTx<ServiceDenormalizedAggregateKeyWithEpochPage> {
        val sortedResourceIds = resourceIds.sortedDescending()
        val resourceIdsTail = sortedResourceIds.filter { it < from.key.resourceId }
        val (query, params) = if (resourceIdsTail.isNotEmpty()) {
            val nextQuery = ydbQuerySource
                .getQuery("yql.queries.serviceDenormalizedAggregates.getKeysForOlderEpochsMultiResourceNextPage")
            val nextParams = toGetKeysForOlderEpochsMultiResourceNextPageParams(from, resourceIdsTail,
                currentEpoch, 1000)
            Pair(nextQuery, nextParams)
        } else {
            val lastQuery = ydbQuerySource
                .getQuery("yql.queries.serviceDenormalizedAggregates.getKeysForOlderEpochsMultiResourceLastPage")
            val lastParams = toGetKeysForOlderEpochsMultiResourceLastPageParams(from, 1000)
            Pair(lastQuery, lastParams)
        }
        val page = DaoReader.toModelsWithTx(session.executeDataQueryRetryable(query, params).awaitSingle(),
            this::toKeyWithEpoch)
        return WithTx(ServiceDenormalizedAggregateKeyWithEpochPage(keys = page.value,
            nextFrom = if (page.value.size >= 1000) { page.value.last() } else { null }), page.txId)
    }

    private fun toModel(reader: ResultSetReader) = ServiceDenormalizedAggregateModel(
        key = toKey(reader),
        lastUpdate = reader.getColumn("last_update").timestamp,
        epoch = reader.getColumn("epoch").int64,
        providerId = reader.getColumn("provider_id").utf8,
        transferable = reader.getColumn("transferable").int64,
        deallocatable = reader.getColumn("deallocatable").int64,
        exactAmounts = exactFieldHelper.read(reader.getColumn("exact_amounts"))!!
    )

    private fun toKey(reader: ResultSetReader) = ServiceDenormalizedAggregateKey(
        tenantId = TenantId(reader.getColumn("tenant_id").utf8),
        superTreeServiceId = reader.getColumn("super_tree_service_id").int64,
        resourceId = reader.getColumn("resource_id").utf8,
        serviceId = reader.getColumn("service_id").int64,
    )

    private fun toKeyWithEpoch(reader: ResultSetReader) = ServiceDenormalizedAggregateKeyWithEpoch(
        key = toKey(reader),
        providerId = reader.getColumn("provider_id").utf8,
        epoch = reader.getColumn("epoch").int64
    )

    private fun toKeyParams(id: ServiceDenormalizedAggregateKey) = Params.of(
        "\$id", toKeyStruct(id)
    )

    private fun toKeyListParams(ids: Collection<ServiceDenormalizedAggregateKey>) = Params.of(
        "\$ids", ListValue.of(*ids.map { toKeyStruct(it) }.toTypedArray())
    )

    private fun toKeyStruct(id: ServiceDenormalizedAggregateKey) = StructValue.of(mapOf(
        "tenant_id" to PrimitiveValue.utf8(id.tenantId.id),
        "super_tree_service_id" to PrimitiveValue.int64(id.superTreeServiceId),
        "resource_id" to PrimitiveValue.utf8(id.resourceId),
        "service_id" to PrimitiveValue.int64(id.serviceId),
    ))

    private fun toUpsertOneParams(value: ServiceDenormalizedAggregateModel) = Params.of(
        "\$value", toUpsertStruct(value)
    )

    private fun toUpsertManyParams(values: Collection<ServiceDenormalizedAggregateModel>) = Params.of(
        "\$values", ListValue.of(*values.map { toUpsertStruct(it) }.toTypedArray())
    )

    private fun toUpsertStruct(value: ServiceDenormalizedAggregateModel) = StructValue.of(
        mapOf(
            "tenant_id" to PrimitiveValue.utf8(value.key.tenantId.id),
            "super_tree_service_id" to PrimitiveValue.int64(value.key.superTreeServiceId),
            "resource_id" to PrimitiveValue.utf8(value.key.resourceId),
            "service_id" to PrimitiveValue.int64(value.key.serviceId),
            "last_update" to PrimitiveValue.timestamp(value.lastUpdate),
            "epoch" to PrimitiveValue.int64(value.epoch),
            "provider_id" to PrimitiveValue.utf8(value.providerId),
            "transferable" to PrimitiveValue.int64(value.transferable),
            "deallocatable" to PrimitiveValue.int64(value.deallocatable),
            "exact_amounts" to exactFieldHelper.write(value.exactAmounts)
        )
    )

    private fun toGetKeysForOlderEpochsFirstPageParams(tenantId: TenantId, providerId: String, resourceId: String,
                                                       currentEpoch: Long, limit: Long) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$provider_id", PrimitiveValue.utf8(providerId),
            "\$resource_id", PrimitiveValue.utf8(resourceId),
            "\$current_epoch", PrimitiveValue.int64(currentEpoch),
            "\$limit", PrimitiveValue.uint64(limit)
        )

    private fun toGetKeysForOlderEpochsNextPageParams(from: ServiceDenormalizedAggregateKeyWithEpoch,
                                                      limit: Long) = Params
        .copyOf(mapOf("\$tenant_id" to PrimitiveValue.utf8(from.key.tenantId.id),
            "\$provider_id" to PrimitiveValue.utf8(from.providerId),
            "\$resource_id" to PrimitiveValue.utf8(from.key.resourceId),
            "\$from_epoch" to PrimitiveValue.int64(from.epoch),
            "\$from_super_tree_service_id" to PrimitiveValue.int64(from.key.superTreeServiceId),
            "\$from_service_id" to PrimitiveValue.int64(from.key.serviceId),
            "\$limit" to PrimitiveValue.uint64(limit)
        ))

    private fun toGetKeysForOlderEpochsMultiResourceFirstPageParams(tenantId: TenantId, providerId: String,
                                                                    resourceIds: Collection<String>, currentEpoch: Long,
                                                                    limit: Long) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(tenantId.id),
            "\$provider_id", PrimitiveValue.utf8(providerId),
            "\$resource_ids", ListValue.of(*resourceIds.map { PrimitiveValue.utf8(it) }.toTypedArray()),
            "\$current_epoch", PrimitiveValue.int64(currentEpoch),
            "\$limit", PrimitiveValue.uint64(limit)
        )

    private fun toGetKeysForOlderEpochsMultiResourceNextPageParams(from: ServiceDenormalizedAggregateKeyWithEpoch,
                                                                   resourceIds: Collection<String>, currentEpoch: Long,
                                                                   limit: Long) = Params
        .of("\$tenant_id", PrimitiveValue.utf8(from.key.tenantId.id),
            "\$provider_id", PrimitiveValue.utf8(from.providerId),
            "\$resource_ids", ListValue.of(*resourceIds.map { PrimitiveValue.utf8(it) }.toTypedArray()),
            "\$from_resource_id", PrimitiveValue.utf8(from.key.resourceId),
            "\$current_epoch", PrimitiveValue.int64(currentEpoch),
            "\$from_epoch", PrimitiveValue.int64(from.epoch),
            "\$from_super_tree_service_id", PrimitiveValue.int64(from.key.superTreeServiceId),
            "\$from_service_id", PrimitiveValue.int64(from.key.serviceId),
            "\$limit", PrimitiveValue.uint64(limit)
        )

    private fun toGetKeysForOlderEpochsMultiResourceLastPageParams(from: ServiceDenormalizedAggregateKeyWithEpoch,
                                                                   limit: Long) = Params
        .copyOf(mapOf("\$tenant_id" to PrimitiveValue.utf8(from.key.tenantId.id),
            "\$provider_id" to PrimitiveValue.utf8(from.providerId),
            "\$from_resource_id" to PrimitiveValue.utf8(from.key.resourceId),
            "\$from_epoch" to PrimitiveValue.int64(from.epoch),
            "\$from_super_tree_service_id" to PrimitiveValue.int64(from.key.superTreeServiceId),
            "\$from_service_id" to PrimitiveValue.int64(from.key.serviceId),
            "\$limit" to PrimitiveValue.uint64(limit)
        ))
}
