package ru.yandex.intranet.d.dao.delivery;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import com.fasterxml.jackson.core.type.TypeReference;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.Value;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import ru.yandex.intranet.d.dao.AbstractDao;
import ru.yandex.intranet.d.dao.JsonFieldHelper;
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.delivery.provide.DeliveryAndProvideModel;
import ru.yandex.intranet.d.model.delivery.provide.DeliveryAndProvideOperationListModel;
import ru.yandex.intranet.d.model.delivery.provide.DeliveryAndProvideRequestModel;
import ru.yandex.intranet.d.util.ObjectMapperHolder;

/**
 * Finished deliveries and provided DAO.
 *
 * @author Evgenii Serov <evserov@yandex-team.ru>
 */
@Component
public class DeliveriesAndProvidedRequestsDao extends AbstractDao<DeliveryAndProvideModel, String> {

    private final JsonFieldHelper<DeliveryAndProvideRequestModel> deliveryRequestHelper;
    private final JsonFieldHelper<List<DeliveryAndProvideOperationListModel>> operationsHelper;

    public DeliveriesAndProvidedRequestsDao(YdbQuerySource ydbQuerySource,
                                            @Qualifier("ydbJsonObjectMapper") ObjectMapperHolder objectMapper) {
        super(ydbQuerySource);
        this.deliveryRequestHelper = new JsonFieldHelper<>(objectMapper,
                new TypeReference<>() { });
        this.operationsHelper = new JsonFieldHelper<>(objectMapper, new TypeReference<>() { });
    }

    @Override
    protected WithTenant<String> getIdentityWithTenant(DeliveryAndProvideModel model) {
        return new WithTenant<>(model.getTenantId(), model.getDeliveryId());
    }

    @Override
    protected Params getIdentityParams(String deliveryId) {
        return Params.create()
                .put("$delivery_id", PrimitiveValue.utf8(deliveryId));
    }

    @SuppressWarnings("rawtypes")
    @Override
    protected Map<String, Value> prepareFieldValues(DeliveryAndProvideModel model) {
        HashMap<String, Value> fields = new HashMap<>();
        fields.put(Fields.DELIVERY_ID.field(), PrimitiveValue.utf8(model.getDeliveryId()));
        fields.put(Fields.TENANT_ID.field(), PrimitiveValue.utf8(model.getTenantId().getId()));
        fields.put(Fields.REQUEST.field(), deliveryRequestHelper.write(model.getRequest()));
        fields.put(Fields.OPERATIONS.field(), operationsHelper.write(model.getOperations()));
        return fields;
    }

    @Override
    protected DeliveryAndProvideModel readOneRow(ResultSetReader reader, Map<String, TenantId> tenantIdCache) {
        return new DeliveryAndProvideModel.Builder()
                .deliveryId(reader.getColumn(Fields.DELIVERY_ID.field()).getUtf8())
                .tenantId(tenantIdCache.computeIfAbsent(reader
                        .getColumn(Fields.TENANT_ID.field()).getUtf8(), TenantId::new))
                .request(Objects.requireNonNull(deliveryRequestHelper.read(
                        reader.getColumn(Fields.REQUEST.field()))))
                .addOperations(Objects.requireNonNull(operationsHelper.read(
                        reader.getColumn(Fields.OPERATIONS.field()))))
                .build();
    }

    @Override
    protected String queryKeyPrefix() {
        return "yql.queries.deliveriesAndProvidedRequests";
    }

    public enum Fields {
        TENANT_ID,
        DELIVERY_ID,
        REQUEST,
        OPERATIONS;

        public String field() {
            return name().toLowerCase();
        }
    }

}
