package ru.yandex.intranet.d.dao.delivery;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.core.type.TypeReference;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.Value;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import ru.yandex.intranet.d.dao.AbstractDao;
import ru.yandex.intranet.d.dao.JsonFieldHelper;
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.delivery.DeliveryRequestModel;
import ru.yandex.intranet.d.model.delivery.DeliveryResponseModel;
import ru.yandex.intranet.d.model.delivery.FinishedDeliveryModel;
import ru.yandex.intranet.d.util.ObjectMapperHolder;

/**
 * Finished deliveries DAO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class FinishedDeliveriesDao extends AbstractDao<FinishedDeliveryModel, String> {

    private final JsonFieldHelper<DeliveryRequestModel> deliveryRequestHelper;
    private final JsonFieldHelper<DeliveryResponseModel> deliveryResponseHelper;
    private final JsonFieldHelper<List<String>> operationIdsHelper;

    public FinishedDeliveriesDao(YdbQuerySource ydbQuerySource,
                                 @Qualifier("ydbJsonObjectMapper") ObjectMapperHolder objectMapper) {
        super(ydbQuerySource);
        this.deliveryRequestHelper = new JsonFieldHelper<>(objectMapper,
                new TypeReference<DeliveryRequestModel>() { });
        this.deliveryResponseHelper = new JsonFieldHelper<>(objectMapper,
                new TypeReference<DeliveryResponseModel>() { });
        this.operationIdsHelper = new JsonFieldHelper<>(objectMapper, new TypeReference<List<String>>() { });
    }

    @Override
    protected WithTenant<String> getIdentityWithTenant(FinishedDeliveryModel model) {
        return new WithTenant<>(model.getTenantId(), model.getId());
    }

    @Override
    protected Params getIdentityParams(String id) {
        return Params.create()
                .put("$id", PrimitiveValue.utf8(id));
    }

    @SuppressWarnings("rawtypes")
    @Override
    protected Map<String, Value> prepareFieldValues(FinishedDeliveryModel model) {
        HashMap<String, Value> fields = new HashMap<>();
        fields.put(Fields.ID.field(), PrimitiveValue.utf8(model.getId()));
        fields.put(Fields.TENANT_ID.field(), PrimitiveValue.utf8(model.getTenantId().getId()));
        fields.put(Fields.REQUEST.field(), deliveryRequestHelper.write(model.getRequest()));
        fields.put(Fields.RESPONSE.field(), deliveryResponseHelper.write(model.getResponse()));
        fields.put(Fields.OPERATION_IDS.field(), operationIdsHelper.write(model.getOperationIds()));
        return fields;
    }

    @Override
    protected FinishedDeliveryModel readOneRow(ResultSetReader reader, Map<String, TenantId> tenantIdCache) {
        return FinishedDeliveryModel.builder()
                .id(reader.getColumn(Fields.ID.field()).getUtf8())
                .tenantId(tenantIdCache.computeIfAbsent(reader
                        .getColumn(Fields.TENANT_ID.field()).getUtf8(), TenantId::new))
                .request(deliveryRequestHelper.read(reader.getColumn(Fields.REQUEST.field())))
                .response(deliveryResponseHelper.read(reader.getColumn(Fields.RESPONSE.field())))
                .operationIds(operationIdsHelper.read(reader.getColumn(Fields.OPERATION_IDS.field())))
                .build();
    }

    @Override
    protected String queryKeyPrefix() {
        return "yql.queries.finishedDeliveries";
    }

    public enum Fields {
        TENANT_ID,
        ID,
        REQUEST,
        RESPONSE,
        OPERATION_IDS;

        public String field() {
            return name().toLowerCase();
        }
    }

}
