package ru.yandex.intranet.d.dao.folders;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;

import com.yandex.ydb.table.query.DataQueryResult;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.ListValue;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.TupleValue;
import com.yandex.ydb.table.values.Value;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;

import ru.yandex.intranet.d.dao.AbstractDaoWithStringId;
import ru.yandex.intranet.d.dao.QueryUtils;
import ru.yandex.intranet.d.datasource.Ydb;
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource;
import ru.yandex.intranet.d.datasource.model.WithTxId;
import ru.yandex.intranet.d.datasource.model.YdbReadTableSettings;
import ru.yandex.intranet.d.datasource.model.YdbSession;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.StringIdWithTenant;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.folders.FolderType;

import static ru.yandex.intranet.d.dao.folders.FolderDao.Fields.DELETED;
import static ru.yandex.intranet.d.dao.folders.FolderDao.Fields.DESCRIPTION;
import static ru.yandex.intranet.d.dao.folders.FolderDao.Fields.DISPLAY_NAME;
import static ru.yandex.intranet.d.dao.folders.FolderDao.Fields.FOLDER_TYPE;
import static ru.yandex.intranet.d.dao.folders.FolderDao.Fields.ID;
import static ru.yandex.intranet.d.dao.folders.FolderDao.Fields.NEXT_OP_LOG_ORDER;
import static ru.yandex.intranet.d.dao.folders.FolderDao.Fields.SERVICE_ID;
import static ru.yandex.intranet.d.dao.folders.FolderDao.Fields.TAGS;
import static ru.yandex.intranet.d.dao.folders.FolderDao.Fields.TENANT_ID;
import static ru.yandex.intranet.d.dao.folders.FolderDao.Fields.VERSION;
import static ru.yandex.intranet.d.datasource.Ydb.nullableUtf8;

/**
 * Folder DAO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class FolderDao extends AbstractDaoWithStringId<FolderModel> {
    private final FolderTagsHelper tagsHelper;

    public FolderDao(
            YdbQuerySource ydbQuerySource,
            FolderTagsHelper tagsHelper
    ) {
        super(ydbQuerySource);
        this.tagsHelper = tagsHelper;
    }

    public Mono<WithTxId<List<FolderModel>>> listFoldersByServiceIdWithoutDefaultTx(
            YdbTxSession session,
            TenantId tenantId, long serviceId, boolean includeDeleted,
            int limit, @Nullable String fromId
    ) {
        String query = ydbQuerySource.getQuery("yql.queries.folders.getPageByServiceIdWithoutDefault");
        Params params = Params.of(
                "$service_id", PrimitiveValue.int64(serviceId),
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$include_deleted", PrimitiveValue.bool(includeDeleted),
                "$from_id", Ydb.nullableUtf8(fromId),
                "$limit", PrimitiveValue.uint64(limit)
        );
        return session.executeDataQueryRetryable(query, params)
                .map(r -> new WithTxId<>(toModels(r), r.getTxId()));
    }

    public Mono<WithTxId<Optional<FolderModel>>> getDefaultFolderTx(
            YdbTxSession session,
            TenantId tenantId,
            long serviceId
    ) {
        String query = ydbQuerySource.getQuery("yql.queries.folders.getDefaultFolder");
        Params params = Params.of(
                "$service_id", PrimitiveValue.int64(serviceId),
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId())
        );
        return session.executeDataQueryRetryable(query, params)
                .map(r -> new WithTxId<>(toModel(r), r.getTxId()));
    }

    public Mono<List<FolderModel>> getAllDefaultFoldersByServiceIds(YdbTxSession session,
                                                                    List<WithTenant<Long>> ids) {
        String query = ydbQuerySource.getQuery("yql.queries.folders.getAllDefaultFoldersByServiceIds");
        ListValue idsParam = ListValue.of(ids.stream()
                .map(id -> TupleValue.of(PrimitiveValue.int64(id.getIdentity()),
                        PrimitiveValue.utf8(id.getTenantId().getId()))).toArray(TupleValue[]::new));
        return session.executeDataQueryRetryable(query, Params.of("$ids", idsParam)).map(this::toModels);
    }

    public Mono<List<FolderModel>> getAllReservedFoldersByServiceIds(YdbTxSession session,
                                                                     List<WithTenant<Long>> ids) {
        String query = ydbQuerySource.getQuery("yql.queries.folders.getAllReservedFoldersByServiceIds");
        ListValue idsParam = ListValue.of(ids.stream()
                .map(id -> TupleValue.of(PrimitiveValue.int64(id.getIdentity()),
                        PrimitiveValue.utf8(id.getTenantId().getId()))).toArray(TupleValue[]::new));
        return session.executeDataQueryRetryable(query, Params.of("$ids", idsParam)).map(this::toModels);
    }

    public Mono<WithTxId<List<FolderModel>>> getAllFoldersByServiceIds(YdbTxSession session,
                                                                       Set<WithTenant<Long>> ids) {
        String query = ydbQuerySource.getQuery("yql.queries.folders.getAllFoldersByServiceIds");
        return QueryUtils.getAllRows(session,
                (nextSession, lastId) -> {
                    ListValue idsParam = ListValue.of(ids.stream()
                            .map(id -> TupleValue.of(PrimitiveValue.int64(id.getIdentity()),
                                    PrimitiveValue.utf8(id.getTenantId().getId()))).toArray(TupleValue[]::new));
                    return nextSession.executeDataQueryRetryable(query, Params.of("$ids", idsParam,
                            "$from_id", nullableUtf8(lastId)));
                },
                this::toModels,
                FolderModel::getId);
    }

    public Mono<WithTxId<Set<Long>>> getServiceIdsWithExistingDefaultFolderTx(
            YdbTxSession session,
            Set<Tuple2<Long, String>> serviceIdsWithTenantIds) {
        if (serviceIdsWithTenantIds.isEmpty()) {
            return Mono.error(new IllegalArgumentException("At least one service id is required"));
        }
        String query = ydbQuerySource.getQuery("yql.queries.folders.getDefaultFolderServiceIdsByServicesIds");
        ListValue idsParam = ListValue.of(serviceIdsWithTenantIds.stream()
                .map(id -> TupleValue.of(PrimitiveValue.int64(id.getT1()),
                        PrimitiveValue.utf8(id.getT2()))).toArray(TupleValue[]::new));
        return session.executeDataQueryRetryable(query, Params.of("$ids", idsParam))
                .map(r -> new WithTxId<>(toServiceIds(r), r.getTxId()));
    }

    public Mono<List<FolderModel>> listFolders(YdbTxSession session, TenantId tenantId, String folderIdFrom,
                                               int limit, boolean withDeleted) {
        String query;
        Params params;
        if (folderIdFrom != null) {
            query = ydbQuerySource.getQuery("yql.queries.folders.getNextPage");
            params = Params.of("$from_id", PrimitiveValue.utf8(folderIdFrom),
                    "$limit", PrimitiveValue.uint64(limit),
                    "$include_deleted", PrimitiveValue.bool(withDeleted),
                    "$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        } else {
            query = ydbQuerySource.getQuery("yql.queries.folders.getFirstPage");
            params = Params.of("$limit", PrimitiveValue.uint64(limit),
                    "$include_deleted", PrimitiveValue.bool(withDeleted),
                    "$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        }
        return session.executeDataQueryRetryable(query, params).map(this::toModels);
    }


    public Mono<List<FolderModel>> getFolderWithAccounts(YdbTxSession session, long serviceId, String providerId,
                                                         TenantId tenantId, int limit, String fromId) {

        String query = ydbQuerySource.getQuery("yql.queries.accounts.getFoldersWithAccountsByServiceAndProvider");
        Params params = Params.of(
                "$service_id", PrimitiveValue.int64(serviceId),
                "$provider_id", PrimitiveValue.utf8(providerId),
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$limit", PrimitiveValue.uint64(limit),
                "$from", Ydb.nullableUtf8(fromId));

        return session.executeDataQueryRetryable(query, params).map(this::toModels);
    }

    public Flux<String> getAllFoldersIds(YdbSession session, String tenantId) {
        Objects.requireNonNull(tenantId, "Tenant id is required.");
        return session.readTable(ydbQuerySource.preprocessTableName("folders"),
                YdbReadTableSettings.builder().ordered(true)
                        .addColumns("id", "tenant_id")
                        .build())
                .flatMapIterable(reader -> {
                    List<String> page = new ArrayList<>();
                    while (reader.next()) {
                        if (reader.getColumn("tenant_id").getUtf8().equals(tenantId)) {
                            page.add(reader.getColumn("id").getUtf8());
                        }
                    }
                    return page;
                });
    }

    @Override
    protected String queryKeyPrefix() {
        return "yql.queries.folders";
    }

    @Override
    protected StringIdWithTenant getIdWithTenant(FolderModel model) {
        return new StringIdWithTenant(model.getTenantId(), model.getId());
    }

    @Override
    @SuppressWarnings("rawtypes")
    protected Map<String, Value> prepareFieldValues(FolderModel folder) {
        HashMap<String, Value> fields = new HashMap<>();

        fields.put(ID.field(), PrimitiveValue.utf8(folder.getId()));
        fields.put(TENANT_ID.field(), PrimitiveValue.utf8(folder.getTenantId().getId()));
        fields.put(SERVICE_ID.field(), PrimitiveValue.int64(folder.getServiceId()));
        fields.put(VERSION.field(), PrimitiveValue.int64(folder.getVersion()));
        fields.put(DISPLAY_NAME.field(), PrimitiveValue.utf8(folder.getDisplayName()));
        fields.put(DESCRIPTION.field(), Ydb.nullableUtf8(folder.getDescription().orElse(null)));
        fields.put(DELETED.field(), PrimitiveValue.bool(folder.isDeleted()));
        fields.put(FOLDER_TYPE.field(), PrimitiveValue.utf8(folder.getFolderType().name()));
        fields.put(TAGS.field(), tagsHelper.writeTags(folder.getTags()));
        fields.put(NEXT_OP_LOG_ORDER.field(), PrimitiveValue.int64(folder.getNextOpLogOrder()));

        return fields;
    }

    @Override
    public FolderModel readOneRow(ResultSetReader reader, Map<String, TenantId> tenantIdCache) {
        String id = reader.getColumn(ID.field()).getUtf8();
        TenantId tenantId = TenantId.getInstance(reader.getColumn(TENANT_ID.field()).getUtf8(), tenantIdCache);
        long serviceId = reader.getColumn(SERVICE_ID.field()).getInt64();
        long version = reader.getColumn(VERSION.field()).getInt64();
        String displayName = reader.getColumn(DISPLAY_NAME.field()).getUtf8();
        String description = Ydb.utf8OrNull(reader.getColumn(DESCRIPTION.field()));
        boolean deleted = reader.getColumn(DELETED.field()).getBool();
        FolderType folderType = FolderType.valueOf(reader.getColumn(FOLDER_TYPE.field()).getUtf8());
        Set<String> tags = tagsHelper.readTags(reader.getColumn(TAGS.field()));
        long nextOpLogOrder = reader.getColumn(NEXT_OP_LOG_ORDER.field()).getInt64();

        return new FolderModel(
                id,
                tenantId,
                serviceId,
                version,
                displayName,
                description,
                deleted,
                folderType,
                tags,
                nextOpLogOrder
        );
    }

    @Override
    public List<FolderModel> toModels(DataQueryResult result) {
        return super.toModels(result);
    }

    private Set<Long> toServiceIds(DataQueryResult result) {
        if (result.getResultSetCount() != 1) {
            throw new IllegalStateException("Exactly one result set is required");
        }
        ResultSetReader reader = result.getResultSet(0);
        Set<Long> ids = new HashSet<>();
        while (reader.next()) {
            ids.add(reader.getColumn("service_id").getInt64());
        }
        return ids;
    }

    @SuppressWarnings({"unused", "RedundantSuppression"})
    public enum Fields {
        ID,
        TENANT_ID,
        SERVICE_ID,
        VERSION,
        DISPLAY_NAME,
        DESCRIPTION,
        DELETED,
        FOLDER_TYPE,
        TAGS,
        NEXT_OP_LOG_ORDER;

        public String field() {
            return name().toLowerCase();
        }
    }
}
