package ru.yandex.intranet.d.dao.folders;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;

import com.fasterxml.jackson.core.type.TypeReference;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.result.ValueReader;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.Value;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.dao.AbstractDaoWithSoftRemove;
import ru.yandex.intranet.d.dao.JsonFieldHelper;
import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.delivery.DeliverableMetaHistoryModel;
import ru.yandex.intranet.d.model.folders.FolderOperationLogModel;
import ru.yandex.intranet.d.model.folders.FolderOperationType;
import ru.yandex.intranet.d.model.folders.OperationPhase;
import ru.yandex.intranet.d.model.folders.TransferMetaHistoryModel;
import ru.yandex.intranet.d.services.folders.history.FolderHistoryContinuationToken;
import ru.yandex.intranet.d.util.ObjectMapperHolder;
import ru.yandex.intranet.d.web.model.SortOrderDto;

import static ru.yandex.intranet.d.dao.folders.FolderOperationLogDao.Fields.ACCOUNTS_QUOTAS_OPERATIONS_ID;
import static ru.yandex.intranet.d.dao.folders.FolderOperationLogDao.Fields.ACTUALLY_APPLIED_PROVISIONS;
import static ru.yandex.intranet.d.dao.folders.FolderOperationLogDao.Fields.AUTHOR_PROVIDER_ID;
import static ru.yandex.intranet.d.dao.folders.FolderOperationLogDao.Fields.AUTHOR_USER_ID;
import static ru.yandex.intranet.d.dao.folders.FolderOperationLogDao.Fields.AUTHOR_USER_UID;
import static ru.yandex.intranet.d.dao.folders.FolderOperationLogDao.Fields.DESTINATION_FOLDER_OPERATIONS_LOG_ID;
import static ru.yandex.intranet.d.dao.folders.FolderOperationLogDao.Fields.FOLDER_ID;
import static ru.yandex.intranet.d.dao.folders.FolderOperationLogDao.Fields.ID;
import static ru.yandex.intranet.d.dao.folders.FolderOperationLogDao.Fields.NEW_BALANCE;
import static ru.yandex.intranet.d.dao.folders.FolderOperationLogDao.Fields.NEW_FOLDER_FIELDS;
import static ru.yandex.intranet.d.dao.folders.FolderOperationLogDao.Fields.NEW_PROVISIONS;
import static ru.yandex.intranet.d.dao.folders.FolderOperationLogDao.Fields.NEW_QUOTAS;
import static ru.yandex.intranet.d.dao.folders.FolderOperationLogDao.Fields.OLD_BALANCE;
import static ru.yandex.intranet.d.dao.folders.FolderOperationLogDao.Fields.OLD_FOLDER_FIELDS;
import static ru.yandex.intranet.d.dao.folders.FolderOperationLogDao.Fields.OLD_PROVISIONS;
import static ru.yandex.intranet.d.dao.folders.FolderOperationLogDao.Fields.OLD_QUOTAS;
import static ru.yandex.intranet.d.dao.folders.FolderOperationLogDao.Fields.OPERATION_DATE_TIME;
import static ru.yandex.intranet.d.dao.folders.FolderOperationLogDao.Fields.OPERATION_PHASE;
import static ru.yandex.intranet.d.dao.folders.FolderOperationLogDao.Fields.OPERATION_TYPE;
import static ru.yandex.intranet.d.dao.folders.FolderOperationLogDao.Fields.PROVIDER_REQUEST_ID;
import static ru.yandex.intranet.d.dao.folders.FolderOperationLogDao.Fields.QUOTAS_DEMANDS_ID;
import static ru.yandex.intranet.d.dao.folders.FolderOperationLogDao.Fields.SOURCE_FOLDER_OPERATIONS_LOG_ID;
import static ru.yandex.intranet.d.dao.folders.FolderOperationLogDao.Fields.TENANT_ID;
import static ru.yandex.intranet.d.datasource.Ydb.nullableUtf8;
import static ru.yandex.intranet.d.datasource.Ydb.timestamp;
import static ru.yandex.intranet.d.datasource.Ydb.utf8;
import static ru.yandex.intranet.d.datasource.Ydb.utf8OrNull;
import static ru.yandex.intranet.d.web.model.SortOrderDto.ASC;

/**
 * FolderOperationLogDao.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 23.10.2020
 */
@Component
public class FolderOperationLogDao
        extends AbstractDaoWithSoftRemove<FolderOperationLogModel, FolderOperationLogModel.Identity> {
    private final QuotasJsonHelper quotasHelper;
    private final FolderHistoryFieldsHelper folderHistoryFieldsHelper;
    private final AccountsHistoryHelper accountsHistoryHelper;
    private final JsonFieldHelper<DeliverableMetaHistoryModel> deliveryMetaHelper;
    private final JsonFieldHelper<TransferMetaHistoryModel> transferMetaHelper;

    public FolderOperationLogDao(
            YdbQuerySource ydbQuerySource,
            QuotasJsonHelper quotasHelper,
            FolderHistoryFieldsHelper folderHistoryFieldsHelper,
            AccountsHistoryHelper accountsHistoryHelper,
            @Qualifier("ydbJsonObjectMapper") ObjectMapperHolder objectMapper) {
        super(ydbQuerySource);
        this.quotasHelper = quotasHelper;
        this.folderHistoryFieldsHelper = folderHistoryFieldsHelper;
        this.accountsHistoryHelper = accountsHistoryHelper;
        this.deliveryMetaHelper = new JsonFieldHelper<>(objectMapper,
                new TypeReference<DeliverableMetaHistoryModel>() { });
        this.transferMetaHelper = new JsonFieldHelper<>(objectMapper,
                new TypeReference<>() { });
    }

    public Mono<List<FolderOperationLogModel>> getFirstPageByFolder(YdbTxSession session, TenantId tenantId,
                                                                    String folderId, SortOrderDto sortOrder,
                                                                    int limit) {
        String query = sortOrder == ASC
                ? ydbQuerySource.getQuery("yql.queries.folders.operation.log.getFirstPageByFolderAsc")
                : ydbQuerySource.getQuery("yql.queries.folders.operation.log.getFirstPageByFolderDesc");
        Params params = Params.of(
                "$folder_id", PrimitiveValue.utf8(folderId),
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$limit", PrimitiveValue.uint64(limit)
                );
        return session.executeDataQueryRetryable(query, params).map(this::toModels);
    }

    public Mono<List<FolderOperationLogModel>> getPageByFolder(
            YdbTxSession session, TenantId tenantId, String folderId,
            @Nullable FolderHistoryContinuationToken token,
            SortOrderDto sortOrder, int limit) {

        if (token == null || token.getDateTime() == null || token.getId() == null) {
            return getFirstPageByFolder(session, tenantId, folderId, sortOrder, limit);
        }

        String query = sortOrder == ASC
                ? ydbQuerySource.getQuery("yql.queries.folders.operation.log.getPageByFolderAsc")
                : ydbQuerySource.getQuery("yql.queries.folders.operation.log.getPageByFolderDesc");
        Params params = Params.of(
                "$folder_id", PrimitiveValue.utf8(folderId),
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$limit", PrimitiveValue.uint64(limit),
                "$from_date_time", PrimitiveValue.timestamp(token.getDateTime()),
                "$from_id", PrimitiveValue.utf8(token.getId())
        );
        return session.executeDataQueryRetryable(query, params).map(this::toModels);
    }

    @Override
    protected WithTenant<FolderOperationLogModel.Identity> getIdentityWithTenant(FolderOperationLogModel model) {
        return new WithTenant<>(model.getTenantId(), model.getIdentity());
    }

    @Override
    protected Params getIdentityParams(FolderOperationLogModel.Identity id) {
        return Params.create()
                .put("$folder_id", utf8(id.getFolderId()))
                .put("$operation_date_time", timestamp(id.getOperationDateTime()))
                .put("$id", utf8(id.getId()));
    }

    @SuppressWarnings("rawtypes")
    @Override
    protected Map<String, Value> prepareFieldValues(FolderOperationLogModel model) {
        HashMap<String, Value> fields = new HashMap<>();

        fields.put(TENANT_ID.field(), utf8(model.getTenantId().getId()));
        fields.put(FOLDER_ID.field(), utf8(model.getFolderId()));
        fields.put(OPERATION_DATE_TIME.field(), timestamp(model.getOperationDateTime()));
        fields.put(ID.field(), utf8(model.getId()));
        fields.put(PROVIDER_REQUEST_ID.field(), nullableUtf8(model.getProviderRequestId().orElse(null)));
        fields.put(OPERATION_TYPE.field(), utf8(model.getOperationType().name()));
        fields.put(AUTHOR_USER_ID.field(), nullableUtf8(model.getAuthorUserId().orElse(null)));
        fields.put(AUTHOR_USER_UID.field(), nullableUtf8(model.getAuthorUserUid().orElse(null)));
        fields.put(AUTHOR_PROVIDER_ID.field(), nullableUtf8(model.getAuthorProviderId().orElse(null)));
        fields.put(SOURCE_FOLDER_OPERATIONS_LOG_ID.field(),
                nullableUtf8(model.getSourceFolderOperationsLogId().orElse(null)));
        fields.put(DESTINATION_FOLDER_OPERATIONS_LOG_ID.field(),
                nullableUtf8(model.getDestinationFolderOperationsLogId().orElse(null)));
        fields.put(OLD_FOLDER_FIELDS.field(),
                folderHistoryFieldsHelper.writeFolderHistoryFields(model.getOldFolderFields().orElse(null)));
        fields.put(OLD_QUOTAS.field(), quotasHelper.writeQuotasByResource(model.getOldQuotas()));
        fields.put(OLD_BALANCE.field(), quotasHelper.writeQuotasByResource(model.getOldBalance()));
        fields.put(OLD_PROVISIONS.field(), quotasHelper.writeQuotasByAccount(model.getOldProvisions()));
        fields.put(Fields.OLD_ACCOUNTS.field(),
                accountsHistoryHelper.writeAccountsHistory(model.getOldAccounts().orElse(null)));
        fields.put(NEW_FOLDER_FIELDS.field(),
                folderHistoryFieldsHelper.writeFolderHistoryFields(model.getNewFolderFields().orElse(null)));
        fields.put(NEW_QUOTAS.field(), quotasHelper.writeQuotasByResource(model.getNewQuotas()));
        fields.put(NEW_BALANCE.field(), quotasHelper.writeQuotasByResource(model.getNewBalance()));
        fields.put(NEW_PROVISIONS.field(), quotasHelper.writeQuotasByAccount(model.getNewProvisions()));
        fields.put(ACTUALLY_APPLIED_PROVISIONS.field(), quotasHelper.writeQuotasByAccountO(
                model.getActuallyAppliedProvisions().orElse(null)));
        fields.put(Fields.NEW_ACCOUNTS.field(),
                accountsHistoryHelper.writeAccountsHistory(model.getNewAccounts().orElse(null)));
        fields.put(ACCOUNTS_QUOTAS_OPERATIONS_ID.field(),
                nullableUtf8(model.getAccountsQuotasOperationsId().orElse(null)));
        fields.put(QUOTAS_DEMANDS_ID.field(), nullableUtf8(model.getQuotasDemandsId().orElse(null)));
        fields.put(OPERATION_PHASE.field(), nullableUtf8(model.getOperationPhase().map(Enum::name).orElse(null)));
        fields.put(Fields.ORDER.field(), PrimitiveValue.int64(model.getOrder()));
        fields.put(Fields.COMMENT_ID.field(), nullableUtf8(model.getCommentId().orElse(null)));
        fields.put(Fields.DELIVERY_META.field(), deliveryMetaHelper
                .writeOptional(model.getDeliveryMeta().orElse(null)));
        fields.put(Fields.TRANSFER_META.field(), transferMetaHelper
                .writeOptional(model.getTransferMeta().orElse(null)));
        return fields;
    }

    @Override
    protected FolderOperationLogModel readOneRow(ResultSetReader reader, Map<String, TenantId> tenantIdCache) {
        return new FolderOperationLogModel.Builder()
                .setTenantId(Tenants.getInstance(utf8(reader.getColumn(TENANT_ID.field()))))
                .setFolderId(utf8(reader.getColumn(FOLDER_ID.field())))
                .setOperationDateTime(timestamp(reader.getColumn(OPERATION_DATE_TIME.field())))
                .setId(reader.getColumn(ID.field()).getUtf8())
                .setProviderRequestId(utf8OrNull(reader.getColumn(PROVIDER_REQUEST_ID.field())))
                .setOperationType(FolderOperationType.valueOf(utf8(reader.getColumn(OPERATION_TYPE.field()))))
                .setAuthorUserId(utf8OrNull(reader.getColumn(AUTHOR_USER_ID.field())))
                .setAuthorUserUid(utf8OrNull(reader.getColumn(AUTHOR_USER_UID.field())))
                .setAuthorProviderId(utf8OrNull(reader.getColumn(AUTHOR_PROVIDER_ID.field())))
                .setSourceFolderOperationsLogId(utf8OrNull(reader.getColumn(SOURCE_FOLDER_OPERATIONS_LOG_ID.field())))
                .setDestinationFolderOperationsLogId(
                        utf8OrNull(reader.getColumn(DESTINATION_FOLDER_OPERATIONS_LOG_ID.field())))
                .setOldFolderFields(folderHistoryFieldsHelper
                        .readFolderHistoryFields(reader.getColumn(OLD_FOLDER_FIELDS.field())).orElse(null))
                .setOldQuotas(quotasHelper.readQuotasByResource(reader.getColumn(OLD_QUOTAS.field())))
                .setOldBalance(quotasHelper.readQuotasByResource(reader.getColumn(OLD_BALANCE.field())))
                .setOldProvisions(quotasHelper.readQuotasByAccount(reader.getColumn(OLD_PROVISIONS.field())))
                .setOldAccounts(accountsHistoryHelper
                        .readAccountsHistory(reader.getColumn(Fields.OLD_ACCOUNTS.field())).orElse(null))
                .setNewFolderFields(folderHistoryFieldsHelper
                        .readFolderHistoryFields(reader.getColumn(NEW_FOLDER_FIELDS.field())).orElse(null))
                .setNewQuotas(quotasHelper.readQuotasByResource(reader.getColumn(NEW_QUOTAS.field())))
                .setNewBalance(quotasHelper.readQuotasByResource(reader.getColumn(NEW_BALANCE.field())))
                .setNewProvisions(quotasHelper.readQuotasByAccount(reader.getColumn(NEW_PROVISIONS.field())))
                .setActuallyAppliedProvisions(quotasHelper
                        .readQuotasByAccountO(reader.getColumn(ACTUALLY_APPLIED_PROVISIONS.field())).orElse(null))
                .setNewAccounts(accountsHistoryHelper
                        .readAccountsHistory(reader.getColumn(Fields.NEW_ACCOUNTS.field())).orElse(null))
                .setAccountsQuotasOperationsId(utf8OrNull(reader.getColumn(ACCOUNTS_QUOTAS_OPERATIONS_ID.field())))
                .setQuotasDemandsId(utf8OrNull(reader.getColumn(QUOTAS_DEMANDS_ID.field())))
                .setOperationPhase(operationPhaseOrNull(reader.getColumn(OPERATION_PHASE.field())))
                .setOrder(reader.getColumn(Fields.ORDER.field()).getInt64())
                .setCommentId(utf8OrNull(reader.getColumn(Fields.COMMENT_ID.field())))
                .setDeliveryMeta(deliveryMetaHelper.read(reader.getColumn(Fields.DELIVERY_META.field())))
                .setTransferMeta(transferMetaHelper.read(reader.getColumn(Fields.TRANSFER_META.field())))
                .build();
    }

    private OperationPhase operationPhaseOrNull(ValueReader value) {
        if (value.isOptionalItemPresent()) {
            return OperationPhase.valueOf(value.getUtf8());
        }
        return null;
    }

    @Override
    protected String queryKeyPrefix() {
        return "yql.queries.folders.operation.log";
    }


    @SuppressWarnings({"unused", "RedundantSuppression"})
    public enum Fields {
        TENANT_ID,
        FOLDER_ID,
        OPERATION_DATE_TIME,
        ID,
        PROVIDER_REQUEST_ID,
        OPERATION_TYPE,
        AUTHOR_USER_ID,
        AUTHOR_USER_UID,
        AUTHOR_PROVIDER_ID,
        SOURCE_FOLDER_OPERATIONS_LOG_ID, // null for source row
        DESTINATION_FOLDER_OPERATIONS_LOG_ID, // null for destination row
        OLD_FOLDER_FIELDS,
        OLD_QUOTAS, // assured quotas map by resources
        OLD_BALANCE, // old values of changed balance by resource
        OLD_PROVISIONS, // provided quotas map by accounts and by resources
        OLD_ACCOUNTS,
        NEW_FOLDER_FIELDS,
        NEW_QUOTAS, // assured quotas map by resources
        NEW_BALANCE, // new values of changed balance by resource
        NEW_PROVISIONS, // provided quotas map by accounts and by resources
        ACTUALLY_APPLIED_PROVISIONS, // provided quotas returned by provider
        NEW_ACCOUNTS,
        ACCOUNTS_QUOTAS_OPERATIONS_ID,
        QUOTAS_DEMANDS_ID,
        OPERATION_PHASE,
        ORDER,
        COMMENT_ID,
        DELIVERY_META,
        TRANSFER_META;

        String field() {
            return name().toLowerCase();
        }
    }

}
