package ru.yandex.intranet.d.dao.folders;

import java.io.UncheckedIOException;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.yandex.ydb.table.result.ValueReader;
import com.yandex.ydb.table.values.OptionalType;
import com.yandex.ydb.table.values.OptionalValue;
import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.PrimitiveValue;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import ru.yandex.intranet.d.model.folders.ProvisionHistoryModel;
import ru.yandex.intranet.d.model.folders.ProvisionsByResource;
import ru.yandex.intranet.d.model.folders.QuotasByAccount;
import ru.yandex.intranet.d.model.folders.QuotasByResource;
import ru.yandex.intranet.d.util.ObjectMapperHolder;

/**
 * Write / read folder quotas JSON.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 26.10.2020
 */
@Component
public class QuotasJsonHelper {
    private final ObjectReader mapToLongReader;
    private final ObjectWriter mapToLongWriter;
    private final ObjectReader mapToMapToLongReader;
    private final ObjectWriter mapToMapToLongWriter;

    public QuotasJsonHelper(
            @Qualifier("ydbJsonObjectMapper") ObjectMapperHolder objectMapper
    ) {
        ObjectMapper mapper = objectMapper.getObjectMapper();
        TypeReference<Map<String, Long>> mapToLongTypeReference = new TypeReference<>() {
        };
        mapToLongReader = mapper.readerFor(mapToLongTypeReference);
        mapToLongWriter = mapper.writerFor(mapToLongTypeReference);

        TypeReference<Map<String, Map<String, ProvisionHistoryModel>>> mapInMapTypeReference = new TypeReference<>() {
        };
        mapToMapToLongReader = mapper.readerFor(mapInMapTypeReference);
        mapToMapToLongWriter = mapper.writerFor(mapInMapTypeReference);
    }


    public PrimitiveValue writeQuotasByResource(QuotasByResource quotas) {
        try {
            String json = mapToLongWriter.writeValueAsString(quotas.asMap());
            return PrimitiveValue.jsonDocument(json);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }

    public QuotasByResource readQuotasByResource(ValueReader reader) {
        String json = reader.getJsonDocument();
        try {
            Map<String, Long> map = mapToLongReader.readValue(json);
            return new QuotasByResource(map);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }

    public PrimitiveValue writeQuotasByAccount(QuotasByAccount quotas) {
        Map<String, Map<String, ProvisionHistoryModel>> map = new HashMap<>(quotas.size());
        quotas.forEach((String accountId, ProvisionsByResource quotasByResource) ->
                map.put(accountId, quotasByResource.asMap()));
        try {
            String json = mapToMapToLongWriter.writeValueAsString(map);
            return PrimitiveValue.jsonDocument(json);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }

    public OptionalValue writeQuotasByAccountO(QuotasByAccount quotas) {
        OptionalType type = OptionalType.of(PrimitiveType.jsonDocument());
        if (quotas != null) {
            return type.newValue(writeQuotasByAccount(quotas));
        } else {
            return type.emptyValue();
        }
    }

    public QuotasByAccount readQuotasByAccount(ValueReader reader) {
        String json = reader.getJsonDocument();
        try {
            Map<String, Map<String, ProvisionHistoryModel>> map = mapToMapToLongReader.readValue(json);
            Map<String, ProvisionsByResource> quotasByAccountId = new HashMap<>(map.size());
            map.forEach((accountId, quotasByResourceId) ->
                    quotasByAccountId.put(accountId, new ProvisionsByResource(quotasByResourceId)));
            return new QuotasByAccount(quotasByAccountId);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }

    public Optional<QuotasByAccount> readQuotasByAccountO(ValueReader reader) {
        if (reader.isOptionalItemPresent()) {
            return Optional.of(readQuotasByAccount(reader));
        } else {
            return Optional.empty();
        }
    }
}
