package ru.yandex.intranet.d.dao.loans

import com.fasterxml.jackson.core.type.TypeReference
import com.yandex.ydb.table.query.Params
import com.yandex.ydb.table.result.ResultSetReader
import com.yandex.ydb.table.values.ListValue
import com.yandex.ydb.table.values.PrimitiveValue
import com.yandex.ydb.table.values.StructValue
import kotlinx.coroutines.reactor.awaitSingle
import kotlinx.coroutines.reactor.awaitSingleOrNull
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.stereotype.Component
import ru.yandex.intranet.d.dao.DaoReader
import ru.yandex.intranet.d.dao.JsonFieldHelper
import ru.yandex.intranet.d.datasource.Ydb
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource
import ru.yandex.intranet.d.datasource.model.YdbTxSession
import ru.yandex.intranet.d.kotlin.LoanId
import ru.yandex.intranet.d.kotlin.ServiceId
import ru.yandex.intranet.d.model.TenantId
import ru.yandex.intranet.d.model.loans.LoanActionSubject
import ru.yandex.intranet.d.model.loans.LoanActionSubjects
import ru.yandex.intranet.d.model.loans.LoanAmounts
import ru.yandex.intranet.d.model.loans.LoanDueDate
import ru.yandex.intranet.d.model.loans.LoanModel
import ru.yandex.intranet.d.model.loans.LoanStatus
import ru.yandex.intranet.d.model.loans.LoanSubject
import ru.yandex.intranet.d.model.loans.LoanType
import ru.yandex.intranet.d.model.loans.LoansPageEntry
import ru.yandex.intranet.d.util.ObjectMapperHolder
import java.time.Instant

/**
 * Loans DAO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
class LoansDao(private val ydbQuerySource: YdbQuerySource,
               @Qualifier("ydbJsonObjectMapper") private val objectMapper: ObjectMapperHolder) {

    private val dueDateFieldHelper: JsonFieldHelper<LoanDueDate> = JsonFieldHelper(objectMapper,
        object : TypeReference<LoanDueDate>() {})
    private val actionSubjectFieldHelper: JsonFieldHelper<LoanActionSubject> = JsonFieldHelper(objectMapper,
        object : TypeReference<LoanActionSubject>() {})
    private val actionSubjectsFieldHelper: JsonFieldHelper<LoanActionSubjects> = JsonFieldHelper(objectMapper,
        object : TypeReference<LoanActionSubjects>() {})
    private val subjectFieldHelper: JsonFieldHelper<LoanSubject> = JsonFieldHelper(objectMapper,
        object : TypeReference<LoanSubject>() {})
    private val amountsFieldHelper: JsonFieldHelper<LoanAmounts> = JsonFieldHelper(objectMapper,
        object : TypeReference<LoanAmounts>() {})

    suspend fun getById(session: YdbTxSession, id: LoanId, tenantId: TenantId): LoanModel? {
        val query = ydbQuerySource.getQuery("yql.queries.loans.getById")
        val params = toIdParams(id, tenantId)
        return DaoReader.toModel(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getByIds(session: YdbTxSession,
                         ids: Collection<LoanId>, tenantId: TenantId): List<LoanModel> {
        if (ids.isEmpty()) {
            return listOf()
        }
        return ids.distinct().chunked(1000).map {
            val query = ydbQuerySource.getQuery("yql.queries.loans.getByIds")
            val params = toIdListParams(it, tenantId)
            DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
        }.flatten()
    }

    suspend fun upsertOneRetryable(session: YdbTxSession, value: LoanModel) {
        val query = ydbQuerySource.getQuery("yql.queries.loans.upsertOne")
        val params = toUpsertOneParams(value)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun upsertManyRetryable(session: YdbTxSession, values: Collection<LoanModel>) {
        if (values.isEmpty()) {
            return
        }
        val query = ydbQuerySource.getQuery("yql.queries.loans.upsertMany")
        val params = toUpsertManyParams(values)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun getByServiceStatusInFirstPage(session: YdbTxSession, tenantId: TenantId, serviceId: ServiceId,
                                              status: LoanStatus, limit: Long): List<LoansPageEntry> {
        val query = ydbQuerySource.getQuery("yql.queries.loans.getInByServiceStatusFirstPage")
        val params = toGetByServiceStatusFirstPageParams(tenantId, serviceId, status, limit)
        return DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toPageEntry)
    }

    suspend fun getByServiceStatusInNextPage(session: YdbTxSession, tenantId: TenantId, serviceId: ServiceId,
                                             status: LoanStatus, fromDueAt: Instant, fromLoanId: LoanId,
                                             limit: Long): List<LoansPageEntry> {
        val query = ydbQuerySource.getQuery("yql.queries.loans.getInByServiceStatusNextPage")
        val params = toGetByServiceStatusNextPageParams(tenantId, serviceId, status, fromDueAt, fromLoanId, limit)
        return DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toPageEntry)
    }

    suspend fun getByServiceStatusOutFirstPage(session: YdbTxSession, tenantId: TenantId, serviceId: ServiceId,
                                               status: LoanStatus, limit: Long): List<LoansPageEntry> {
        val query = ydbQuerySource.getQuery("yql.queries.loans.getOutByServiceStatusFirstPage")
        val params = toGetByServiceStatusFirstPageParams(tenantId, serviceId, status, limit)
        return DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toPageEntry)
    }

    suspend fun getByServiceStatusOutNextPage(session: YdbTxSession, tenantId: TenantId, serviceId: ServiceId,
                                              status: LoanStatus, fromDueAt: Instant, fromLoanId: LoanId,
                                              limit: Long): List<LoansPageEntry> {
        val query = ydbQuerySource.getQuery("yql.queries.loans.getOutByServiceStatusNextPage")
        val params = toGetByServiceStatusNextPageParams(tenantId, serviceId, status, fromDueAt, fromLoanId, limit)
        return DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toPageEntry)
    }

    private fun toPageEntry(reader: ResultSetReader) = LoansPageEntry(
        loan = toModel(reader),
        serviceId = reader.getColumn("service_id").int64,
        dueAtTimestamp = reader.getColumn("due_at_timestamp").timestamp
    )

    private fun toModel(reader: ResultSetReader) = LoanModel(
        tenantId = TenantId(reader.getColumn("tenant_id").utf8),
        id = reader.getColumn("id").utf8,
        status = LoanStatus.valueOf(reader.getColumn("loan_status").utf8),
        type = LoanType.valueOf(reader.getColumn("loan_type").utf8),
        createdAt = reader.getColumn("created_at").timestamp,
        dueAt = dueDateFieldHelper.read(reader.getColumn("due_at"))!!,
        settledAt = Ydb.timestampOrNull(reader.getColumn("settled_at")),
        updatedAt = Ydb.timestampOrNull(reader.getColumn("updated_at")),
        version = reader.getColumn("version").int64,
        requestedBy = actionSubjectFieldHelper.read(reader.getColumn("requested_by"))!!,
        requestApprovedBy = actionSubjectsFieldHelper.read(reader.getColumn("request_approved_by"))!!,
        borrowTransferRequestId = reader.getColumn("borrow_transfer_request_id").utf8,
        borrowedFrom = subjectFieldHelper.read(reader.getColumn("borrowed_from"))!!,
        borrowedTo = subjectFieldHelper.read(reader.getColumn("borrowed_to"))!!,
        payOffFrom = subjectFieldHelper.read(reader.getColumn("pay_off_from"))!!,
        payOffTo = subjectFieldHelper.read(reader.getColumn("pay_off_to"))!!,
        borrowedAmounts = amountsFieldHelper.read(reader.getColumn("borrowed_amounts"))!!,
        payOffAmounts = amountsFieldHelper.read(reader.getColumn("pay_off_amounts"))!!,
        dueAmounts = amountsFieldHelper.read(reader.getColumn("due_amounts"))!!,
        dueAtTimestamp = reader.getColumn("due_at_timestamp").timestamp
    )

    private fun toIdParams(id: LoanId, tenantId: TenantId) = Params.of(
        "\$id", toIdStruct(id, tenantId)
    )

    private fun toIdListParams(ids: Collection<LoanId>, tenantId: TenantId) = Params.of(
        "\$ids", ListValue.of(*ids.map { toIdStruct(it, tenantId) }.toTypedArray())
    )

    private fun toIdStruct(id: LoanId, tenantId: TenantId) = StructValue.of(mapOf(
        "tenant_id" to PrimitiveValue.utf8(tenantId.id),
        "id" to PrimitiveValue.utf8(id),
    ))

    private fun toUpsertOneParams(value: LoanModel) = Params.of(
        "\$value", toUpsertStruct(value)
    )

    private fun toUpsertManyParams(values: Collection<LoanModel>) = Params.of(
        "\$values", ListValue.of(*values.map { toUpsertStruct(it) }.toTypedArray())
    )

    private fun toUpsertStruct(value: LoanModel) = StructValue.of(
        mapOf(
            "tenant_id" to PrimitiveValue.utf8(value.tenantId.id),
            "id" to PrimitiveValue.utf8(value.id),
            "loan_status" to PrimitiveValue.utf8(value.status.name),
            "loan_type" to PrimitiveValue.utf8(value.type.name),
            "created_at" to PrimitiveValue.timestamp(value.createdAt),
            "due_at" to dueDateFieldHelper.write(value.dueAt),
            "settled_at" to Ydb.nullableTimestamp(value.settledAt),
            "updated_at" to Ydb.nullableTimestamp(value.updatedAt),
            "version" to PrimitiveValue.int64(value.version),
            "requested_by" to actionSubjectFieldHelper.write(value.requestedBy),
            "request_approved_by" to actionSubjectsFieldHelper.write(value.requestApprovedBy),
            "borrow_transfer_request_id" to PrimitiveValue.utf8(value.borrowTransferRequestId),
            "borrowed_from" to subjectFieldHelper.write(value.borrowedFrom),
            "borrowed_to" to subjectFieldHelper.write(value.borrowedTo),
            "pay_off_from" to subjectFieldHelper.write(value.payOffFrom),
            "pay_off_to" to subjectFieldHelper.write(value.payOffTo),
            "borrowed_amounts" to amountsFieldHelper.write(value.borrowedAmounts),
            "pay_off_amounts" to amountsFieldHelper.write(value.payOffAmounts),
            "due_amounts" to amountsFieldHelper.write(value.dueAmounts),
            "due_at_timestamp" to PrimitiveValue.timestamp(value.dueAtTimestamp)
        )
    )

    private fun toGetByServiceStatusFirstPageParams(tenantId: TenantId, serviceId: ServiceId,
                                                    status: LoanStatus, limit: Long) = Params.of(
        "\$tenant_id", PrimitiveValue.utf8(tenantId.id),
        "\$service_id", PrimitiveValue.int64(serviceId),
        "\$loan_status", PrimitiveValue.utf8(status.name),
        "\$limit", PrimitiveValue.uint64(limit)
    )

    private fun toGetByServiceStatusNextPageParams(tenantId: TenantId, serviceId: ServiceId,
                                                   status: LoanStatus, fromDueAt: Instant, fromLoanId: LoanId,
                                                   limit: Long) = Params.of(
        "\$tenant_id", PrimitiveValue.utf8(tenantId.id),
        "\$service_id", PrimitiveValue.int64(serviceId),
        "\$loan_status", PrimitiveValue.utf8(status.name),
        "\$from_due_at", PrimitiveValue.timestamp(fromDueAt),
        "\$from_loan_id", PrimitiveValue.utf8(fromLoanId),
        "\$limit", PrimitiveValue.uint64(limit)
    )

}
