package ru.yandex.intranet.d.dao.loans

import com.fasterxml.jackson.core.type.TypeReference
import com.yandex.ydb.table.query.Params
import com.yandex.ydb.table.result.ResultSetReader
import com.yandex.ydb.table.values.ListValue
import com.yandex.ydb.table.values.PrimitiveValue
import com.yandex.ydb.table.values.StructValue
import kotlinx.coroutines.reactor.awaitSingle
import kotlinx.coroutines.reactor.awaitSingleOrNull
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.stereotype.Component
import ru.yandex.intranet.d.dao.DaoReader
import ru.yandex.intranet.d.dao.JsonFieldHelper
import ru.yandex.intranet.d.datasource.Ydb
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource
import ru.yandex.intranet.d.datasource.model.YdbTxSession
import ru.yandex.intranet.d.kotlin.LoanHistoryId
import ru.yandex.intranet.d.kotlin.LoanId
import ru.yandex.intranet.d.model.TenantId
import ru.yandex.intranet.d.model.loans.LoanActionSubject
import ru.yandex.intranet.d.model.loans.LoanActionSubjects
import ru.yandex.intranet.d.model.loans.LoanEventType
import ru.yandex.intranet.d.model.loans.LoansHistoryFields
import ru.yandex.intranet.d.model.loans.LoansHistoryKey
import ru.yandex.intranet.d.model.loans.LoansHistoryModel
import ru.yandex.intranet.d.util.ObjectMapperHolder
import java.time.Instant

/**
 * Loans history DAO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
class LoansHistoryDao(private val ydbQuerySource: YdbQuerySource,
                      @Qualifier("ydbJsonObjectMapper") private val objectMapper: ObjectMapperHolder) {

    private val actionSubjectFieldHelper: JsonFieldHelper<LoanActionSubject> = JsonFieldHelper(objectMapper,
        object : TypeReference<LoanActionSubject>() {})
    private val actionSubjectsFieldHelper: JsonFieldHelper<LoanActionSubjects> = JsonFieldHelper(objectMapper,
        object : TypeReference<LoanActionSubjects>() {})
    private val historyFieldsFieldHelper: JsonFieldHelper<LoansHistoryFields> = JsonFieldHelper(objectMapper,
        object : TypeReference<LoansHistoryFields>() {})

    suspend fun getById(session: YdbTxSession, id: LoansHistoryKey): LoansHistoryModel? {
        val query = ydbQuerySource.getQuery("yql.queries.loansHistory.getById")
        val params = toIdParams(id)
        return DaoReader.toModel(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getByIds(session: YdbTxSession,
                         ids: Collection<LoansHistoryKey>): List<LoansHistoryModel> {
        if (ids.isEmpty()) {
            return listOf()
        }
        return ids.distinct().chunked(1000).map {
            val query = ydbQuerySource.getQuery("yql.queries.loansHistory.getByIds")
            val params = toIdListParams(it)
            DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
        }.flatten()
    }

    suspend fun upsertOneRetryable(session: YdbTxSession, value: LoansHistoryModel) {
        val query = ydbQuerySource.getQuery("yql.queries.loansHistory.upsertOne")
        val params = toUpsertOneParams(value)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun upsertManyRetryable(session: YdbTxSession, values: Collection<LoansHistoryModel>) {
        if (values.isEmpty()) {
            return
        }
        val query = ydbQuerySource.getQuery("yql.queries.loansHistory.upsertMany")
        val params = toUpsertManyParams(values)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun getByLoanFirstPage(session: YdbTxSession, tenantId: TenantId, loanId: LoanId,
                                   limit: Long): List<LoansHistoryModel> {
        val query = ydbQuerySource.getQuery("yql.queries.loansHistory.getByLoanFirstPage")
        val params = toGetByLoanFirstPageParams(tenantId, loanId, limit)
        return DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getByLoanNextPage(session: YdbTxSession, tenantId: TenantId, loanId: LoanId,
                                  fromEventTimestamp: Instant, fromId: LoanHistoryId,
                                  limit: Long): List<LoansHistoryModel> {
        val query = ydbQuerySource.getQuery("yql.queries.loansHistory.getByLoanNextPage")
        val params = toGetByLoanNextPageParams(tenantId, loanId, fromEventTimestamp, fromId, limit)
        return DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    private fun toModel(reader: ResultSetReader) = LoansHistoryModel(
        key = toKey(reader),
        eventAuthor = actionSubjectFieldHelper.read(reader.getColumn("event_author"))!!,
        eventApprovedBy = actionSubjectsFieldHelper.read(reader.getColumn("event_approved_by")),
        eventType = LoanEventType.valueOf(reader.getColumn("event_type").utf8),
        transferRequestId = Ydb.utf8OrNull(reader.getColumn("transfer_request_id")),
        oldFields = historyFieldsFieldHelper.read(reader.getColumn("old_fields")),
        newFields = historyFieldsFieldHelper.read(reader.getColumn("new_fields"))
    )

    private fun toKey(reader: ResultSetReader) = LoansHistoryKey(
        tenantId = TenantId(reader.getColumn("tenant_id").utf8),
        id = reader.getColumn("id").utf8,
        loanId = reader.getColumn("loan_id").utf8,
        eventTimestamp = reader.getColumn("event_timestamp").timestamp
    )

    private fun toIdParams(id: LoansHistoryKey) = Params.of(
        "\$id", toIdStruct(id)
    )

    private fun toIdListParams(ids: Collection<LoansHistoryKey>) = Params.of(
        "\$ids", ListValue.of(*ids.map { toIdStruct(it) }.toTypedArray())
    )

    private fun toIdStruct(id: LoansHistoryKey) = StructValue.of(mapOf(
        "tenant_id" to PrimitiveValue.utf8(id.tenantId.id),
        "id" to PrimitiveValue.utf8(id.id),
        "loan_id" to PrimitiveValue.utf8(id.loanId),
        "event_timestamp" to PrimitiveValue.timestamp(id.eventTimestamp)
    ))

    private fun toUpsertOneParams(value: LoansHistoryModel) = Params.of(
        "\$value", toUpsertStruct(value)
    )

    private fun toUpsertManyParams(values: Collection<LoansHistoryModel>) = Params.of(
        "\$values", ListValue.of(*values.map { toUpsertStruct(it) }.toTypedArray())
    )

    private fun toUpsertStruct(value: LoansHistoryModel) = StructValue.of(
        mapOf(
            "tenant_id" to PrimitiveValue.utf8(value.key.tenantId.id),
            "id" to PrimitiveValue.utf8(value.key.id),
            "loan_id" to PrimitiveValue.utf8(value.key.loanId),
            "event_timestamp" to PrimitiveValue.timestamp(value.key.eventTimestamp),
            "event_author" to actionSubjectFieldHelper.write(value.eventAuthor),
            "event_approved_by" to actionSubjectsFieldHelper.writeOptional(value.eventApprovedBy),
            "event_type" to PrimitiveValue.utf8(value.eventType.name),
            "transfer_request_id" to Ydb.nullableUtf8(value.transferRequestId),
            "old_fields" to historyFieldsFieldHelper.writeOptional(value.oldFields),
            "new_fields" to historyFieldsFieldHelper.writeOptional(value.newFields)
        )
    )

    private fun toGetByLoanFirstPageParams(tenantId: TenantId, loanId: LoanId, limit: Long) = Params.of(
        "\$tenant_id", PrimitiveValue.utf8(tenantId.id),
        "\$loan_id", PrimitiveValue.utf8(loanId),
        "\$limit", PrimitiveValue.uint64(limit)
    )

    private fun toGetByLoanNextPageParams(tenantId: TenantId, loanId: LoanId, fromEventTimestamp: Instant,
                                          fromId: LoanHistoryId, limit: Long) = Params.of(
        "\$tenant_id", PrimitiveValue.utf8(tenantId.id),
        "\$loan_id", PrimitiveValue.utf8(loanId),
        "\$from_event_timestamp", PrimitiveValue.timestamp(fromEventTimestamp),
        "\$from_id", PrimitiveValue.utf8(fromId),
        "\$limit", PrimitiveValue.uint64(limit)
    )

}
