package ru.yandex.intranet.d.dao.loans

import com.fasterxml.jackson.core.type.TypeReference
import com.yandex.ydb.table.query.Params
import com.yandex.ydb.table.result.ResultSetReader
import com.yandex.ydb.table.values.ListValue
import com.yandex.ydb.table.values.PrimitiveValue
import com.yandex.ydb.table.values.StructValue
import kotlinx.coroutines.reactor.awaitSingle
import kotlinx.coroutines.reactor.awaitSingleOrNull
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.stereotype.Component
import ru.yandex.intranet.d.dao.DaoReader
import ru.yandex.intranet.d.dao.JsonFieldHelper
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource
import ru.yandex.intranet.d.datasource.model.YdbTxSession
import ru.yandex.intranet.d.kotlin.LoanId
import ru.yandex.intranet.d.model.TenantId
import ru.yandex.intranet.d.model.loans.LoanDueDate
import ru.yandex.intranet.d.model.loans.LoanNotifications
import ru.yandex.intranet.d.model.loans.PendingLoanKey
import ru.yandex.intranet.d.model.loans.PendingLoanModel
import ru.yandex.intranet.d.util.ObjectMapperHolder
import java.time.Instant

/**
 * Pending loans DAO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
class PendingLoansDao(private val ydbQuerySource: YdbQuerySource,
                      @Qualifier("ydbJsonObjectMapper") private val objectMapper: ObjectMapperHolder) {

    private val dueDateFieldHelper: JsonFieldHelper<LoanDueDate> = JsonFieldHelper(objectMapper,
        object : TypeReference<LoanDueDate>() {})
    private val notificationsFieldHelper: JsonFieldHelper<LoanNotifications> = JsonFieldHelper(objectMapper,
        object : TypeReference<LoanNotifications>() {})

    suspend fun getById(session: YdbTxSession, id: PendingLoanKey): PendingLoanModel? {
        val query = ydbQuerySource.getQuery("yql.queries.pendingLoans.getById")
        val params = toIdParams(id)
        return DaoReader.toModel(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getByIds(session: YdbTxSession,
                         ids: Collection<PendingLoanKey>): List<PendingLoanModel> {
        if (ids.isEmpty()) {
            return listOf()
        }
        return ids.distinct().chunked(1000).map {
            val query = ydbQuerySource.getQuery("yql.queries.pendingLoans.getByIds")
            val params = toIdListParams(it)
            DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
        }.flatten()
    }

    suspend fun upsertOneRetryable(session: YdbTxSession, value: PendingLoanModel) {
        val query = ydbQuerySource.getQuery("yql.queries.pendingLoans.upsertOne")
        val params = toUpsertOneParams(value)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun upsertManyRetryable(session: YdbTxSession, values: Collection<PendingLoanModel>) {
        if (values.isEmpty()) {
            return
        }
        val query = ydbQuerySource.getQuery("yql.queries.pendingLoans.upsertMany")
        val params = toUpsertManyParams(values)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun deleteOneRetryable(session: YdbTxSession, id: PendingLoanKey) {
        val query = ydbQuerySource.getQuery("yql.queries.pendingLoans.deleteById")
        val params = toIdParams(id)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun deleteManyRetryable(session: YdbTxSession, ids: Collection<PendingLoanKey>) {
        if (ids.isEmpty()) {
            return
        }
        val query = ydbQuerySource.getQuery("yql.queries.pendingLoans.deleteByIds")
        val params = toIdListParams(ids)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun getOverdueFirstPage(session: YdbTxSession, tenantId: TenantId, upToTimestamp: Instant,
                                    limit: Long): List<PendingLoanModel> {
        val query = ydbQuerySource.getQuery("yql.queries.pendingLoans.getOverdueFirstPage")
        val params = toGetOverdueFirstPageParams(tenantId, upToTimestamp, limit)
        return DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getOverdueNextPage(session: YdbTxSession, tenantId: TenantId, fromDueAtTimestamp: Instant,
                                   fromLoanId: LoanId, limit: Long): List<PendingLoanModel> {
        val query = ydbQuerySource.getQuery("yql.queries.pendingLoans.getOverdueNextPage")
        val params = toGetOverdueNextPageParams(tenantId, fromDueAtTimestamp, fromLoanId, limit)
        return DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    private fun toModel(reader: ResultSetReader) = PendingLoanModel(
        key = toKey(reader),
        dueAt = dueDateFieldHelper.read(reader.getColumn("due_at"))!!,
        notifications = notificationsFieldHelper.read(reader.getColumn("notifications"))
    )

    private fun toKey(reader: ResultSetReader) = PendingLoanKey(
        tenantId = TenantId(reader.getColumn("tenant_id").utf8),
        dueAtTimestamp = reader.getColumn("due_at_timestamp").timestamp,
        loanId = reader.getColumn("loan_id").utf8
    )

    private fun toIdParams(id: PendingLoanKey) = Params.of(
        "\$id", toIdStruct(id)
    )

    private fun toIdListParams(ids: Collection<PendingLoanKey>) = Params.of(
        "\$ids", ListValue.of(*ids.map { toIdStruct(it) }.toTypedArray())
    )

    private fun toIdStruct(id: PendingLoanKey) = StructValue.of(mapOf(
        "tenant_id" to PrimitiveValue.utf8(id.tenantId.id),
        "due_at_timestamp" to PrimitiveValue.timestamp(id.dueAtTimestamp),
        "loan_id" to PrimitiveValue.utf8(id.loanId)
    ))

    private fun toUpsertOneParams(value: PendingLoanModel) = Params.of(
        "\$value", toUpsertStruct(value)
    )

    private fun toUpsertManyParams(values: Collection<PendingLoanModel>) = Params.of(
        "\$values", ListValue.of(*values.map { toUpsertStruct(it) }.toTypedArray())
    )

    private fun toUpsertStruct(value: PendingLoanModel) = StructValue.of(
        mapOf(
            "tenant_id" to PrimitiveValue.utf8(value.key.tenantId.id),
            "due_at_timestamp" to PrimitiveValue.timestamp(value.key.dueAtTimestamp),
            "loan_id" to PrimitiveValue.utf8(value.key.loanId),
            "due_at" to dueDateFieldHelper.write(value.dueAt),
            "notifications" to notificationsFieldHelper.writeOptional(value.notifications)
        )
    )

    private fun toGetOverdueFirstPageParams(tenantId: TenantId, upToTimestamp: Instant, limit: Long) = Params.of(
        "\$tenant_id", PrimitiveValue.utf8(tenantId.id),
        "\$up_to_timestamp", PrimitiveValue.timestamp(upToTimestamp),
        "\$limit", PrimitiveValue.uint64(limit)
    )

    private fun toGetOverdueNextPageParams(tenantId: TenantId, fromDueAtTimestamp: Instant,
                                           fromLoanId: LoanId, limit: Long) = Params.of(
        "\$tenant_id", PrimitiveValue.utf8(tenantId.id),
        "\$from_due_at_timestamp", PrimitiveValue.timestamp(fromDueAtTimestamp),
        "\$from_loan_id", PrimitiveValue.utf8(fromLoanId),
        "\$limit", PrimitiveValue.uint64(limit)
    )

}
