package ru.yandex.intranet.d.dao.loans

import com.yandex.ydb.table.query.Params
import com.yandex.ydb.table.result.ResultSetReader
import com.yandex.ydb.table.values.ListValue
import com.yandex.ydb.table.values.PrimitiveValue
import com.yandex.ydb.table.values.StructValue
import kotlinx.coroutines.reactor.awaitSingle
import kotlinx.coroutines.reactor.awaitSingleOrNull
import org.springframework.stereotype.Component
import ru.yandex.intranet.d.dao.DaoPagination
import ru.yandex.intranet.d.dao.DaoReader
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource
import ru.yandex.intranet.d.datasource.model.YdbTxSession
import ru.yandex.intranet.d.kotlin.ProviderId
import ru.yandex.intranet.d.kotlin.ResourceId
import ru.yandex.intranet.d.kotlin.ServiceId
import ru.yandex.intranet.d.model.TenantId
import ru.yandex.intranet.d.model.loans.ServiceLoanBalanceKey
import ru.yandex.intranet.d.model.loans.ServiceLoanBalanceModel
import java.math.BigInteger

/**
 * Service loans balance DAO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
class ServiceLoansBalanceDao(private val ydbQuerySource: YdbQuerySource) {

    suspend fun getById(session: YdbTxSession, id: ServiceLoanBalanceKey): ServiceLoanBalanceModel? {
        val query = ydbQuerySource.getQuery("yql.queries.serviceLoansBalance.getById")
        val params = toIdParams(id)
        return DaoReader.toModel(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getByIds(session: YdbTxSession,
                         ids: Collection<ServiceLoanBalanceKey>): List<ServiceLoanBalanceModel> {
        if (ids.isEmpty()) {
            return listOf()
        }
        return ids.distinct().chunked(1000).map {
            val query = ydbQuerySource.getQuery("yql.queries.serviceLoansBalance.getByIds")
            val params = toIdListParams(it)
            DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
        }.flatten()
    }

    suspend fun upsertOneRetryable(session: YdbTxSession, value: ServiceLoanBalanceModel) {
        val query = ydbQuerySource.getQuery("yql.queries.serviceLoansBalance.upsertOne")
        val params = toUpsertOneParams(value)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun upsertManyRetryable(session: YdbTxSession, values: Collection<ServiceLoanBalanceModel>) {
        if (values.isEmpty()) {
            return
        }
        val query = ydbQuerySource.getQuery("yql.queries.serviceLoansBalance.upsertMany")
        val params = toUpsertManyParams(values)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun getByServiceFirstPage(session: YdbTxSession, tenantId: TenantId, serviceId: ServiceId,
                                      limit: Long): List<ServiceLoanBalanceModel> {
        val query = ydbQuerySource.getQuery("yql.queries.serviceLoansBalance.getByServiceFirstPage")
        val params = toGetByServiceFirstPageParams(tenantId, serviceId, limit)
        return DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getByServiceNextPage(session: YdbTxSession, tenantId: TenantId, serviceId: ServiceId,
                                     fromProviderId: ProviderId, fromResourceId: ResourceId,
                                     limit: Long): List<ServiceLoanBalanceModel> {
        val query = ydbQuerySource.getQuery("yql.queries.serviceLoansBalance.getByServiceNextPage")
        val params = toGetByServiceNextPageParams(tenantId, serviceId, fromProviderId, fromResourceId, limit)
        return DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getAllByService(session: YdbTxSession, tenantId: TenantId,
                                serviceId: ServiceId): List<ServiceLoanBalanceModel> {
        return DaoPagination.getAllPages(session,
            { ydbQuerySource.getQuery("yql.queries.serviceLoansBalance.getByServiceFirstPage") },
            { ydbQuerySource.getQuery("yql.queries.serviceLoansBalance.getByServiceNextPage") },
            { limit -> toGetByServiceFirstPageParams(tenantId, serviceId, limit) },
            { limit, lastBalance -> toGetByServiceNextPageParams(tenantId, serviceId,
                lastBalance.key.providerId, lastBalance.key.resourceId, limit) },
            this::toModel
        )
    }

    private fun toModel(reader: ResultSetReader) = ServiceLoanBalanceModel(
        key = toKey(reader),
        amountIn = BigInteger(reader.getColumn("amount_in").utf8),
        amountOut = BigInteger(reader.getColumn("amount_out").utf8)
    )

    private fun toKey(reader: ResultSetReader) = ServiceLoanBalanceKey(
        tenantId = TenantId(reader.getColumn("tenant_id").utf8),
        serviceId = reader.getColumn("service_id").int64,
        providerId = reader.getColumn("provider_id").utf8,
        resourceId = reader.getColumn("resource_id").utf8
    )

    private fun toIdParams(id: ServiceLoanBalanceKey) = Params.of(
        "\$id", toIdStruct(id)
    )

    private fun toIdListParams(ids: Collection<ServiceLoanBalanceKey>) = Params.of(
        "\$ids", ListValue.of(*ids.map { toIdStruct(it) }.toTypedArray())
    )

    private fun toIdStruct(id: ServiceLoanBalanceKey) = StructValue.of(mapOf(
        "tenant_id" to PrimitiveValue.utf8(id.tenantId.id),
        "service_id" to PrimitiveValue.int64(id.serviceId),
        "provider_id" to PrimitiveValue.utf8(id.providerId),
        "resource_id" to PrimitiveValue.utf8(id.resourceId)
    ))

    private fun toUpsertOneParams(value: ServiceLoanBalanceModel) = Params.of(
        "\$value", toUpsertStruct(value)
    )

    private fun toUpsertManyParams(values: Collection<ServiceLoanBalanceModel>) = Params.of(
        "\$values", ListValue.of(*values.map { toUpsertStruct(it) }.toTypedArray())
    )

    private fun toUpsertStruct(value: ServiceLoanBalanceModel) = StructValue.of(
        mapOf(
            "tenant_id" to PrimitiveValue.utf8(value.key.tenantId.id),
            "service_id" to PrimitiveValue.int64(value.key.serviceId),
            "provider_id" to PrimitiveValue.utf8(value.key.providerId),
            "resource_id" to PrimitiveValue.utf8(value.key.resourceId),
            "amount_in" to PrimitiveValue.utf8(value.amountIn.toString()),
            "amount_out" to PrimitiveValue.utf8(value.amountOut.toString())
        )
    )

    private fun toGetByServiceFirstPageParams(tenantId: TenantId, serviceId: ServiceId, limit: Long) = Params.of(
        "\$tenant_id", PrimitiveValue.utf8(tenantId.id),
        "\$service_id", PrimitiveValue.int64(serviceId),
        "\$limit", PrimitiveValue.uint64(limit)
    )

    private fun toGetByServiceNextPageParams(tenantId: TenantId, serviceId: ServiceId, fromProviderId: ProviderId,
                                             fromResourceId: ResourceId, limit: Long) = Params.of(
        "\$tenant_id", PrimitiveValue.utf8(tenantId.id),
        "\$service_id", PrimitiveValue.int64(serviceId),
        "\$from_provider_id", PrimitiveValue.utf8(fromProviderId),
        "\$from_resource_id", PrimitiveValue.utf8(fromResourceId),
        "\$limit", PrimitiveValue.uint64(limit)
    )

}
