package ru.yandex.intranet.d.dao.loans

import com.yandex.ydb.table.query.Params
import com.yandex.ydb.table.result.ResultSetReader
import com.yandex.ydb.table.values.ListValue
import com.yandex.ydb.table.values.PrimitiveValue
import com.yandex.ydb.table.values.StructValue
import kotlinx.coroutines.reactor.awaitSingle
import kotlinx.coroutines.reactor.awaitSingleOrNull
import org.springframework.stereotype.Component
import reactor.core.publisher.Mono
import ru.yandex.intranet.d.dao.DaoReader
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource
import ru.yandex.intranet.d.datasource.model.YdbTxSession
import ru.yandex.intranet.d.kotlin.LoanId
import ru.yandex.intranet.d.kotlin.ServiceId
import ru.yandex.intranet.d.kotlin.mono
import ru.yandex.intranet.d.model.TenantId
import ru.yandex.intranet.d.model.loans.LoanStatus
import ru.yandex.intranet.d.model.loans.ServiceLoanInModel
import java.time.Instant

/**
 * Service loans incoming DAO. Service will receive payments on these loans.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
class ServiceLoansInDao(private val ydbQuerySource: YdbQuerySource) {

    suspend fun getById(session: YdbTxSession, id: ServiceLoanInModel): ServiceLoanInModel? {
        val query = ydbQuerySource.getQuery("yql.queries.serviceLoansIn.getById")
        val params = toIdParams(id)
        return DaoReader.toModel(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getByIds(session: YdbTxSession,
                         ids: Collection<ServiceLoanInModel>): List<ServiceLoanInModel> {
        if (ids.isEmpty()) {
            return listOf()
        }
        return ids.distinct().chunked(1000).map {
            val query = ydbQuerySource.getQuery("yql.queries.serviceLoansIn.getByIds")
            val params = toIdListParams(it)
            DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
        }.flatten()
    }

    suspend fun getByServiceOrderByDueFirstPage(
        session: YdbTxSession, tenantId: TenantId, serviceId: ServiceId, limit: Int
    ): List<ServiceLoanInModel> {
        val params = toGetByServiceParamsFirstPage(tenantId, serviceId, limit)
        val query = ydbQuerySource.getQuery("yql.queries.serviceLoansIn.getByServiceOrderByDueFirstPage")
        return DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getByServiceOrderByDueNextPage(
        session: YdbTxSession, tenantId: TenantId, serviceId: ServiceId,
        fromStatus: LoanStatus, fromDueAt: Instant, fromLoanId: LoanId, limit: Int
    ): List<ServiceLoanInModel> {
        val params = toGetByServiceParamsNextPage(tenantId, serviceId, fromStatus, fromDueAt, fromLoanId, limit)
        val query = ydbQuerySource.getQuery("yql.queries.serviceLoansIn.getByServiceOrderByDueNextPage")
        return DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getByServiceStatusOrderByDueFirstPage(
        session: YdbTxSession, tenantId: TenantId, serviceId: ServiceId, status: LoanStatus, limit: Int
    ): List<ServiceLoanInModel> {
        val params = toGetByServiceStatusParamsFirstPage(tenantId, serviceId, status, limit)
        val query = ydbQuerySource.getQuery("yql.queries.serviceLoansIn.getByServiceStatusOrderByDueFirstPage")
        return DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getByServiceStatusOrderByDueNextPage(
        session: YdbTxSession, tenantId: TenantId, serviceId: ServiceId,
        status: LoanStatus, fromDueAt: Instant, fromLoanId: LoanId, limit: Int
    ): List<ServiceLoanInModel> {
        val params = toGetByServiceStatusParamsNextPage(tenantId, serviceId, status, fromDueAt, fromLoanId, limit)
        val query = ydbQuerySource.getQuery("yql.queries.serviceLoansIn.getByServiceStatusOrderByDueNextPage")
        return DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    fun filterServiceIdsByLoanStatusMono(
        session: YdbTxSession, tenantId: TenantId, serviceIds: Collection<ServiceId>, status: LoanStatus
    ): Mono<Set<ServiceId>> = mono {
        filterServiceIdsByLoanStatus(session, tenantId, serviceIds, status)
    }

    suspend fun filterServiceIdsByLoanStatus(
        session: YdbTxSession, tenantId: TenantId, serviceIds: Collection<ServiceId>, status: LoanStatus
    ): Set<ServiceId> = serviceIds
        .chunked(1000)
        .map {
            val params = toGetServiceIdsByStatus(tenantId, serviceIds, status)
            val query = ydbQuerySource.getQuery("yql.queries.serviceLoansIn.filterServiceIdsByLoanStatus")
            DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toServiceId)
        }.flatten()
        .toSet()

    suspend fun upsertOneRetryable(session: YdbTxSession, value: ServiceLoanInModel) {
        val query = ydbQuerySource.getQuery("yql.queries.serviceLoansIn.upsertOne")
        val params = toUpsertOneParams(value)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun upsertManyRetryable(session: YdbTxSession, values: Collection<ServiceLoanInModel>) {
        if (values.isEmpty()) {
            return
        }
        val query = ydbQuerySource.getQuery("yql.queries.serviceLoansIn.upsertMany")
        val params = toUpsertManyParams(values)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun deleteById(session: YdbTxSession, id: ServiceLoanInModel) {
        val query = ydbQuerySource.getQuery("yql.queries.serviceLoansIn.deleteById")
        val params = toIdParams(id)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun deleteByIds(session: YdbTxSession, ids: Collection<ServiceLoanInModel>) {
        if (ids.isEmpty()) {
            return
        }
        val query = ydbQuerySource.getQuery("yql.queries.serviceLoansIn.deleteByIds")
        val params = toIdListParams(ids)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    private fun toModel(reader: ResultSetReader) = ServiceLoanInModel(
        tenantId = TenantId(reader.getColumn("tenant_id").utf8),
        serviceId = reader.getColumn("service_id").int64,
        status = LoanStatus.valueOf(reader.getColumn("loan_status").utf8),
        dueAt = reader.getColumn("due_at").timestamp,
        loanId = reader.getColumn("loan_id").utf8
    )

    private fun toServiceId(reader: ResultSetReader) = reader.getColumn("service_id").int64

    private fun toIdParams(id: ServiceLoanInModel) = Params.of(
        "\$id", toIdStruct(id)
    )

    private fun toIdListParams(ids: Collection<ServiceLoanInModel>) = Params.of(
        "\$ids", ListValue.of(*ids.map { toIdStruct(it) }.toTypedArray())
    )

    private fun toIdStruct(id: ServiceLoanInModel) = StructValue.of(mapOf(
        "tenant_id" to PrimitiveValue.utf8(id.tenantId.id),
        "service_id" to PrimitiveValue.int64(id.serviceId),
        "loan_status" to PrimitiveValue.utf8(id.status.name),
        "due_at" to PrimitiveValue.timestamp(id.dueAt),
        "loan_id" to PrimitiveValue.utf8(id.loanId)
    ))

    private fun toUpsertOneParams(value: ServiceLoanInModel) = Params.of(
        "\$value", toUpsertStruct(value)
    )

    private fun toUpsertManyParams(values: Collection<ServiceLoanInModel>) = Params.of(
        "\$values", ListValue.of(*values.map { toUpsertStruct(it) }.toTypedArray())
    )

    private fun toUpsertStruct(value: ServiceLoanInModel) = StructValue.of(
        mapOf(
            "tenant_id" to PrimitiveValue.utf8(value.tenantId.id),
            "service_id" to PrimitiveValue.int64(value.serviceId),
            "loan_status" to PrimitiveValue.utf8(value.status.name),
            "due_at" to PrimitiveValue.timestamp(value.dueAt),
            "loan_id" to PrimitiveValue.utf8(value.loanId)
        )
    )

    private fun toGetByServiceParamsFirstPage(
        tenantId: TenantId, serviceId: ServiceId, limit: Int
    ) = Params.of(
        "\$tenant_id", PrimitiveValue.utf8(tenantId.id),
        "\$service_id", PrimitiveValue.int64(serviceId),
        "\$limit", PrimitiveValue.uint64(limit.toLong())
    )

    private fun toGetByServiceParamsNextPage(
        tenantId: TenantId, serviceId: ServiceId, fromStatus: LoanStatus,
        fromDueAt: Instant, fromLoanId: LoanId, limit: Int
    ) = Params.of(
        "\$tenant_id", PrimitiveValue.utf8(tenantId.id),
        "\$service_id", PrimitiveValue.int64(serviceId),
        "\$limit", PrimitiveValue.uint64(limit.toLong()),
        "\$from_loan_status", PrimitiveValue.utf8(fromStatus.name),
        "\$from_due_at", PrimitiveValue.timestamp(fromDueAt),
        "\$from_loan_id", PrimitiveValue.utf8(fromLoanId)

    )

    private fun toGetByServiceStatusParamsFirstPage(
        tenantId: TenantId, serviceId: ServiceId, status: LoanStatus, limit: Int
    ) = Params.of(
        "\$tenant_id", PrimitiveValue.utf8(tenantId.id),
        "\$service_id", PrimitiveValue.int64(serviceId),
        "\$loan_status", PrimitiveValue.utf8(status.name),
        "\$limit", PrimitiveValue.uint64(limit.toLong())
    )

    private fun toGetByServiceStatusParamsNextPage(
        tenantId: TenantId, serviceId: ServiceId, status: LoanStatus, fromDueAt: Instant, fromLoanId: LoanId, limit: Int
    ) = Params.of(
        "\$tenant_id", PrimitiveValue.utf8(tenantId.id),
        "\$service_id", PrimitiveValue.int64(serviceId),
        "\$loan_status", PrimitiveValue.utf8(status.name),
        "\$limit", PrimitiveValue.uint64(limit.toLong()),
        "\$from_due_at", PrimitiveValue.timestamp(fromDueAt),
        "\$from_loan_id", PrimitiveValue.utf8(fromLoanId)
    )

    private fun toGetServiceIdsByStatus(
        tenantId: TenantId, serviceIds: Collection<ServiceId>, status: LoanStatus
    ) = Params.of(
        "\$tenant_id", PrimitiveValue.utf8(tenantId.id),
        "\$service_ids", ListValue.of(*serviceIds.map { PrimitiveValue.int64(it) }.toTypedArray()),
        "\$loan_status", PrimitiveValue.utf8(status.name)
    )
}
