package ru.yandex.intranet.d.dao.loans

import com.yandex.ydb.table.query.Params
import com.yandex.ydb.table.result.ResultSetReader
import com.yandex.ydb.table.values.ListValue
import com.yandex.ydb.table.values.PrimitiveValue
import com.yandex.ydb.table.values.StructValue
import kotlinx.coroutines.reactor.awaitSingle
import kotlinx.coroutines.reactor.awaitSingleOrNull
import org.springframework.stereotype.Component
import reactor.core.publisher.Mono
import ru.yandex.intranet.d.dao.DaoReader
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource
import ru.yandex.intranet.d.datasource.model.YdbTxSession
import ru.yandex.intranet.d.kotlin.LoanId
import ru.yandex.intranet.d.kotlin.ServiceId
import ru.yandex.intranet.d.kotlin.mono
import ru.yandex.intranet.d.model.TenantId
import ru.yandex.intranet.d.model.loans.LoanStatus
import ru.yandex.intranet.d.model.loans.ServiceLoanOutModel
import java.time.Instant

/**
 * Service loans outgoing DAO. Service will send payments on these loans.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
class ServiceLoansOutDao(private val ydbQuerySource: YdbQuerySource) {

    suspend fun getById(session: YdbTxSession, id: ServiceLoanOutModel): ServiceLoanOutModel? {
        val query = ydbQuerySource.getQuery("yql.queries.serviceLoansOut.getById")
        val params = toIdParams(id)
        return DaoReader.toModel(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getByIds(session: YdbTxSession,
                         ids: Collection<ServiceLoanOutModel>): List<ServiceLoanOutModel> {
        if (ids.isEmpty()) {
            return listOf()
        }
        return ids.distinct().chunked(1000).map {
            val query = ydbQuerySource.getQuery("yql.queries.serviceLoansOut.getByIds")
            val params = toIdListParams(it)
            DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
        }.flatten()
    }

    suspend fun getByServiceOrderByDueFirstPage(
        session: YdbTxSession, tenantId: TenantId, serviceId: ServiceId, limit: Int
    ): List<ServiceLoanOutModel> {
        val params = toGetByServiceParamsFirstPage(tenantId, serviceId, limit)
        val query = ydbQuerySource.getQuery("yql.queries.serviceLoansOut.getByServiceOrderByDueFirstPage")
        return DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getByServiceOrderByDueNextPage(
        session: YdbTxSession, tenantId: TenantId, serviceId: ServiceId,
        fromStatus: LoanStatus, fromDueAt: Instant, fromLoanId: LoanId, limit: Int
    ): List<ServiceLoanOutModel> {
        val params = toGetByServiceParamsNextPage(tenantId, serviceId, fromStatus, fromDueAt, fromLoanId, limit)
        val query = ydbQuerySource.getQuery("yql.queries.serviceLoansOut.getByServiceOrderByDueNextPage")
        return DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getByServiceStatusOrderByDueFirstPage(
        session: YdbTxSession, tenantId: TenantId, serviceId: ServiceId, status: LoanStatus, limit: Int
    ): List<ServiceLoanOutModel> {
        val params = toGetByServiceStatusParamsFirstPage(tenantId, serviceId, status, limit)
        val query = ydbQuerySource.getQuery("yql.queries.serviceLoansOut.getByServiceStatusOrderByDueFirstPage")
        return DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getByServiceStatusOrderByDueNextPage(
        session: YdbTxSession, tenantId: TenantId, serviceId: ServiceId,
        status: LoanStatus, fromDueAt: Instant, fromLoanId: LoanId, limit: Int
    ): List<ServiceLoanOutModel> {
        val params = toGetByServiceStatusParamsNextPage(tenantId, serviceId, status, fromDueAt, fromLoanId, limit)
        val query = ydbQuerySource.getQuery("yql.queries.serviceLoansOut.getByServiceStatusOrderByDueNextPage")
        return DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    fun filterServiceIdsByLoanStatusMono(
        session: YdbTxSession, tenantId: TenantId, serviceIds: Collection<ServiceId>, status: LoanStatus
    ): Mono<Set<ServiceId>> = mono {
        filterServiceIdsByLoanStatus(session, tenantId, serviceIds, status)
    }

    suspend fun filterServiceIdsByLoanStatus(
        session: YdbTxSession, tenantId: TenantId, serviceIds: Collection<ServiceId>, status: LoanStatus
    ): Set<ServiceId> = serviceIds
        .chunked(1000)
        .map {
            val params = toGetServiceIdsByStatus(tenantId, serviceIds, status)
            val query = ydbQuerySource.getQuery("yql.queries.serviceLoansOut.filterServiceIdsByLoanStatus")
            DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toServiceId)
        }.flatten()
        .toSet()

    suspend fun upsertOneRetryable(session: YdbTxSession, value: ServiceLoanOutModel) {
        val query = ydbQuerySource.getQuery("yql.queries.serviceLoansOut.upsertOne")
        val params = toUpsertOneParams(value)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun upsertManyRetryable(session: YdbTxSession, values: Collection<ServiceLoanOutModel>) {
        if (values.isEmpty()) {
            return
        }
        val query = ydbQuerySource.getQuery("yql.queries.serviceLoansOut.upsertMany")
        val params = toUpsertManyParams(values)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun deleteById(session: YdbTxSession, id: ServiceLoanOutModel) {
        val query = ydbQuerySource.getQuery("yql.queries.serviceLoansOut.deleteById")
        val params = toIdParams(id)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun deleteByIds(session: YdbTxSession, ids: Collection<ServiceLoanOutModel>) {
        if (ids.isEmpty()) {
            return
        }
        val query = ydbQuerySource.getQuery("yql.queries.serviceLoansOut.deleteByIds")
        val params = toIdListParams(ids)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    private fun toModel(reader: ResultSetReader) = ServiceLoanOutModel(
        tenantId = TenantId(reader.getColumn("tenant_id").utf8),
        serviceId = reader.getColumn("service_id").int64,
        status = LoanStatus.valueOf(reader.getColumn("loan_status").utf8),
        dueAt = reader.getColumn("due_at").timestamp,
        loanId = reader.getColumn("loan_id").utf8
    )

    private fun toServiceId(reader: ResultSetReader) = reader.getColumn("service_id").int64

    private fun toIdParams(id: ServiceLoanOutModel) = Params.of(
        "\$id", toIdStruct(id)
    )

    private fun toIdListParams(ids: Collection<ServiceLoanOutModel>) = Params.of(
        "\$ids", ListValue.of(*ids.map { toIdStruct(it) }.toTypedArray())
    )

    private fun toIdStruct(id: ServiceLoanOutModel) = StructValue.of(mapOf(
        "tenant_id" to PrimitiveValue.utf8(id.tenantId.id),
        "service_id" to PrimitiveValue.int64(id.serviceId),
        "loan_status" to PrimitiveValue.utf8(id.status.name),
        "due_at" to PrimitiveValue.timestamp(id.dueAt),
        "loan_id" to PrimitiveValue.utf8(id.loanId)
    ))

    private fun toUpsertOneParams(value: ServiceLoanOutModel) = Params.of(
        "\$value", toUpsertStruct(value)
    )

    private fun toUpsertManyParams(values: Collection<ServiceLoanOutModel>) = Params.of(
        "\$values", ListValue.of(*values.map { toUpsertStruct(it) }.toTypedArray())
    )

    private fun toUpsertStruct(value: ServiceLoanOutModel) = StructValue.of(
        mapOf(
            "tenant_id" to PrimitiveValue.utf8(value.tenantId.id),
            "service_id" to PrimitiveValue.int64(value.serviceId),
            "loan_status" to PrimitiveValue.utf8(value.status.name),
            "due_at" to PrimitiveValue.timestamp(value.dueAt),
            "loan_id" to PrimitiveValue.utf8(value.loanId)
        )
    )

    private fun toGetByServiceParamsFirstPage(
        tenantId: TenantId, serviceId: ServiceId, limit: Int
    ) = Params.of(
        "\$tenant_id", PrimitiveValue.utf8(tenantId.id),
        "\$service_id", PrimitiveValue.int64(serviceId),
        "\$limit", PrimitiveValue.uint64(limit.toLong())
    )

    private fun toGetByServiceParamsNextPage(
        tenantId: TenantId, serviceId: ServiceId, fromStatus: LoanStatus, fromDueAt: Instant, fromLoanId: LoanId, limit: Int
    ) = Params.of(
        "\$tenant_id", PrimitiveValue.utf8(tenantId.id),
        "\$service_id", PrimitiveValue.int64(serviceId),
        "\$limit", PrimitiveValue.uint64(limit.toLong()),
        "\$from_loan_status", PrimitiveValue.utf8(fromStatus.name),
        "\$from_due_at", PrimitiveValue.timestamp(fromDueAt),
        "\$from_loan_id", PrimitiveValue.utf8(fromLoanId)

    )

    private fun toGetByServiceStatusParamsFirstPage(
        tenantId: TenantId, serviceId: ServiceId, status: LoanStatus, limit: Int
    ) = Params.of(
        "\$tenant_id", PrimitiveValue.utf8(tenantId.id),
        "\$service_id", PrimitiveValue.int64(serviceId),
        "\$loan_status", PrimitiveValue.utf8(status.name),
        "\$limit", PrimitiveValue.uint64(limit.toLong())
    )

    private fun toGetByServiceStatusParamsNextPage(
        tenantId: TenantId, serviceId: ServiceId, status: LoanStatus, fromDueAt: Instant, fromLoanId: LoanId, limit: Int
    ) = Params.of(
        "\$tenant_id", PrimitiveValue.utf8(tenantId.id),
        "\$service_id", PrimitiveValue.int64(serviceId),
        "\$loan_status", PrimitiveValue.utf8(status.name),
        "\$limit", PrimitiveValue.uint64(limit.toLong()),
        "\$from_due_at", PrimitiveValue.timestamp(fromDueAt),
        "\$from_loan_id", PrimitiveValue.utf8(fromLoanId)
    )

    private fun toGetServiceIdsByStatus(
        tenantId: TenantId, serviceIds: Collection<ServiceId>, status: LoanStatus
    ) = Params.of(
        "\$tenant_id", PrimitiveValue.utf8(tenantId.id),
        "\$service_ids", ListValue.of(*serviceIds.map { PrimitiveValue.int64(it) }.toTypedArray()),
        "\$loan_status", PrimitiveValue.utf8(status.name)
    )
}
