package ru.yandex.intranet.d.dao.providers;

import java.io.UncheckedIOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.annotation.Nullable;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.yandex.ydb.table.query.DataQueryResult;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.ListValue;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.StructValue;
import com.yandex.ydb.table.values.TupleValue;
import com.yandex.ydb.table.values.Value;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.QueryUtils;
import ru.yandex.intranet.d.datasource.Ydb;
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource;
import ru.yandex.intranet.d.datasource.model.WithTxId;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.providers.AccountsSettingsModel;
import ru.yandex.intranet.d.model.providers.AggregationAlgorithm;
import ru.yandex.intranet.d.model.providers.AggregationSettings;
import ru.yandex.intranet.d.model.providers.BillingMeta;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.providers.ProviderUISettings;
import ru.yandex.intranet.d.model.providers.RelatedResourceMapping;
import ru.yandex.intranet.d.util.ObjectMapperHolder;

import static com.yandex.ydb.table.values.PrimitiveValue.uint64;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static ru.yandex.intranet.d.dao.QueryUtils.toDeletedParam;
import static ru.yandex.intranet.d.datasource.Ydb.bool;
import static ru.yandex.intranet.d.datasource.Ydb.nullableUtf8;

/**
 * Providers DAO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@SuppressWarnings("rawtypes")
@Component
public class ProvidersDao {
    private static final TypeReference<Map<String, RelatedResourceMapping>> RELATED_RESOURCE_MAPPING_REFERENCE =
            new TypeReference<>() {
            };

    private final YdbQuerySource ydbQuerySource;
    private final ObjectReader accountsSettingsReader;
    private final ObjectWriter accountsSettingsWriter;
    private final ObjectReader billingMetaReader;
    private final ObjectWriter billingMetaWriter;
    private final ObjectReader relatedResourceMappingReader;
    private final ObjectWriter relatedResourceMappingWriter;
    private final ObjectReader aggregationSettingsReader;
    private final ObjectWriter aggregationSettingsWriter;
    private final ObjectReader aggregationAlgorithmReader;
    private final ObjectWriter aggregationAlgorithmWriter;
    private final ObjectReader providerUISettingsReader;
    private final ObjectWriter providerUISettingsWriter;

    public ProvidersDao(YdbQuerySource ydbQuerySource,
                        @Qualifier("ydbJsonObjectMapper") ObjectMapperHolder objectMapper) {
        this.ydbQuerySource = ydbQuerySource;
        this.accountsSettingsReader = objectMapper.getObjectMapper().readerFor(AccountsSettingsModel.class);
        this.accountsSettingsWriter = objectMapper.getObjectMapper().writerFor(AccountsSettingsModel.class);
        this.billingMetaReader = objectMapper.getObjectMapper().readerFor(BillingMeta.class);
        this.billingMetaWriter = objectMapper.getObjectMapper().writerFor(BillingMeta.class);
        this.relatedResourceMappingReader = objectMapper.getObjectMapper()
                .readerFor(RELATED_RESOURCE_MAPPING_REFERENCE);
        this.relatedResourceMappingWriter = objectMapper.getObjectMapper()
                .writerFor(RELATED_RESOURCE_MAPPING_REFERENCE);
        this.aggregationSettingsReader = objectMapper.getObjectMapper().readerFor(AggregationSettings.class);
        this.aggregationSettingsWriter = objectMapper.getObjectMapper().writerFor(AggregationSettings.class);
        this.aggregationAlgorithmReader = objectMapper.getObjectMapper().readerFor(AggregationAlgorithm.class);
        this.aggregationAlgorithmWriter = objectMapper.getObjectMapper().writerFor(AggregationAlgorithm.class);
        this.providerUISettingsReader = objectMapper.getObjectMapper().readerFor(ProviderUISettings.class);
        this.providerUISettingsWriter = objectMapper.getObjectMapper().writerFor(ProviderUISettings.class);
    }

    public Mono<Optional<ProviderModel>> getById(YdbTxSession session, String id, TenantId tenantId) {
        String query = ydbQuerySource.getQuery("yql.queries.providers.getOneById");
        Params params = Params.of("$id", utf8(id),
                "$tenant_id", utf8(tenantId.getId()));
        return session.executeDataQueryRetryable(query, params).map(this::toProvider);
    }

    public Mono<Tuple2<Optional<ProviderModel>, String>> getByIdStartTx(YdbTxSession session, String id,
                                                                        TenantId tenantId) {
        String query = ydbQuerySource.getQuery("yql.queries.providers.getOneById");
        Params params = Params.of("$id", utf8(id),
                "$tenant_id", utf8(tenantId.getId()));
        return session.executeDataQueryRetryable(query, params).map(r -> Tuples.of(toProvider(r), r.getTxId()));
    }

    public Mono<List<ProviderModel>> getBySourceTvmId(YdbTxSession session, long sourceTvmId,
                                                      TenantId tenantId) {
        String query = ydbQuerySource.getQuery("yql.queries.providers.getOneBySourceTvmId");
        Params params = Params.of("$source_tvm_id", PrimitiveValue.int64(sourceTvmId),
                "$tenant_id", utf8(tenantId.getId()));
        return session.executeDataQueryRetryable(query, params).map(this::toProviders);
    }

    public Mono<List<ProviderModel>> getByIds(YdbTxSession session, List<Tuple2<String, TenantId>> ids) {
        if (ids.isEmpty()) {
            return Mono.just(List.of());
        }
        String query = ydbQuerySource.getQuery("yql.queries.providers.getByIds");
        ListValue idsParam = ListValue.of(ids.stream().map(id -> TupleValue.of(utf8(id.getT1()),
                utf8(id.getT2().getId()))).toArray(TupleValue[]::new));
        Params params = Params.of("$ids", idsParam);
        return session.executeDataQueryRetryable(query, params).map(this::toProviders);
    }

    public Mono<List<ProviderModel>> getBySourceTvmIds(YdbTxSession session,
                                                       List<Tuple2<Long, TenantId>> sourceTvmIds) {
        if (sourceTvmIds.isEmpty()) {
            return Mono.just(List.of());
        }
        String query = ydbQuerySource.getQuery("yql.queries.providers.getBySourceTvmIds");
        ListValue idsParam = ListValue.of(sourceTvmIds.stream().map(id ->
                TupleValue.of(PrimitiveValue.int64(id.getT1()),
                        utf8(id.getT2().getId()))).toArray(TupleValue[]::new));
        Params params = Params.of("$ids", idsParam);
        return session.executeDataQueryRetryable(query, params).map(this::toProviders);
    }

    public Mono<Void> upsertProviderRetryable(YdbTxSession session, ProviderModel provider) {
        String query = ydbQuerySource.getQuery("yql.queries.providers.upsertOneProvider");
        Map<String, Value> fields = prepareProviderFields(provider);
        Params params = Params.of("$provider", StructValue.of(fields));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<Void> upsertProvidersRetryable(YdbTxSession session, List<ProviderModel> providers) {
        if (providers.isEmpty()) {
            return Mono.empty();
        }
        String query = ydbQuerySource.getQuery("yql.queries.providers.upsertManyProviders");
        Params params = Params.of("$providers", ListValue.of(providers.stream().map(provider -> {
            Map<String, Value> fields = prepareProviderFields(provider);
            return StructValue.of(fields);
        }).toArray(StructValue[]::new)));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<Void> updateProviderRetryable(YdbTxSession session, ProviderModel provider) {
        String query = ydbQuerySource.getQuery("yql.queries.providers.updateOneProvider");
        Map<String, Value> fields = prepareProviderFields(provider);
        Params params = Params.of("$provider", StructValue.of(fields));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<Void> updateProvidersRetryable(YdbTxSession session, List<ProviderModel> providers) {
        if (providers.isEmpty()) {
            return Mono.empty();
        }
        String query = ydbQuerySource.getQuery("yql.queries.providers.updateManyProviders");
        Params params = Params.of("$providers", ListValue.of(providers.stream().map(provider -> {
            Map<String, Value> fields = prepareProviderFields(provider);
            return StructValue.of(fields);
        }).toArray(StructValue[]::new)));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<List<ProviderModel>> getByTenant(YdbTxSession session, TenantId tenantId,
                                                 String providerIdFrom, int limit, boolean withDeleted) {
        String query;
        Params params;
        ListValue deletedParam = toDeletedParam(withDeleted);
        if (providerIdFrom != null) {
            query = ydbQuerySource.getQuery("yql.queries.providers.getNextPage");
            params = Params.of("$from_id", utf8(providerIdFrom),
                    "$limit", uint64(limit),
                    "$deleted", deletedParam,
                    "$tenant_id", utf8(tenantId.getId()));
        } else {
            query = ydbQuerySource.getQuery("yql.queries.providers.getFirstPage");
            params = Params.of("$limit", uint64(limit),
                    "$deleted", deletedParam,
                    "$tenant_id", utf8(tenantId.getId()));
        }
        return session.executeDataQueryRetryable(query, params).map(this::toProviders);
    }

    public Mono<WithTxId<List<ProviderModel>>> getAllByTenant(
            YdbTxSession session,
            TenantId tenantId,
            boolean withDeleted
    ) {
        return QueryUtils.getAllRows(session,
                (nextSession, providerIdFrom) -> {
                    ListValue deletedParam = toDeletedParam(withDeleted);
                    String query = ydbQuerySource.getQuery("yql.queries.providers.getMaxRows");
                    Params params = Params.of(
                            "$tenant_id", utf8(tenantId.getId()),
                            "$deleted", deletedParam,
                            "$from_id", nullableUtf8(providerIdFrom)
                    );
                    return session.executeDataQueryRetryable(query, params);
                },
                this::toProviders,
                ProviderModel::getId
        );
    }

    public Mono<Boolean> existsByKey(YdbTxSession session, TenantId tenantId, String key, boolean withDeleted) {
        ListValue deletedParam = toDeletedParam(withDeleted);
        String query = ydbQuerySource.getQuery("yql.queries.providers.existsByKey");
        Params params = Params.of("$key", utf8(key),
                "$tenant_id", utf8(tenantId.getId()),
                "$deleted", deletedParam);
        return session.executeDataQueryRetryable(query, params).map(this::toExists);
    }

    public Mono<Optional<ProviderModel>> getByKey(
            YdbTxSession session, TenantId tenantId, String key, boolean withDeleted
    ) {
        ListValue deletedParam = toDeletedParam(withDeleted);
        String query = ydbQuerySource.getQuery("yql.queries.providers.getOneByKey");
        Params params = Params.of("$key", utf8(key),
                "$tenant_id", utf8(tenantId.getId()),
                "$deleted", deletedParam);
        return session.executeDataQueryRetryable(query, params).map(this::toProvider);
    }

    public Mono<WithTxId<Void>> setHasDefaultQuotasRetryable(
            YdbTxSession session, TenantId tenantId, String id, boolean hasDefaultQuotas
    ) {
        String query = ydbQuerySource.getQuery("yql.queries.providers.setHasDefaultQuotas");
        Params params = Params.of(
                "$id", utf8(id),
                "$tenant_id", utf8(tenantId.getId()),
                "$has_default_quotas", bool(hasDefaultQuotas)
        );
        return session.executeDataQueryRetryable(query, params).map(result -> new WithTxId<>(null, result.getTxId()));
    }

    private boolean toExists(DataQueryResult result) {
        if (result.isEmpty() || result.getResultSetCount() > 1) {
            throw new IllegalStateException("Exactly one result set is required");
        }
        ResultSetReader reader = result.getResultSet(0);
        if (!reader.next() || reader.getRowCount() > 1) {
            throw new IllegalStateException("Exactly one result is required");
        }
        return reader.getColumn("provider_exists").getBool();
    }

    private Map<String, Value> prepareProviderFields(ProviderModel provider) {
        Map<String, Value> fields = new HashMap<>();
        fields.put("id", utf8(provider.getId()));
        fields.put("tenant_id", utf8(provider.getTenantId().getId()));
        fields.put("version", PrimitiveValue.int64(provider.getVersion()));
        fields.put("name_en", utf8(provider.getNameEn()));
        fields.put("name_ru", utf8(provider.getNameRu()));
        fields.put("description_en", utf8(provider.getDescriptionEn()));
        fields.put("description_ru", utf8(provider.getDescriptionRu()));
        fields.put("rest_api_uri", nullableUtf8(provider.getRestApiUri().orElse(null)));
        fields.put("grpc_api_uri", nullableUtf8(provider.getGrpcApiUri().orElse(null)));
        fields.put("source_tvm_id", PrimitiveValue.int64(provider.getSourceTvmId()));
        fields.put("destination_tvm_id", PrimitiveValue.int64(provider.getDestinationTvmId()));
        fields.put("service_id", PrimitiveValue.int64(provider.getServiceId()));
        fields.put("deleted", PrimitiveValue.bool(provider.isDeleted()));
        fields.put("read_only", PrimitiveValue.bool(provider.isReadOnly()));
        fields.put("multiple_accounts_per_folder", PrimitiveValue.bool(provider.isMultipleAccountsPerFolder()));
        fields.put("account_transfer_with_quota", PrimitiveValue.bool(provider.isAccountTransferWithQuota()));
        fields.put("managed", PrimitiveValue.bool(provider.isManaged()));
        fields.put("key", utf8(provider.getKey()));
        fields.put("accounts_settings",
                PrimitiveValue.jsonDocument(writeAccountsSettings(provider.getAccountsSettings())));
        fields.put("import_allowed", PrimitiveValue.bool(provider.isImportAllowed()));
        fields.put("accounts_spaces_supported", PrimitiveValue.bool(provider.isAccountsSpacesSupported()));
        fields.put("sync_enabled", PrimitiveValue.bool(provider.isSyncEnabled()));
        fields.put("grpc_tls_on", PrimitiveValue.bool(provider.isGrpcTlsOn()));
        fields.put("billing_meta", Ydb.nullableJsonDocument(provider.getBillingMeta().orElse(null),
                this::writeBillingMeta));
        fields.put("related_resource_mapping", Ydb.nullableJsonDocument(provider.getRelatedResourcesByResourceId()
                        .orElse(null),
                this::writeRelatedResourcesMapping));
        fields.put("tracker_component_id", PrimitiveValue.int64(provider.getTrackerComponentId()));
        fields.put("reserve_folder_id", Ydb.nullableUtf8(provider.getReserveFolderId().orElse(null)));
        fields.put("has_default_quotas", bool(provider.hasDefaultQuotas()));
        fields.put("allocated_supported", Ydb.nullableBool(provider.isAllocatedSupported().orElse(null)));
        fields.put("aggregation_settings", Ydb.nullableJsonDocument(provider.getAggregationSettings().orElse(null),
                this::writeAggregationSettings));
        fields.put("aggregation_algorithm", Ydb.nullableJsonDocument(provider.getAggregationAlgorithm().orElse(null),
                this::writeAggregationAlgorithm));
        fields.put("ui_settings", Ydb.nullableJsonDocument(provider.getUiSettings().orElse(null),
                this::writeUISettings));
        return fields;
    }

    private Optional<ProviderModel> toProvider(DataQueryResult result) {
        if (result.isEmpty()) {
            return Optional.empty();
        }
        if (result.getResultSetCount() > 1) {
            throw new IllegalStateException("Too many result sets");
        }
        ResultSetReader reader = result.getResultSet(0);
        if (!reader.next()) {
            return Optional.empty();
        }
        if (reader.getRowCount() > 1) {
            throw new IllegalStateException("Non unique provider");
        }
        ProviderModel provider = readOneProvider(reader, new HashMap<>());
        return Optional.of(provider);
    }

    private List<ProviderModel> toProviders(DataQueryResult result) {
        return toProviders(toReader(result));
    }

    @Nullable
    private ResultSetReader toReader(DataQueryResult result) {
        if (result.isEmpty()) {
            return null;
        }
        if (result.getResultSetCount() > 1) {
            throw new IllegalStateException("Too many result sets");
        }
        return result.getResultSet(0);
    }

    private List<ProviderModel> toProviders(ResultSetReader reader) {
        if (reader == null) {
            return List.of();
        }
        List<ProviderModel> providers = new ArrayList<>();
        Map<String, TenantId> tenantIdCache = new HashMap<>();
        while (reader.next()) {
            providers.add(readOneProvider(reader, tenantIdCache));
        }
        return providers;
    }

    private ProviderModel readOneProvider(ResultSetReader reader, Map<String, TenantId> tenantIdCache) {
        String id = reader.getColumn("id").getUtf8();
        String tenantIdString = reader.getColumn("tenant_id").getUtf8();
        TenantId tenantId = tenantIdCache.computeIfAbsent(tenantIdString, TenantId::new);
        long version = reader.getColumn("version").getInt64();
        String nameEn = reader.getColumn("name_en").getUtf8();
        String nameRu = reader.getColumn("name_ru").getUtf8();
        String descriptionEn = reader.getColumn("description_en").getUtf8();
        String descriptionRu = reader.getColumn("description_ru").getUtf8();
        String restApiUri = Ydb.utf8OrNull(reader.getColumn("rest_api_uri"));
        String grpcApiUri = Ydb.utf8OrNull(reader.getColumn("grpc_api_uri"));
        long sourceTvmId = reader.getColumn("source_tvm_id").getInt64();
        long destinationTvmId = reader.getColumn("destination_tvm_id").getInt64();
        long serviceId = reader.getColumn("service_id").getInt64();
        boolean deleted = reader.getColumn("deleted").getBool();
        boolean readOnly = reader.getColumn("read_only").getBool();
        boolean multipleAccountsPerFolder = reader.getColumn("multiple_accounts_per_folder").getBool();
        boolean accountTransferWithQuota = reader.getColumn("account_transfer_with_quota").getBool();
        boolean managed = reader.getColumn("managed").getBool();
        String key = reader.getColumn("key").getUtf8();
        AccountsSettingsModel accountsSettings = readAccountsSettings(reader
                .getColumn("accounts_settings").getJsonDocument());
        boolean importAllowed = reader.getColumn("import_allowed").getBool();
        boolean accountsSpacesSupported = reader.getColumn("accounts_spaces_supported").getBool();
        boolean syncEnabled = reader.getColumn("sync_enabled").getBool();
        boolean grpcTlsOn = reader.getColumn("grpc_tls_on").getBool();
        BillingMeta billingMeta = Ydb.jsonDocumentOrNull(reader.getColumn("billing_meta"), this::readBillingMeta);
        var relatedResourcesByResourceId = Ydb.jsonDocumentOrNull(reader.getColumn("related_resource_mapping"),
                this::readRelatedResourcesMapping);
        long trackerComponentId = reader.getColumn("tracker_component_id").getInt64();
        String reserveFolderId = Ydb.utf8OrNull(reader.getColumn("reserve_folder_id"));
        boolean hasDefaultQuotas = Ydb.boolOrDefault(reader.getColumn("has_default_quotas"), false);
        Boolean allocatedSupported = Ydb.boolOrNull(reader.getColumn("allocated_supported"));
        AggregationSettings aggregationSettings = Ydb.jsonDocumentOrNull(reader.getColumn("aggregation_settings"),
                this::readAggregationSettings);
        AggregationAlgorithm aggregationAlgorithm = Ydb.jsonDocumentOrNull(reader.getColumn("aggregation_algorithm"),
                this::readAggregationAlgorithm);
        ProviderUISettings uiSettings = Ydb.jsonDocumentOrNull(reader.getColumn("ui_settings"),
                this::readUISettings);

        return new ProviderModel(id, tenantId, version, nameEn, nameRu,
                descriptionEn, descriptionRu, restApiUri, grpcApiUri, sourceTvmId,
                destinationTvmId, serviceId, deleted, readOnly, multipleAccountsPerFolder,
                accountTransferWithQuota, managed, key, accountsSettings, importAllowed, accountsSpacesSupported,
                syncEnabled, grpcTlsOn, billingMeta, relatedResourcesByResourceId, trackerComponentId, reserveFolderId,
                hasDefaultQuotas, allocatedSupported, aggregationSettings, aggregationAlgorithm, uiSettings);
    }

    private AccountsSettingsModel readAccountsSettings(String json) {
        try {
            return accountsSettingsReader.readValue(json);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }

    private String writeAccountsSettings(AccountsSettingsModel accountsSettings) {
        try {
            return accountsSettingsWriter.writeValueAsString(accountsSettings);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }

    private BillingMeta readBillingMeta(String json) {
        try {
            return billingMetaReader.readValue(json);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }

    private String writeBillingMeta(BillingMeta billingMeta) {
        try {
            return billingMetaWriter.writeValueAsString(billingMeta);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }

    private Map<String, RelatedResourceMapping> readRelatedResourcesMapping(String json) {
        try {
            return relatedResourceMappingReader.readValue(json);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }

    private String writeRelatedResourcesMapping(Map<String, RelatedResourceMapping> relatedResourcesByResourceId) {
        try {
            return relatedResourceMappingWriter.writeValueAsString(relatedResourcesByResourceId);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }

    private AggregationSettings readAggregationSettings(String json) {
        try {
            return aggregationSettingsReader.readValue(json);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }

    private String writeAggregationSettings(AggregationSettings aggregationSettings) {
        try {
            return aggregationSettingsWriter.writeValueAsString(aggregationSettings);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }

    private AggregationAlgorithm readAggregationAlgorithm(String json) {
        try {
            return aggregationAlgorithmReader.readValue(json);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }

    private String writeAggregationAlgorithm(AggregationAlgorithm aggregationAlgorithm) {
        try {
            return aggregationAlgorithmWriter.writeValueAsString(aggregationAlgorithm);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }

    private ProviderUISettings readUISettings(String json) {
        try {
            return providerUISettingsReader.readValue(json);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }

    private String writeUISettings(ProviderUISettings value) {
        try {
            return providerUISettingsWriter.writeValueAsString(value);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }
}
