package ru.yandex.intranet.d.dao.quotas;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import com.yandex.ydb.table.query.DataQueryResult;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.ListValue;
import com.yandex.ydb.table.values.OptionalValue;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.StructValue;
import com.yandex.ydb.table.values.TupleType;
import com.yandex.ydb.table.values.TupleValue;
import com.yandex.ydb.table.values.Value;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuple3;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.AbstractDao;
import ru.yandex.intranet.d.dao.QueryUtils;
import ru.yandex.intranet.d.dao.folders.FolderDao;
import ru.yandex.intranet.d.datasource.Ydb;
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource;
import ru.yandex.intranet.d.datasource.model.WithTxId;
import ru.yandex.intranet.d.datasource.model.YdbExecuteScanQuerySettings;
import ru.yandex.intranet.d.datasource.model.YdbSession;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.folders.FolderType;
import ru.yandex.intranet.d.model.quotas.QuotaAggregationModel;
import ru.yandex.intranet.d.model.quotas.QuotaModel;

import static com.yandex.ydb.table.values.PrimitiveType.utf8;
import static ru.yandex.intranet.d.datasource.Ydb.nullableValue;

/**
 * Quotas DAO.
 *
 * @author Nikita Minin <spasitel@yandex-team.ru>
 */
@Component
public class QuotasDao extends AbstractDao<QuotaModel, QuotaModel.Key> {
    public static final int MAX_ROW_ANSWER = 1000; //TODO найти в библиотеке
    private final FolderDao folderDao;

    public QuotasDao(YdbQuerySource ydbQuerySource, FolderDao folderDao) {
        super(ydbQuerySource);
        this.folderDao = folderDao;
    }

    public Mono<Tuple2<List<FolderModel>, List<QuotaModel>>> getByServiceAndProvider(
            YdbTxSession session,
            long serviceId,
            String providerId,
            TenantId tenantId,
            int limit,
            @Nullable String from) {
        return getByServiceAndProviderStartTx(session, serviceId, providerId, tenantId, limit, from)
                .map(tuple3 -> Tuples.of(tuple3.getT1(), tuple3.getT2()));
    }

    public Mono<Tuple3<List<FolderModel>, List<QuotaModel>, String>> getByServiceAndProviderStartTx(
            YdbTxSession session,
            long serviceId,
            String providerId,
            TenantId tenantId,
            int limit,
            @Nullable String from) {

        //получить список по одному запросу и потом в getByFoldersAndProvider
        return getFoldersWithQuotas(session, serviceId, providerId, tenantId, limit, from).flatMap(folders ->
                folders.get().isEmpty() ?
                        Mono.just(Tuples.of(List.of(), List.of(), folders.getTransactionId())) :
                        getByFoldersAndProvider(session,
                                folders.get().stream().map(FolderModel::getId).collect(Collectors.toList()),
                                tenantId,
                                providerId)
                                .map(quotaModels -> Tuples.of(folders.get(), quotaModels, folders.getTransactionId())));
    }

    public Mono<WithTxId<List<FolderModel>>> getFoldersWithQuotas(YdbTxSession session, long serviceId,
                                                                   String providerId, TenantId tenantId, int limit,
                                                                   @Nullable String from) {
        String query = ydbQuerySource.getQuery("yql.queries.quota.getFoldersWithQuotaByProvider");
        Params params = Params.of(
                "$service_id", PrimitiveValue.int64(serviceId),
                "$provider_id", PrimitiveValue.utf8(providerId),
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$limit", PrimitiveValue.uint64(limit),
                "$from", Ydb.nullableUtf8(from));
        return session.executeDataQueryRetryable(query, params)
                .map(r -> new WithTxId<>(folderDao.toModels(r), r.getTxId()));
    }

    public Mono<List<QuotaModel>> getByFolders(
            YdbTxSession session,
            List<String> folderIds,
            TenantId tenantId) {
        return getByFolders(session, folderIds, tenantId, true);
    }

    /**
     * Get by folders.
     *
     * @param includeCompletelyZero - включать ли в результат полностью нулевые квоты, т.е. такие,
     *                              что их quota, balance и frozen_quota ноль.
     */
    public Mono<List<QuotaModel>> getByFolders(
            YdbTxSession session,
            List<String> folderIds,
            TenantId tenantId,
            boolean includeCompletelyZero) {
        return getByFoldersAndProvider(session, folderIds, tenantId, null, includeCompletelyZero);
    }

    /**
     * Get by folders and start transaction.
     *
     * @param includeCompletelyZero - включать ли в результат полностью нулевые квоты, т.е. такие,
     *                              что их quota, balance и frozen_quota ноль.
     */
    public Mono<Tuple2<List<QuotaModel>, String>> getByFoldersStartTx(
            YdbTxSession session,
            List<String> folderIds,
            TenantId tenantId,
            boolean includeCompletelyZero) {
        return getByFoldersAndProviderStartTx(session, folderIds, tenantId, null, includeCompletelyZero);
    }

    public Mono<List<QuotaModel>> getByFoldersAndProvider(
            YdbTxSession session,
            List<String> folderIds,
            TenantId tenantId,
            String providerId) {
        return getByFoldersAndProvider(session, folderIds, tenantId, providerId, true);
    }

    /**
     * Get by folders and provider.
     *
     * @param includeCompletelyZero - включать ли в результат полностью нулевые квоты, т.е. такие,
     *                              что их quota, balance и frozen_quota ноль.
     */
    public Mono<List<QuotaModel>> getByFoldersAndProvider(
            YdbTxSession session,
            List<String> folderIds,
            TenantId tenantId,
            String providerId,
            boolean includeCompletelyZero) {
        return getByFoldersAndProviderStartTx(session, folderIds, tenantId, providerId, includeCompletelyZero)
                .map(Tuple2::getT1);
    }

    public Mono<Tuple2<List<QuotaModel>, String>> getByFoldersAndProviderStartTx(
            YdbTxSession session,
            List<String> folderIds,
            TenantId tenantId,
            String providerId,
            boolean includeCompletelyZero) {

        String query = ydbQuerySource.getQuery("yql.queries.quota.getQuotas");
        Params params = Params.of(
                "$folder_ids", ListValue.of(folderIds.stream().
                        map(PrimitiveValue::utf8).toArray(PrimitiveValue[]::new)),
                "$provider_id", Ydb.nullableUtf8(providerId),
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$include_completely_zero", PrimitiveValue.bool(includeCompletelyZero),
                "$limit", PrimitiveValue.uint64(MAX_ROW_ANSWER + 1));

        return session.executeDataQueryRetryable(query, params).flatMap(dataQueryResult -> {
            //Первая пачка квот
            List<QuotaModel> models = toQuotas(dataQueryResult);
            final ResultSetReader resultSetReader = dataQueryResult.getResultSet(0);
            if (!resultSetReader.isTruncated()) {
                return Mono.just(Tuples.of(models, dataQueryResult.getTxId()));
            }
            return getNextPage(session,
                    folderIds,
                    tenantId,
                    providerId,
                    models.get(models.size() - 1),
                    includeCompletelyZero)
                    .expand(tuple -> {
                        if (tuple.getT2()) {
                            return getNextPage(session, folderIds, tenantId, providerId,
                                    tuple.getT1().get(tuple.getT1().size() - 1),
                                    includeCompletelyZero);
                        } else {
                            return Mono.empty();
                        }
                    }).map(Tuple2::getT1).reduce(models, (subSumQM, newQM) ->
                            Stream.concat(subSumQM.stream(), newQM.stream()).collect(Collectors.toList())
                    ).map(quotaModels -> Tuples.of(quotaModels, dataQueryResult.getTxId()));
        });
    }

    public Mono<List<QuotaModel>> listQuotasByFolder(YdbTxSession session, TenantId tenantId, String folderId,
                                                     int limit, String fromProviderId, String fromResourceId) {
        String query;
        Params params;
        if (fromProviderId != null && fromResourceId != null) {
            query = ydbQuerySource.getQuery("yql.queries.quota.getNextPageByFolder");
            params = Params.of("$from_provider_id", PrimitiveValue.utf8(fromProviderId),
                    "$from_resource_id", PrimitiveValue.utf8(fromResourceId),
                    "$limit", PrimitiveValue.uint64(limit),
                    "$folder_id", PrimitiveValue.utf8(folderId),
                    "$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        } else {
            query = ydbQuerySource.getQuery("yql.queries.quota.getFirstPageByFolder");
            params = Params.of("$limit", PrimitiveValue.uint64(limit),
                    "$folder_id", PrimitiveValue.utf8(folderId),
                    "$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        }
        return session.executeDataQueryRetryable(query, params).map(this::toQuotas);
    }

    private Mono<Tuple2<List<QuotaModel>, Boolean>> getNextPage(
            YdbTxSession session,
            List<String> folderIds,
            TenantId tenantId,
            String providerId,
            QuotaModel from,
            boolean includeCompletelyZero) {
        String query = ydbQuerySource.getQuery("yql.queries.quota.getQuotasFrom");
        //noinspection rawtypes
        Map<String, Value> fields = prepareFieldValues(from);
        Params params = Params.of(
                "$folder_ids", ListValue.of(folderIds.stream().
                        map(PrimitiveValue::utf8).toArray(PrimitiveValue[]::new)),
                "$provider_id", Ydb.nullableUtf8(providerId),
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$include_completely_zero", PrimitiveValue.bool(includeCompletelyZero),
                "$limit", PrimitiveValue.uint64(MAX_ROW_ANSWER + 1),
                "$from", StructValue.of(fields));

        return session.executeDataQueryRetryable(query, params).flatMap(dataQueryResult -> {
            List<QuotaModel> models = toQuotas(dataQueryResult);
            final ResultSetReader resultSetReader = dataQueryResult.getResultSet(0);
            return Mono.just(Tuples.of(models, resultSetReader.isTruncated()));
        });
    }

    public Mono<Optional<QuotaModel>> getOneQuota(
            YdbTxSession session,
            String folderId,
            String providerId,
            String resourceId,
            TenantId tenantId) {
        String query = ydbQuerySource.getQuery("yql.queries.quota.getById");
        Params params = Params.of("$folder_id", PrimitiveValue.utf8(folderId),
                "$provider_id", PrimitiveValue.utf8(providerId),
                "$resource_id", PrimitiveValue.utf8(resourceId),
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        return session.executeDataQueryRetryable(query, params).map(this::toQuota);
    }

    public Mono<Tuple2<List<FolderModel>, List<QuotaModel>>> getByServiceAndProviderAndResource(
            YdbTxSession session,
            long serviceId,
            String providerId,
            String resourceId,
            TenantId tenantId,
            int limit,
            @Nullable String from) {
        return getByServiceAndProviderAndResourceStartTx(
                session, serviceId, providerId, resourceId, tenantId, limit, from)
                .map(tuple3 -> Tuples.of(tuple3.getT1(), tuple3.getT2()));
    }

    public Mono<Tuple3<List<FolderModel>, List<QuotaModel>, String>> getByServiceAndProviderAndResourceStartTx(
            YdbTxSession session,
            long serviceId,
            String providerId,
            String resourceId,
            TenantId tenantId,
            int limit,
            @Nullable String from) {
        String query = ydbQuerySource.getQuery("yql.queries.quota.getByProviderAndResource");
        Params params = Params.create();
        params.put("$service_id", PrimitiveValue.int64(serviceId));
        params.put("$provider_id", Ydb.nullableUtf8(providerId));
        params.put("$resource_id", Ydb.nullableUtf8(resourceId));
        params.put("$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        params.put("$limit", PrimitiveValue.uint64(limit));
        params.put("$from", Ydb.nullableUtf8(from));
        return session.executeDataQueryRetryable(query, params).map(this::toQuotasAndFolders);

    }

    public Mono<List<QuotaModel>> getByKeys(YdbTxSession session, List<WithTenant<QuotaModel.Key>> keys) {
        if (keys.isEmpty()) {
            return Mono.just(List.of());
        }
        String query = ydbQuerySource.getQuery("yql.queries.quota.getByKeys");
        ListValue idsParam = ListValue.of(keys.stream()
                .map(id -> TupleValue.of(PrimitiveValue.utf8(id.getTenantId().getId()),
                        PrimitiveValue.utf8(id.getIdentity().getFolderId()),
                        PrimitiveValue.utf8(id.getIdentity().getProviderId()),
                        PrimitiveValue.utf8(id.getIdentity().getResourceId()))).toArray(TupleValue[]::new));
        return session.executeDataQueryRetryable(query, Params.of("$ids", idsParam)).map(this::toQuotas);
    }

    public Mono<WithTxId<List<QuotaModel>>> getAllByResourceIds(
            YdbTxSession session, TenantId tenantId, List<String> resourceIds
    ) {
        PrimitiveValue tenantIdValue = PrimitiveValue.utf8(tenantId.getId());
        ListValue resourceIdsValue = ListValue.of(
                resourceIds.stream().map(PrimitiveValue::utf8).toArray(PrimitiveValue[]::new)
        );
        TupleType fromKeyType = TupleType.of(List.of(utf8(), utf8(), utf8(), utf8()));
        String query = ydbQuerySource.getQuery("yql.queries.quota.getAllByResourceIds");
        return QueryUtils.getAllRows(session, (nextSession, quotaLastPrimaryKey) -> {
                    OptionalValue fromKey = nullableValue(fromKeyType,
                            quotaLastPrimaryKey == null ? null : TupleValue.of(
                                    // tenant_id, folder_id, provider_id, resource_id
                                    tenantIdValue,
                                    PrimitiveValue.utf8(quotaLastPrimaryKey.getFolderId()),
                                    PrimitiveValue.utf8(quotaLastPrimaryKey.getProviderId()),
                                    PrimitiveValue.utf8(quotaLastPrimaryKey.getResourceId())
                            )
                    );
                    Params params = Params.of(
                            "$tenant_id", tenantIdValue,
                            "$resource_ids", resourceIdsValue,
                            "$from_key", fromKey
                    );
                    return nextSession.executeDataQueryRetryable(query, params);
                },
                this::toModels,
                QuotaModel::toKey
        );
    }

    public Mono<List<QuotaModel>> getByResourceId(YdbTxSession session, String resourceId, int limit) {
        String query = ydbQuerySource.getQuery("yql.queries.quota.getQuotasByResourceId");
        Params params = Params.of("$resource_id", PrimitiveValue.utf8(resourceId),
                "$limit", PrimitiveValue.uint64(limit));
        return session.executeDataQueryRetryable(query, params).map(this::toQuotas);
    }

    public Mono<List<QuotaModel>> getByResourceIds(YdbTxSession session, List<String> resourceIds, int limit) {
        String query = ydbQuerySource.getQuery("yql.queries.quota.getQuotasByResourceIds");
        Params params = Params.of("$resource_ids", ListValue.of(resourceIds.stream().
                        map(PrimitiveValue::utf8).toArray(PrimitiveValue[]::new)),
                "$limit", PrimitiveValue.uint64(limit));
        return session.executeDataQueryRetryable(query, params).map(this::toQuotas);
    }

    private Tuple3<List<FolderModel>, List<QuotaModel>, String> toQuotasAndFolders(DataQueryResult result) {
        if (result.isEmpty()) {
            return Tuples.of(List.of(), List.of(), result.getTxId());
        }
        if (result.getResultSetCount() > 1) {
            throw new IllegalStateException("Too many result sets");
        }
        ResultSetReader reader = result.getResultSet(0);
        List<FolderModel> folders = new ArrayList<>();
        List<QuotaModel> quotas = new ArrayList<>();
        Map<String, TenantId> tenantIdCache = new HashMap<>();
        while (reader.next()) {
            quotas.add(readOneRow(reader, tenantIdCache));
            folders.add(folderDao.readOneRow(reader, tenantIdCache));
        }
        return Tuples.of(folders, quotas, result.getTxId());
    }

    public Mono<Void> deleteQuotaRetryable(
            YdbTxSession session,
            String folderId,
            String providerId,
            String resourceId,
            TenantId tenantId) {

        String query = ydbQuerySource.getQuery("yql.queries.quota.deleteOneQuota");
        Params params = Params.of("$folder_id", PrimitiveValue.utf8(folderId),
                "$resource_id", PrimitiveValue.utf8(resourceId),
                "$provider_id", PrimitiveValue.utf8(providerId),
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<WithTxId<Void>> deleteQuotasByModelRetryable(YdbTxSession session, List<QuotaModel> quotaModels) {
        return deleteQuotasRetryable(session, quotaModels.stream()
                .map(q -> new WithTenant<>(q.getTenantId(), q.toKey()))
                .collect(Collectors.toList()));
    }

    public Mono<WithTxId<Void>> deleteQuotasRetryable(YdbTxSession session, List<WithTenant<QuotaModel.Key>> ids) {
        if (ids.isEmpty()) {
            return Mono.empty();
        }
        String query = ydbQuerySource.getQuery("yql.queries.quota.deleteManyQuotas");
        ListValue idsParam = ListValue.of(ids.stream()
                .map(id -> TupleValue.of(PrimitiveValue.utf8(id.getTenantId().getId()),
                        PrimitiveValue.utf8(id.getIdentity().getFolderId()),
                        PrimitiveValue.utf8(id.getIdentity().getProviderId()),
                        PrimitiveValue.utf8(id.getIdentity().getResourceId()))).toArray(TupleValue[]::new));
        return session.executeDataQueryRetryable(query, Params.of("$ids", idsParam))
                .map(v -> new WithTxId<>(null, v.getTxId()));
    }

    public Mono<Void> deleteQuotasModelsRetryable(YdbTxSession session, List<QuotaModel> quotas) {
        if (quotas.isEmpty()) {
            return Mono.empty();
        }
        String query = ydbQuerySource.getQuery("yql.queries.quota.deleteManyQuotas");
        ListValue idsParam = ListValue.of(quotas.stream()
                .map(quota -> TupleValue.of(PrimitiveValue.utf8(quota.getTenantId().getId()),
                        PrimitiveValue.utf8(quota.getFolderId()),
                        PrimitiveValue.utf8(quota.getProviderId()),
                        PrimitiveValue.utf8(quota.getResourceId()))).toArray(TupleValue[]::new));
        return session.executeDataQueryRetryable(query, Params.of("$ids", idsParam)).then();
    }

    public Mono<List<QuotaModel>> getByProviderFoldersResources(YdbTxSession session, TenantId tenantId,
                                                                Set<String> folderIds, String providerId,
                                                                Set<String> resourceIds) {
        return getByProviderFoldersResources(session, tenantId, folderIds, providerId, resourceIds,
                Ydb.MAX_RESPONSE_ROWS);
    }

    public Flux<WithTenant<QuotaModel.Key>> scanZeroQuotas(YdbSession session, long limit, Duration timeout) {
        String query = ydbQuerySource.getQuery("yql.queries.quota.scanZeroQuotas");
        return session.executeScanQuery(query, Params.of("$limit", PrimitiveValue.uint64(limit)),
                        YdbExecuteScanQuerySettings.builder().timeout(timeout).build())
                .flatMapIterable(reader -> {
                    List<WithTenant<QuotaModel.Key>> page = new ArrayList<>();
                    Map<String, TenantId> tenantIdCache = new HashMap<>();
                    while (reader.next()) {
                        String tenantIdString = reader.getColumn("tenant_id").getUtf8();
                        TenantId tenantId = tenantIdCache.computeIfAbsent(tenantIdString, TenantId::new);
                        String folderId = reader.getColumn("folder_id").getUtf8();
                        String resourceId = reader.getColumn("resource_id").getUtf8();
                        String providerId = reader.getColumn("provider_id").getUtf8();
                        page.add(new WithTenant<>(tenantId, new QuotaModel.Key(folderId, providerId, resourceId)));
                    }
                    return page;
                });
    }

    public Mono<List<QuotaAggregationModel>> scanQuotasAggregationSubset(
            YdbSession session, TenantId tenantId, String providerId, Collection<String> resourceIds,
            Duration timeout) {
        String query = ydbQuerySource.getQuery("yql.queries.quota.scanQuotasAggregationSubset");
        Params params = Params.of("$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$provider_id", PrimitiveValue.utf8(providerId),
                "$resource_ids", ListValue.of(resourceIds.stream()
                        .map(PrimitiveValue::utf8).toArray(PrimitiveValue[]::new)));
        return session.executeScanQuery(query, params, YdbExecuteScanQuerySettings.builder().timeout(timeout).build())
                .flatMapIterable(reader -> {
                    List<QuotaAggregationModel> page = new ArrayList<>();
                    while (reader.next()) {
                        String resourceId = reader.getColumn("resource_id").getUtf8();
                        Long quota = Ydb.int64OrNull(reader.getColumn("quota"));
                        Long balance = Ydb.int64OrNull(reader.getColumn("balance"));
                        Long provided = Ydb.int64OrNull(reader.getColumn("provided"));
                        Long allocated = Ydb.int64OrNull(reader.getColumn("allocated"));
                        long serviceId = reader.getColumn("service_id").getInt64();
                        String folderId = reader.getColumn("folder_id").getUtf8();
                        String accountId = reader.getColumn("account_id").getUtf8();
                        FolderType folderType = FolderType.valueOf(reader.getColumn("folder_type").getUtf8());
                        page.add(new QuotaAggregationModel(resourceId, quota, balance, provided, allocated,
                                serviceId, folderId, accountId, folderType));
                    }
                    return page;
                }).collectList();
    }

    public Mono<List<QuotaAggregationModel>> getForAggregationByProviderAndResources(
            YdbTxSession session, TenantId tenantId, String providerId, Collection<String> resourceIds) {
        List<String> sortedResourceIds = resourceIds.stream().sorted().toList();
        String firstPageQuery = ydbQuerySource
                .getQuery("yql.queries.quota.getQuotasAggregationFirstPage");
        Params firstPageParams = Params.of("$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$provider_id", PrimitiveValue.utf8(providerId),
                "$resource_ids", ListValue.of(sortedResourceIds.stream().map(PrimitiveValue::utf8)
                        .toArray(PrimitiveValue[]::new)),
                "$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS));
        return session.executeDataQueryRetryable(firstPageQuery, firstPageParams).flatMap(firstPageResult -> {
            QuotaAggregationPage page = toQuotaAggregationModels(firstPageResult);
            if ((!firstPageResult.getResultSet(0).isTruncated() && page.models.size() < Ydb.MAX_RESPONSE_ROWS)
                    || page.cursor == null) {
                return Mono.just(page.models);
            }
            return getNextPageForAggregationByProviderAndResources(session, tenantId, providerId, sortedResourceIds,
                    page.cursor).expand(tuple -> {
                if ((!tuple.getT2() && tuple.getT1().models.size() < Ydb.MAX_RESPONSE_ROWS)
                        || tuple.getT1().cursor == null) {
                    return Mono.empty();
                } else {
                    return getNextPageForAggregationByProviderAndResources(session, tenantId, providerId,
                            sortedResourceIds, tuple.getT1().cursor);
                }
            }).map(p -> p.getT1().models).reduce(page.models, (l, r) -> Stream.concat(l.stream(), r.stream())
                    .collect(Collectors.toList()));
        });
    }

    public Mono<List<QuotaAggregationModel>> getForAggregationByProvider(
            YdbTxSession session, TenantId tenantId, String providerId) {
        String firstPageQuery = ydbQuerySource
                .getQuery("yql.queries.quota.getProviderQuotasAggregationFirstPage");
        Params firstPageParams = Params.of("$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$provider_id", PrimitiveValue.utf8(providerId),
                "$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS));
        return session.executeDataQueryRetryable(firstPageQuery, firstPageParams).flatMap(firstPageResult -> {
            QuotaAggregationPage page = toQuotaAggregationModels(firstPageResult);
            if ((!firstPageResult.getResultSet(0).isTruncated() && page.models.size() < Ydb.MAX_RESPONSE_ROWS)
                    || page.cursor == null) {
                return Mono.just(page.models);
            }
            return getNextPageForAggregationByProvider(session, tenantId, providerId, page.cursor).expand(tuple -> {
                if ((!tuple.getT2() && tuple.getT1().models.size() < Ydb.MAX_RESPONSE_ROWS)
                        || tuple.getT1().cursor == null) {
                    return Mono.empty();
                } else {
                    return getNextPageForAggregationByProvider(session, tenantId, providerId, tuple.getT1().cursor);
                }
            }).map(p -> p.getT1().models).reduce(page.models, (l, r) -> Stream.concat(l.stream(), r.stream())
                    .collect(Collectors.toList()));
        });
    }

    Mono<List<QuotaModel>> getByProviderFoldersResources(YdbTxSession session, TenantId tenantId,
                                                         Set<String> folderIds, String providerId,
                                                         Set<String> resourceIds, long limit) {
        if (folderIds.isEmpty() || resourceIds.isEmpty()) {
            return Mono.empty();
        }
        List<String> sortedFolderIds = folderIds.stream().sorted().collect(Collectors.toList());
        List<String> sortedResourceIds = resourceIds.stream().sorted().collect(Collectors.toList());
        String firstPageQuery = ydbQuerySource.getQuery("yql.queries.quota.getByProviderFoldersResourcesFirstPage");
        ListValue firstPageFolderIdsParam = ListValue.of(sortedFolderIds.stream()
                .map(PrimitiveValue::utf8)
                .toArray(PrimitiveValue[]::new));
        ListValue firstPageResourceIdsParam = ListValue.of(sortedResourceIds.stream()
                .map(PrimitiveValue::utf8)
                .toArray(PrimitiveValue[]::new));
        HashMap<String, Value<?>> firstPageParamsMap = new HashMap<>();
        firstPageParamsMap.put("$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        firstPageParamsMap.put("$provider_id", PrimitiveValue.utf8(providerId));
        firstPageParamsMap.put("$folder_ids", firstPageFolderIdsParam);
        firstPageParamsMap.put("$resource_ids", firstPageResourceIdsParam);
        firstPageParamsMap.put("$limit", PrimitiveValue.uint64(limit));
        Params firstPageParams = Params.copyOf(firstPageParamsMap);
        return session.executeDataQueryRetryable(firstPageQuery, firstPageParams).flatMap(firstPageResult -> {
            List<QuotaModel> firstPageModels = toModels(firstPageResult);
            if (!firstPageResult.getResultSet(0).isTruncated() && firstPageModels.size() < limit) {
                return Mono.just(firstPageModels);
            }
            QuotaModel firstFrom = firstPageModels.get(firstPageModels.size() - 1);
            List<String> firstRemainingFolderIds = sortedFolderIds.stream()
                    .filter(id -> id.compareTo(firstFrom.getFolderId()) > 0)
                    .toList();
            List<String> firstRemainingResourceIds = sortedResourceIds.stream()
                    .filter(id -> id.compareTo(firstFrom.getResourceId()) > 0)
                    .toList();
            if (firstRemainingFolderIds.isEmpty() && firstRemainingResourceIds.isEmpty()) {
                return Mono.just(firstPageModels);
            }
            return getByProviderFoldersResources(session, tenantId, providerId, sortedFolderIds, sortedResourceIds,
                    firstFrom, limit).expand(tuple -> {
                if (!tuple.getT2() && tuple.getT1().size() < limit) {
                    return Mono.empty();
                } else {
                    QuotaModel nextFrom = tuple.getT1().get(tuple.getT1().size() - 1);
                    List<String> nextRemainingFolderIds = sortedFolderIds.stream()
                            .filter(id -> id.compareTo(nextFrom.getFolderId()) > 0)
                            .toList();
                    List<String> nextRemainingResourceIds = sortedResourceIds.stream()
                            .filter(id -> id.compareTo(nextFrom.getResourceId()) > 0)
                            .toList();
                    if (nextRemainingFolderIds.isEmpty() && nextRemainingResourceIds.isEmpty()) {
                        return Mono.empty();
                    }
                    return getByProviderFoldersResources(session, tenantId, providerId, sortedFolderIds,
                            sortedResourceIds, nextFrom, limit);
                }
            }).map(Tuple2::getT1).reduce(firstPageModels, (l, r) -> Stream.concat(l.stream(), r.stream())
                    .collect(Collectors.toList()));
        });
    }

    private Mono<Tuple2<List<QuotaModel>, Boolean>> getByProviderFoldersResources(
            YdbTxSession session, TenantId tenantId, String providerId, List<String> sortedFolderIds,
            List<String> sortedResourceIds, QuotaModel from, long limit) {
        List<String> remainingFolderIds = sortedFolderIds.stream()
                .filter(id -> id.compareTo(from.getFolderId()) > 0)
                .toList();
        List<String> remainingResourceIds = sortedResourceIds.stream()
                .filter(id -> id.compareTo(from.getResourceId()) > 0)
                .toList();
        String nextPageQuery;
        Map<String, Value<?>> nextPageParamsMap = new HashMap<>();
        nextPageParamsMap.put("$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        nextPageParamsMap.put("$provider_id", PrimitiveValue.utf8(providerId));
        nextPageParamsMap.put("$limit", PrimitiveValue.uint64(limit));
        if (remainingFolderIds.isEmpty() && !remainingResourceIds.isEmpty()) {
            nextPageQuery = ydbQuerySource
                    .getQuery("yql.queries.quota.getByProviderFoldersResourcesLastPage");
            ListValue remainingResourceIdsParam = ListValue.of(remainingResourceIds.stream()
                    .map(PrimitiveValue::utf8)
                    .toArray(PrimitiveValue[]::new));
            nextPageParamsMap.put("$from_folder_id", PrimitiveValue.utf8(from.getFolderId()));
            nextPageParamsMap.put("$from_resource_ids", remainingResourceIdsParam);
        } else if (!remainingFolderIds.isEmpty() && remainingResourceIds.isEmpty()) {
            nextPageQuery = ydbQuerySource
                    .getQuery("yql.queries.quota.getByProviderFoldersResourcesFirstPage");
            ListValue remainingFolderIdsParam = ListValue.of(remainingFolderIds.stream()
                    .map(PrimitiveValue::utf8)
                    .toArray(PrimitiveValue[]::new));
            ListValue resourceIdsParam = ListValue.of(sortedResourceIds.stream()
                    .map(PrimitiveValue::utf8)
                    .toArray(PrimitiveValue[]::new));
            nextPageParamsMap.put("$folder_ids", remainingFolderIdsParam);
            nextPageParamsMap.put("$resource_ids", resourceIdsParam);
        } else {
            nextPageQuery = ydbQuerySource
                    .getQuery("yql.queries.quota.getByProviderFoldersResourcesNextPage");
            ListValue resourceIdsParam = ListValue.of(sortedResourceIds.stream()
                    .map(PrimitiveValue::utf8)
                    .toArray(PrimitiveValue[]::new));
            ListValue remainingFolderIdsParam = ListValue.of(remainingFolderIds.stream()
                    .map(PrimitiveValue::utf8)
                    .toArray(PrimitiveValue[]::new));
            ListValue remainingResourceIdsParam = ListValue.of(remainingResourceIds.stream()
                    .map(PrimitiveValue::utf8)
                    .toArray(PrimitiveValue[]::new));
            nextPageParamsMap.put("$resource_ids", resourceIdsParam);
            nextPageParamsMap.put("$from_folder_id", PrimitiveValue.utf8(from.getFolderId()));
            nextPageParamsMap.put("$from_folder_ids", remainingFolderIdsParam);
            nextPageParamsMap.put("$from_resource_ids", remainingResourceIdsParam);
        }
        Params nextPageParams = Params.copyOf(nextPageParamsMap);
        return session.executeDataQueryRetryable(nextPageQuery, nextPageParams).map(nextPageResult -> {
            List<QuotaModel> nextPageModels = toModels(nextPageResult);
            return Tuples.of(nextPageModels, nextPageResult.getResultSet(0).isTruncated());
        });
    }

    private Mono<Tuple2<QuotaAggregationPage, Boolean>> getNextPageForAggregationByProviderAndResources(
            YdbTxSession session, TenantId tenantId, String providerId, List<String> resourceIds,
            QuotaAggregationCursor from) {
        String fromResourceId = from.model.getResourceId();
        String fromFolderId = from.folderId;
        List<String> remainingResourceIds = resourceIds.stream()
                .filter(id -> id.compareTo(fromResourceId) > 0).toList();
        String nextPageQuery = remainingResourceIds.isEmpty()
                ? ydbQuerySource.getQuery("yql.queries.quota.getQuotasAggregationLastPage")
                : ydbQuerySource.getQuery("yql.queries.quota.getQuotasAggregationNextPage");
        Map<String, Value<?>> paramsMap = new HashMap<>();
        paramsMap.put("$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS));
        paramsMap.put("$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        paramsMap.put("$provider_id", PrimitiveValue.utf8(providerId));
        paramsMap.put("$from_folder_id", PrimitiveValue.utf8(fromFolderId));
        paramsMap.put("$from_resource_id", PrimitiveValue.utf8(fromResourceId));
        if (!remainingResourceIds.isEmpty()) {
            paramsMap.put("$resource_ids", ListValue.of(remainingResourceIds.stream().map(PrimitiveValue::utf8)
                    .toArray(PrimitiveValue[]::new)));
        }
        Params nextPageParams = Params.copyOf(paramsMap);
        return session.executeDataQueryRetryable(nextPageQuery, nextPageParams).map(nextPageResult -> {
            QuotaAggregationPage page = toQuotaAggregationModels(nextPageResult);
            return Tuples.of(page, nextPageResult.getResultSet(0).isTruncated());
        });
    }

    private Mono<Tuple2<QuotaAggregationPage, Boolean>> getNextPageForAggregationByProvider(
            YdbTxSession session, TenantId tenantId, String providerId, QuotaAggregationCursor from) {
        String fromResourceId = from.model.getResourceId();
        String fromFolderId = from.folderId;
        String nextPageQuery = ydbQuerySource
                .getQuery("yql.queries.quota.getProviderQuotasAggregationNextPage");
        Map<String, Value<?>> paramsMap = new HashMap<>();
        paramsMap.put("$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS));
        paramsMap.put("$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        paramsMap.put("$provider_id", PrimitiveValue.utf8(providerId));
        paramsMap.put("$from_folder_id", PrimitiveValue.utf8(fromFolderId));
        paramsMap.put("$from_resource_id", PrimitiveValue.utf8(fromResourceId));
        Params nextPageParams = Params.copyOf(paramsMap);
        return session.executeDataQueryRetryable(nextPageQuery, nextPageParams).map(nextPageResult -> {
            QuotaAggregationPage page = toQuotaAggregationModels(nextPageResult);
            return Tuples.of(page, nextPageResult.getResultSet(0).isTruncated());
        });
    }

    private QuotaAggregationPage toQuotaAggregationModels(DataQueryResult result) {
        if (result.isEmpty()) {
            return new QuotaAggregationPage(List.of(), null);
        }
        if (result.getResultSetCount() > 1) {
            throw new IllegalStateException("Too many result sets");
        }
        ResultSetReader reader = result.getResultSet(0);
        List<QuotaAggregationModel> list = new ArrayList<>();
        String folderId = null;
        while (reader.next()) {
            String resourceId = reader.getColumn("resource_id").getUtf8();
            long quota = reader.getColumn("quota").getInt64();
            long balance = reader.getColumn("balance").getInt64();
            long serviceId = reader.getColumn("service_id").getInt64();
            folderId = reader.getColumn("folder_id").getUtf8();
            FolderType folderType = FolderType.valueOf(reader.getColumn("folder_type").getUtf8());
            list.add(new QuotaAggregationModel(resourceId, quota, balance, null, null, serviceId,
                    folderId, null, folderType));
        }
        QuotaAggregationCursor cursor = !list.isEmpty() && folderId != null
                ? new QuotaAggregationCursor(list.get(list.size() - 1), folderId) : null;
        return new QuotaAggregationPage(list, cursor);
    }

    private Optional<QuotaModel> toQuota(DataQueryResult result) {
        if (result.isEmpty()) {
            return Optional.empty();
        }
        if (result.getResultSetCount() > 1) {
            throw new IllegalStateException("Too many result sets");
        }
        ResultSetReader reader = result.getResultSet(0);
        if (!reader.next()) {
            return Optional.empty();
        }
        if (reader.getRowCount() > 1) {
            throw new IllegalStateException("Non unique quota");
        }
        QuotaModel quota = readOneRow(reader, new HashMap<>());
        return Optional.of(quota);
    }

    private List<QuotaModel> toQuotas(DataQueryResult result) {
        if (result.isEmpty()) {
            return List.of();
        }
        if (result.getResultSetCount() > 1) {
            throw new IllegalStateException("Too many result sets");
        }
        ResultSetReader reader = result.getResultSet(0);
        List<QuotaModel> quotas = new ArrayList<>();
        Map<String, TenantId> tenantIdCache = new HashMap<>();
        while (reader.next()) {
            quotas.add(readOneRow(reader, tenantIdCache));
        }
        return quotas;
    }

    @Override
    protected WithTenant<QuotaModel.Key> getIdentityWithTenant(QuotaModel model) {
        return new WithTenant<>(model.getTenantId(), model.toKey());
    }

    @Override
    protected Params getIdentityParams(QuotaModel.Key id) {
        return Params.create()
                .put("$folder_id", PrimitiveValue.utf8(id.getFolderId()))
                .put("$provider_id", PrimitiveValue.utf8(id.getProviderId()))
                .put("$resource_id", PrimitiveValue.utf8(id.getResourceId()));
    }

    @SuppressWarnings("rawtypes")
    @Override
    protected Map<String, Value> prepareFieldValues(QuotaModel quota) {
        Map<String, Value> fields = new HashMap<>();
        fields.put("tenant_id", PrimitiveValue.utf8(quota.getTenantId().getId()));
        fields.put("folder_id", PrimitiveValue.utf8(quota.getFolderId()));
        fields.put("provider_id", PrimitiveValue.utf8(quota.getProviderId()));
        fields.put("resource_id", PrimitiveValue.utf8(quota.getResourceId()));
        fields.put("quota", PrimitiveValue.int64(quota.getQuota()));
        fields.put("balance", PrimitiveValue.int64(quota.getBalance()));
        fields.put("frozen_quota", PrimitiveValue.int64(quota.getFrozenQuota()));
        return fields;
    }

    @Override
    protected QuotaModel readOneRow(ResultSetReader reader, Map<String, TenantId> tenantIdCache) {
        String tenantIdString = reader.getColumn("tenant_id").getUtf8();
        TenantId tenantId = tenantIdCache.computeIfAbsent(tenantIdString, TenantId::new);
        String folderId = reader.getColumn("folder_id").getUtf8();
        String resourceId = reader.getColumn("resource_id").getUtf8();
        String providerId = reader.getColumn("provider_id").getUtf8();
        long quota = reader.getColumn("quota").getInt64();
        long balance = reader.getColumn("balance").getInt64();
        long frozenQuota = reader.getColumn("frozen_quota").getInt64();
        return new QuotaModel(tenantId, folderId, providerId, resourceId, quota, balance, frozenQuota);
    }

    @Override
    protected String queryKeyPrefix() {
        return "yql.queries.quota";
    }

    private record QuotaAggregationPage(List<QuotaAggregationModel> models,
                                        @Nullable QuotaAggregationCursor cursor) {
    }

    private record QuotaAggregationCursor(QuotaAggregationModel model, String folderId) {
    }

}
