package ru.yandex.intranet.d.dao.resources;

import java.io.UncheckedIOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.google.common.collect.Lists;
import com.yandex.ydb.table.query.DataQueryResult;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.ListValue;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.StructValue;
import com.yandex.ydb.table.values.TupleValue;
import com.yandex.ydb.table.values.Value;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;
import ru.yandex.intranet.d.dao.JsonFieldHelper;
import ru.yandex.intranet.d.dao.QueryUtils;
import ru.yandex.intranet.d.datasource.Ydb;
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource;
import ru.yandex.intranet.d.datasource.model.WithTxId;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.providers.AggregationSettings;
import ru.yandex.intranet.d.model.resources.ResourceBaseIdentity;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.resources.ResourceSegmentSettingsModel;
import ru.yandex.intranet.d.model.resources.ResourceUnitsModel;
import ru.yandex.intranet.d.services.imports.ResourceIdentity;
import ru.yandex.intranet.d.util.ObjectMapperHolder;

import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static ru.yandex.intranet.d.dao.QueryUtils.toDeletedParam;
import static ru.yandex.intranet.d.datasource.Ydb.nullableUtf8;
import static ru.yandex.intranet.d.datasource.Ydb.uint64;

/**
 * Resources DAO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class ResourcesDao {

    private final YdbQuerySource ydbQuerySource;
    private final JsonFieldHelper<Set<ResourceSegmentSettingsModel>> segmentsHelper;
    private final ObjectReader resourceUnitsReader;
    private final ObjectWriter resourceUnitsWriter;
    private final JsonFieldHelper<Set<String>> requestedSegmentsHelper;
    private final ObjectReader aggregationSettingsReader;
    private final ObjectWriter aggregationSettingsWriter;

    public ResourcesDao(YdbQuerySource ydbQuerySource,
                        @Qualifier("ydbJsonObjectMapper") ObjectMapperHolder objectMapper) {
        this.ydbQuerySource = ydbQuerySource;
        this.segmentsHelper = new JsonFieldHelper<>(objectMapper, new TypeReference<>() {
        });
        this.requestedSegmentsHelper = new JsonFieldHelper<>(objectMapper, new TypeReference<>() { });
        this.resourceUnitsReader = objectMapper.getObjectMapper().readerFor(ResourceUnitsModel.class);
        this.resourceUnitsWriter = objectMapper.getObjectMapper().writerFor(ResourceUnitsModel.class);
        this.aggregationSettingsReader = objectMapper.getObjectMapper().readerFor(AggregationSettings.class);
        this.aggregationSettingsWriter = objectMapper.getObjectMapper().writerFor(AggregationSettings.class);
    }

    public Mono<Optional<ResourceModel>> getById(YdbTxSession session, String id, TenantId tenantId) {
        String query = ydbQuerySource.getQuery("yql.queries.resources.getOneById");
        Params params = Params.of("$id", PrimitiveValue.utf8(id),
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        return session.executeDataQueryRetryable(query, params).map(this::toResource);
    }

    public Mono<List<ResourceModel>> getAllByIds(YdbTxSession session, List<Tuple2<String, TenantId>> ids) {
        String query = ydbQuerySource.getQuery("yql.queries.folders.getAllByIds");
        return Flux.fromIterable(Lists.partition(ids, 5000))
                .concatMap(sublist -> QueryUtils.getAllRows(session,
                                (nextSession, lastId) -> {
                                    ListValue idsParam = ListValue.of(sublist.stream()
                                            .map(id -> TupleValue.of(PrimitiveValue.utf8(id.getT1()),
                                                    PrimitiveValue.utf8(id.getT2().getId())))
                                            .toArray(TupleValue[]::new));
                                    return nextSession.executeDataQueryRetryable(query, Params.of("$ids", idsParam,
                                            "$from_id", nullableUtf8(lastId)));
                                },
                                this::getResourceModels,
                                ResourceModel::getId)
                        .map(WithTxId::get))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    @SuppressWarnings("checkstyle:ParameterNumber")
    public Mono<List<ResourceModel>> getByProviderResourceTypeAndSegments(
            YdbTxSession session, TenantId tenantId, String providerId, String resourceTypeId,
            Set<String> requestedSegments, String fromId, int limit, boolean withDeleted
    ) {
        String query = ydbQuerySource.getQuery("yql.queries.folders.getByProviderResourceTypeAndSegments");
        Params params = Params.copyOf(Map.of(
                "$tenant_id", utf8(tenantId.getId()),
                "$provider_id", utf8(providerId),
                "$resource_type_id", utf8(resourceTypeId),
                "$deleted", toDeletedParam(withDeleted),
                "$requested_segments", requestedSegmentsHelper.writeOptionalJson(requestedSegments),
                "$from_id", nullableUtf8(fromId),
                "$limit", uint64(limit))
        );
        return session.executeDataQueryRetryable(query, params).map(this::toResources);
    }

    public Mono<Tuple2<Optional<ResourceModel>, String>> getByIdStartTx(YdbTxSession session, String id,
                                                                        TenantId tenantId) {
        String query = ydbQuerySource.getQuery("yql.queries.resources.getOneById");
        Params params = Params.of("$id", PrimitiveValue.utf8(id),
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        return session.executeDataQueryRetryable(query, params).map(r -> Tuples.of(toResource(r), r.getTxId()));
    }

    public Mono<List<ResourceModel>> getByIds(YdbTxSession session, List<Tuple2<String, TenantId>> ids) {
        if (ids.isEmpty()) {
            return Mono.just(List.of());
        }
        String query = ydbQuerySource.getQuery("yql.queries.resources.getByIds");
        ListValue idsParam = ListValue.of(ids.stream().map(id -> TupleValue.of(PrimitiveValue.utf8(id.getT1()),
                PrimitiveValue.utf8(id.getT2().getId()))).toArray(TupleValue[]::new));
        Params params = Params.of("$ids", idsParam);
        return session.executeDataQueryRetryable(query, params).map(this::toResources);
    }

    public Mono<List<ResourceModel>> getAllByIdentities(YdbTxSession session, List<ResourceIdentity> identities,
                                                        boolean withDeleted) {
        if (identities.isEmpty()) {
            return Mono.just(List.of());
        }
        String query = ydbQuerySource.getQuery("yql.queries.resources.getByIdentity");
        Set<ResourceIdentity> identitySet = new HashSet<>(identities);

        return QueryUtils.getAllRows(session,
                (nextSession, lastId) -> {
                    ListValue idsParam = ListValue.of(identities.stream().map(id -> TupleValue.of(
                            PrimitiveValue.utf8(id.getTenantId().getId()),
                            PrimitiveValue.utf8(id.getProviderId()),
                            PrimitiveValue.utf8(id.getResourceTypeId()))
                    ).toArray(TupleValue[]::new));
                    ListValue deletedParam = toDeletedParam(withDeleted);
                    Params params = Params.of("$ids", idsParam,
                            "$from_id", nullableUtf8(lastId),
                            "$deleted", deletedParam);
                    return nextSession.executeDataQueryRetryable(query, params);
                },
                this::getResourceModels,
                ResourceModel::getId
        ).map(withTxId -> withTxId.get().stream()
                .filter(resourceModel -> identitySet.contains(toIdentity(resourceModel)))
                .collect(Collectors.toList())
        );
    }

    public Mono<List<ResourceModel>> getAllByBaseIdentities(YdbTxSession session,
                                                            List<ResourceBaseIdentity> identities,
                                                            boolean withDeleted) {
        if (identities.isEmpty()) {
            return Mono.just(List.of());
        }
        String query = ydbQuerySource.getQuery("yql.queries.resources.getByIdentity");

        return QueryUtils.getAllRows(session,
                (nextSession, lastId) -> {
                    ListValue idsParam = ListValue.of(identities.stream().map(id -> TupleValue.of(
                            PrimitiveValue.utf8(id.getTenantId().getId()),
                            PrimitiveValue.utf8(id.getProviderId()),
                            PrimitiveValue.utf8(id.getResourceTypeId()))
                    ).toArray(TupleValue[]::new));
                    ListValue deletedParam = toDeletedParam(withDeleted);
                    Params params = Params.of("$ids", idsParam,
                            "$from_id", nullableUtf8(lastId),
                            "$deleted", deletedParam);
                    return nextSession.executeDataQueryRetryable(query, params);
                },
                this::getResourceModels,
                ResourceModel::getId
        ).map(WithTxId::get);
    }

    private ResourceIdentity toIdentity(ResourceModel resourceModels) {
        return new ResourceIdentity(
                resourceModels.getResourceTypeId(),
                resourceModels.getTenantId(),
                resourceModels.getAccountsSpacesId(),
                resourceModels.getSegments(),
                resourceModels.getProviderId()
        );
    }

    public Mono<Void> upsertResourceRetryable(YdbTxSession session, ResourceModel resource) {
        String query = ydbQuerySource.getQuery("yql.queries.resources.upsertOneResource");
        Map<String, Value> fields = prepareResourceFields(resource);
        Params params = Params.of("$resource", StructValue.of(fields));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<Void> upsertResourcesRetryable(YdbTxSession session, List<ResourceModel> resources) {
        if (resources.isEmpty()) {
            return Mono.empty();
        }
        String query = ydbQuerySource.getQuery("yql.queries.resources.upsertManyResources");
        Params params = Params.of("$resources", ListValue.of(resources.stream().map(resource -> {
            Map<String, Value> fields = prepareResourceFields(resource);
            return StructValue.of(fields);
        }).toArray(StructValue[]::new)));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<Void> updateResourceRetryable(YdbTxSession session, ResourceModel resource) {
        String query = ydbQuerySource.getQuery("yql.queries.resources.updateOneResource");
        Map<String, Value> fields = prepareResourceFields(resource);
        Params params = Params.of("$resource", StructValue.of(fields));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<Void> updateResourcesRetryable(YdbTxSession session, List<ResourceModel> resources) {
        if (resources.isEmpty()) {
            return Mono.empty();
        }
        String query = ydbQuerySource.getQuery("yql.queries.resources.updateManyResources");
        Params params = Params.of("$resources", ListValue.of(resources.stream().map(resource -> {
            Map<String, Value> fields = prepareResourceFields(resource);
            return StructValue.of(fields);
        }).toArray(StructValue[]::new)));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<List<ResourceModel>> getByTenant(YdbTxSession session, TenantId tenantId,
                                                 String resourceIdFrom, int limit, boolean withDeleted) {
        String query;
        Params params;
        ListValue deletedParam = toDeletedParam(withDeleted);
        if (resourceIdFrom != null) {
            query = ydbQuerySource.getQuery("yql.queries.resources.getNextPage");
            params = Params.of("$from_id", PrimitiveValue.utf8(resourceIdFrom),
                    "$limit", PrimitiveValue.uint64(limit),
                    "$deleted", deletedParam,
                    "$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        } else {
            query = ydbQuerySource.getQuery("yql.queries.resources.getFirstPage");
            params = Params.of("$limit", PrimitiveValue.uint64(limit),
                    "$deleted", deletedParam,
                    "$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        }
        return session.executeDataQueryRetryable(query, params).map(this::toResources);
    }

    public Mono<List<ResourceModel>> getByProvider(YdbTxSession session, String providerId, TenantId tenantId,
                                                   String resourceIdFrom, int limit, boolean withDeleted) {
        String query;
        Params params;
        ListValue deletedParam = toDeletedParam(withDeleted);
        if (resourceIdFrom != null) {
            query = ydbQuerySource.getQuery("yql.queries.resources.getByProviderNextPage");
            params = Params.of("$from_id", PrimitiveValue.utf8(resourceIdFrom),
                    "$limit", PrimitiveValue.uint64(limit),
                    "$provider_id", PrimitiveValue.utf8(providerId),
                    "$deleted", deletedParam,
                    "$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        } else {
            query = ydbQuerySource.getQuery("yql.queries.resources.getByProviderFirstPage");
            params = Params.of("$limit", PrimitiveValue.uint64(limit),
                    "$provider_id", PrimitiveValue.utf8(providerId),
                    "$deleted", deletedParam,
                    "$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        }
        return session.executeDataQueryRetryable(query, params).map(this::toResources);
    }

    public Mono<List<ResourceModel>> getAllByProvider(YdbTxSession session, String providerId, TenantId tenantId,
                                                      boolean withDeleted) {
        return getAllByProvider(session, providerId, tenantId, withDeleted, false);
    }

    public Mono<List<ResourceModel>> getAllByProvider(YdbTxSession session, String providerId, TenantId tenantId,
                                                      boolean withDeleted, boolean onlyWithDefaultQuota) {
        ListValue deletedParam = toDeletedParam(withDeleted);

        return QueryUtils.getAllRows(session,
                (nextSession, lastId) -> {
                    String query = ydbQuerySource.getQuery("yql.queries.resources.getByProviderMaxRows");
                    Params params = Params.of(
                            "$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                            "$provider_id", PrimitiveValue.utf8(providerId),
                            "$deleted", deletedParam,
                            "$from_id", nullableUtf8(lastId),
                            "$only_with_default_quota", PrimitiveValue.bool(onlyWithDefaultQuota)
                    );
                    return nextSession.executeDataQueryRetryable(query, params);
                },
                this::getResourceModels,
                ResourceModel::getId
        ).map(WithTxId::get);
    }

    public Mono<List<ResourceModel>> getAllByProviders(YdbTxSession session, Collection<String> providerIds,
                                                       TenantId tenantId, boolean withDeleted) {
        if (providerIds.isEmpty()) {
            return Mono.just(List.of());
        }
        String firstPageQuery = ydbQuerySource.getQuery("yql.queries.resources.getByProvidersFirstPage");
        ListValue deletedParam = toDeletedParam(withDeleted);
        Params firstPageParams = Params.of(
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS),
                "$deleted", deletedParam,
                "$provider_ids", ListValue.of(providerIds.stream().map(PrimitiveValue::utf8)
                        .toArray(PrimitiveValue[]::new))
        );
        return session.executeDataQueryRetryable(firstPageQuery, firstPageParams).flatMap(firstPageResult -> {
            List<ResourceModel> models = toResources(firstPageResult);
            if (!firstPageResult.getResultSet(0).isTruncated() && models.size() < Ydb.MAX_RESPONSE_ROWS) {
                return Mono.just(models);
            }
            return getNextPageByProviders(session, providerIds, tenantId, withDeleted, models.get(models.size() - 1))
                    .expand(tuple -> {
                if (!tuple.getT2() && tuple.getT1().size() < Ydb.MAX_RESPONSE_ROWS) {
                    return Mono.empty();
                } else {
                    return getNextPageByProviders(session, providerIds, tenantId, withDeleted,
                            tuple.getT1().get(tuple.getT1().size() - 1));
                }
            }).map(Tuple2::getT1).reduce(models, (l, r) -> Stream.concat(l.stream(), r.stream())
                    .collect(Collectors.toList()));
        });
    }

    public Mono<Boolean> existsByUnitsEnsembleId(YdbTxSession session, String unitsEnsembleId, TenantId tenantId,
                                                 boolean withDeleted) {
        ListValue deletedParam = toDeletedParam(withDeleted);
        String query = ydbQuerySource.getQuery("yql.queries.resources.existsByUnitsEnsembleId");
        Params params = Params.of("$units_ensemble_id", PrimitiveValue.utf8(unitsEnsembleId),
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$deleted", deletedParam);
        return session.executeDataQueryRetryable(query, params).map(this::toExists);
    }

    public Mono<Boolean> existsByProviderAndKey(YdbTxSession session, String providerId, TenantId tenantId,
                                                String key, boolean withDeleted) {
        ListValue deletedParam = toDeletedParam(withDeleted);
        String query = ydbQuerySource.getQuery("yql.queries.resources.existsByProviderAndKey");
        Params params = Params.of("$provider_id", PrimitiveValue.utf8(providerId),
                "$key", PrimitiveValue.utf8(key),
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$deleted", deletedParam);
        return session.executeDataQueryRetryable(query, params).map(this::toExists);
    }

    public Mono<List<ResourceModel>> getAllByProviderResourceType(YdbTxSession session, String providerId,
                                                                  String resourceTypeId, TenantId tenantId,
                                                                  boolean withDeleted) {
        String firstPageQuery = ydbQuerySource.getQuery("yql.queries.resources.getByProviderResourceTypeFirstPage");
        ListValue deletedParam = toDeletedParam(withDeleted);
        Params firstPageParams = Params.of(
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS),
                "$deleted", deletedParam,
                "$provider_id", PrimitiveValue.utf8(providerId),
                "$resource_type_id", PrimitiveValue.utf8(resourceTypeId)
        );
        return session.executeDataQueryRetryable(firstPageQuery, firstPageParams).flatMap(firstPageResult -> {
            List<ResourceModel> models = toResources(firstPageResult);
            if (!firstPageResult.getResultSet(0).isTruncated() && models.size() < Ydb.MAX_RESPONSE_ROWS) {
                return Mono.just(models);
            }
            return getNextPageByProviderResourceType(session, providerId, resourceTypeId, tenantId, withDeleted,
                    models.get(models.size() - 1))
                    .expand(tuple -> {
                        if (!tuple.getT2() && tuple.getT1().size() < Ydb.MAX_RESPONSE_ROWS) {
                            return Mono.empty();
                        } else {
                            return getNextPageByProviderResourceType(session, providerId, resourceTypeId, tenantId,
                                    withDeleted, tuple.getT1().get(tuple.getT1().size() - 1));
                        }
                    }).map(Tuple2::getT1).reduce(models, (l, r) -> Stream.concat(l.stream(), r.stream())
                            .collect(Collectors.toList()));
        });
    }

    public Mono<List<ResourceModel>> getAllByProviderAccountsSpace(YdbTxSession session, String providerId,
                                                                   String accountsSpaceId, TenantId tenantId,
                                                                   boolean withDeleted) {
        String firstPageQuery = ydbQuerySource.getQuery("yql.queries.resources.getByProviderAccountsSpaceFirstPage");
        ListValue deletedParam = toDeletedParam(withDeleted);
        Params firstPageParams = Params.of(
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS),
                "$deleted", deletedParam,
                "$provider_id", PrimitiveValue.utf8(providerId),
                "$accounts_spaces_id", PrimitiveValue.utf8(accountsSpaceId)
        );
        return session.executeDataQueryRetryable(firstPageQuery, firstPageParams).flatMap(firstPageResult -> {
            List<ResourceModel> models = toResources(firstPageResult);
            if (!firstPageResult.getResultSet(0).isTruncated() && models.size() < Ydb.MAX_RESPONSE_ROWS) {
                return Mono.just(models);
            }
            return getNextPageByProviderAccountsSpace(session, providerId, accountsSpaceId, tenantId, withDeleted,
                    models.get(models.size() - 1))
                    .expand(tuple -> {
                        if (!tuple.getT2() && tuple.getT1().size() < Ydb.MAX_RESPONSE_ROWS) {
                            return Mono.empty();
                        } else {
                            return getNextPageByProviderAccountsSpace(session, providerId, accountsSpaceId, tenantId,
                                    withDeleted, tuple.getT1().get(tuple.getT1().size() - 1));
                        }
                    }).map(Tuple2::getT1).reduce(models, (l, r) -> Stream.concat(l.stream(), r.stream())
                            .collect(Collectors.toList()));
        });
    }

    private Mono<Tuple2<List<ResourceModel>, Boolean>> getNextPageByProviders(
            YdbTxSession session, Collection<String> providerIds, TenantId tenantId, boolean withDeleted,
            ResourceModel from) {
        String nextPageQuery = ydbQuerySource.getQuery("yql.queries.resources.getByProvidersNextPage");
        ListValue deletedParam = toDeletedParam(withDeleted);
        Map<String, Value<?>> paramsMap = Map.of(
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS),
                "$deleted", deletedParam,
                "$provider_ids", ListValue.of(providerIds.stream().map(PrimitiveValue::utf8)
                        .toArray(PrimitiveValue[]::new)),
                "$from_id", PrimitiveValue.utf8(from.getId()));
        Params nextPageParams = Params.copyOf(paramsMap);
        return session.executeDataQueryRetryable(nextPageQuery, nextPageParams).map(nextPageResult -> {
            List<ResourceModel> models = toResources(nextPageResult);
            return Tuples.of(models, nextPageResult.getResultSet(0).isTruncated());
        });
    }

    private Mono<Tuple2<List<ResourceModel>, Boolean>> getNextPageByProviderResourceType(
            YdbTxSession session, String providerId, String resourceTypeId, TenantId tenantId, boolean withDeleted,
            ResourceModel from) {
        String nextPageQuery = ydbQuerySource.getQuery("yql.queries.resources.getByProviderResourceTypeNextPage");
        ListValue deletedParam = toDeletedParam(withDeleted);
        Map<String, Value<?>> paramsMap = Map.of(
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS),
                "$deleted", deletedParam,
                "$provider_id", PrimitiveValue.utf8(providerId),
                "$resource_type_id", PrimitiveValue.utf8(resourceTypeId),
                "$from_id", PrimitiveValue.utf8(from.getId()));
        Params nextPageParams = Params.copyOf(paramsMap);
        return session.executeDataQueryRetryable(nextPageQuery, nextPageParams).map(nextPageResult -> {
            List<ResourceModel> models = toResources(nextPageResult);
            return Tuples.of(models, nextPageResult.getResultSet(0).isTruncated());
        });
    }

    private Mono<Tuple2<List<ResourceModel>, Boolean>> getNextPageByProviderAccountsSpace(
            YdbTxSession session, String providerId, String accountsSpaceId, TenantId tenantId, boolean withDeleted,
            ResourceModel from) {
        String nextPageQuery = ydbQuerySource.getQuery("yql.queries.resources.getByProviderAccountsSpaceNextPage");
        ListValue deletedParam = toDeletedParam(withDeleted);
        Map<String, Value<?>> paramsMap = Map.of(
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS),
                "$deleted", deletedParam,
                "$provider_id", PrimitiveValue.utf8(providerId),
                "$accounts_spaces_id", PrimitiveValue.utf8(accountsSpaceId),
                "$from_id", PrimitiveValue.utf8(from.getId()));
        Params nextPageParams = Params.copyOf(paramsMap);
        return session.executeDataQueryRetryable(nextPageQuery, nextPageParams).map(nextPageResult -> {
            List<ResourceModel> models = toResources(nextPageResult);
            return Tuples.of(models, nextPageResult.getResultSet(0).isTruncated());
        });
    }

    private boolean toExists(DataQueryResult result) {
        if (result.isEmpty() || result.getResultSetCount() > 1) {
            throw new IllegalStateException("Exactly one result set is required");
        }
        ResultSetReader reader = result.getResultSet(0);
        if (!reader.next() || reader.getRowCount() > 1) {
            throw new IllegalStateException("Exactly one result is required");
        }
        return reader.getColumn("resource_exists").getBool();
    }

    private Optional<ResourceModel> toResource(DataQueryResult result) {
        if (result.isEmpty()) {
            return Optional.empty();
        }
        if (result.getResultSetCount() > 1) {
            throw new IllegalStateException("Too many result sets");
        }
        ResultSetReader reader = result.getResultSet(0);
        if (!reader.next()) {
            return Optional.empty();
        }
        if (reader.getRowCount() > 1) {
            throw new IllegalStateException("Non unique resource");
        }
        ResourceModel resource = readOneResource(reader, new HashMap<>());
        return Optional.of(resource);
    }

    private List<ResourceModel> toResources(DataQueryResult result) {
        if (result.isEmpty()) {
            return List.of();
        }
        if (result.getResultSetCount() > 1) {
            throw new IllegalStateException("Too many result sets");
        }
        ResultSetReader reader = result.getResultSet(0);
        return getResourceModels(reader);
    }

    private List<ResourceModel> getResourceModels(ResultSetReader reader) {
        List<ResourceModel> resources = new ArrayList<>();
        Map<String, TenantId> tenantIdCache = new HashMap<>();
        while (reader.next()) {
            resources.add(readOneResource(reader, tenantIdCache));
        }
        return resources;
    }

    private ResourceModel readOneResource(ResultSetReader reader, Map<String, TenantId> tenantIdCache) {
        String id = reader.getColumn("id").getUtf8();
        String tenantIdString = reader.getColumn("tenant_id").getUtf8();
        TenantId tenantId = tenantIdCache.computeIfAbsent(tenantIdString, TenantId::new);
        long version = reader.getColumn("version").getInt64();
        String nameEn = reader.getColumn("name_en").getUtf8();
        String nameRu = reader.getColumn("name_ru").getUtf8();
        String descriptionEn = reader.getColumn("description_en").getUtf8();
        String descriptionRu = reader.getColumn("description_ru").getUtf8();
        boolean deleted = reader.getColumn("deleted").getBool();
        String unitsEnsembleId = reader.getColumn("units_ensemble_id").getUtf8();
        String providerId = reader.getColumn("provider_id").getUtf8();
        boolean managed = reader.getColumn("managed").getBool();
        boolean orderable = reader.getColumn("orderable").getBool();
        String resourceTypeId = Ydb.utf8OrNull(reader.getColumn("resource_type_id"));
        Set<ResourceSegmentSettingsModel> segments = segmentsHelper.read(reader.getColumn("segments"));
        ResourceUnitsModel resourceUnits = readResourceUnits(reader
                .getColumn("resource_units").getJsonDocument());
        String key = reader.getColumn("key").getUtf8();
        boolean readOnly = reader.getColumn("read_only").getBool();
        String baseUnitId = reader.getColumn("base_unit_id").getUtf8();
        String accountsSpacesId = Ydb.utf8OrNull(reader.getColumn("accounts_spaces_id"));
        Long defaultQuota = Ydb.int64OrNull(reader.getColumn("default_quota"));
        boolean virtual = Ydb.boolOrDefault(reader.getColumn("virtual"), false);
        Boolean allocatedSupported = Ydb.boolOrNull(reader.getColumn("allocated_supported"));
        AggregationSettings aggregationSettings = Ydb.jsonDocumentOrNull(reader.getColumn("aggregation_settings"),
                this::readAggregationSettings);
        return new ResourceModel(id, tenantId, version, nameEn, nameRu, descriptionEn, descriptionRu,
                deleted, unitsEnsembleId, providerId, resourceTypeId, segments, resourceUnits, managed, orderable,
                key, readOnly, baseUnitId, accountsSpacesId, defaultQuota, virtual, allocatedSupported,
                aggregationSettings);
    }

    private ResourceUnitsModel readResourceUnits(String json) {
        try {
            return resourceUnitsReader.readValue(json);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }

    private String writeResourceUnits(ResourceUnitsModel resourceUnits) {
        try {
            return resourceUnitsWriter.writeValueAsString(resourceUnits);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }

    private AggregationSettings readAggregationSettings(String json) {
        try {
            return aggregationSettingsReader.readValue(json);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }

    private String writeAggregationSettings(AggregationSettings aggregationSettings) {
        try {
            return aggregationSettingsWriter.writeValueAsString(aggregationSettings);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }

    private Map<String, Value> prepareResourceFields(ResourceModel resource) {
        Map<String, Value> fields = new HashMap<>();
        fields.put("id", PrimitiveValue.utf8(resource.getId()));
        fields.put("tenant_id", PrimitiveValue.utf8(resource.getTenantId().getId()));
        fields.put("version", PrimitiveValue.int64(resource.getVersion()));
        fields.put("name_en", PrimitiveValue.utf8(resource.getNameEn()));
        fields.put("name_ru", PrimitiveValue.utf8(resource.getNameRu()));
        fields.put("description_en", PrimitiveValue.utf8(resource.getDescriptionEn()));
        fields.put("description_ru", PrimitiveValue.utf8(resource.getDescriptionRu()));
        fields.put("deleted", PrimitiveValue.bool(resource.isDeleted()));
        fields.put("units_ensemble_id", PrimitiveValue.utf8(resource.getUnitsEnsembleId()));
        fields.put("provider_id", PrimitiveValue.utf8(resource.getProviderId()));
        fields.put("managed", PrimitiveValue.bool(resource.isManaged()));
        fields.put("orderable", PrimitiveValue.bool(resource.isOrderable()));
        fields.put("resource_type_id", Ydb.nullableUtf8(resource.getResourceTypeId()));
        fields.put("segments", segmentsHelper.writeOptional(resource.getSegments()));
        fields.put("resource_units", PrimitiveValue.jsonDocument(writeResourceUnits(resource.getResourceUnits())));
        fields.put("key", PrimitiveValue.utf8(resource.getKey()));
        fields.put("read_only", PrimitiveValue.bool(resource.isReadOnly()));
        fields.put("base_unit_id", PrimitiveValue.utf8(resource.getBaseUnitId()));
        fields.put("accounts_spaces_id", Ydb.nullableUtf8(resource.getAccountsSpacesId()));
        fields.put("default_quota", Ydb.nullableInt64(resource.getDefaultQuota().orElse(null)));
        fields.put("virtual", PrimitiveValue.bool(resource.isVirtual()));
        fields.put("allocated_supported", Ydb.nullableBool(resource.isAllocatedSupported().orElse(null)));
        fields.put("aggregation_settings", Ydb.nullableJsonDocument(resource.getAggregationSettings().orElse(null),
                this::writeAggregationSettings));
        return fields;
    }
}
