package ru.yandex.intranet.d.dao.resources.segmentations;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import com.yandex.ydb.table.query.DataQueryResult;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.ListValue;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.StructValue;
import com.yandex.ydb.table.values.TupleValue;
import com.yandex.ydb.table.values.Value;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.JsonFieldHelper;
import ru.yandex.intranet.d.datasource.Ydb;
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.model.resources.segmentations.SegmentationUISettings;
import ru.yandex.intranet.d.util.ObjectMapperHolder;

/**
 * Resource segmentations DAO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class ResourceSegmentationsDao {

    private final YdbQuerySource ydbQuerySource;
    private final JsonFieldHelper<SegmentationUISettings> uiSettingsHelper;

    public ResourceSegmentationsDao(
            YdbQuerySource ydbQuerySource,
            @Qualifier("ydbJsonObjectMapper") ObjectMapperHolder objectMapper
    ) {
        this.ydbQuerySource = ydbQuerySource;
        this.uiSettingsHelper = new JsonFieldHelper<>(objectMapper, SegmentationUISettings.class);
    }

    public Mono<Optional<ResourceSegmentationModel>> getById(YdbTxSession session, String id, TenantId tenantId) {
        String query = ydbQuerySource.getQuery("yql.queries.resourceSegmentations.getOneById");
        Params params = Params.of("$id", PrimitiveValue.utf8(id),
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        return session.executeDataQueryRetryable(query, params).map(this::toResourceSegmentation);
    }

    public Mono<Tuple2<Optional<ResourceSegmentationModel>, String>> getByIdStartTx(YdbTxSession session, String id,
                                                                                    TenantId tenantId) {
        String query = ydbQuerySource.getQuery("yql.queries.resourceSegmentations.getOneById");
        Params params = Params.of("$id", PrimitiveValue.utf8(id),
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        return session.executeDataQueryRetryable(query, params)
                .map(r -> Tuples.of(toResourceSegmentation(r), r.getTxId()));
    }

    public Mono<List<ResourceSegmentationModel>> getByIds(YdbTxSession session, List<Tuple2<String, TenantId>> ids) {
        if (ids.isEmpty()) {
            return Mono.just(List.of());
        }
        String query = ydbQuerySource.getQuery("yql.queries.resourceSegmentations.getByIds");
        ListValue idsParam = ListValue.of(ids.stream().map(id -> TupleValue.of(PrimitiveValue.utf8(id.getT1()),
                PrimitiveValue.utf8(id.getT2().getId()))).toArray(TupleValue[]::new));
        Params params = Params.of("$ids", idsParam);
        return session.executeDataQueryRetryable(query, params).map(this::toResourceTypeSegmentations);
    }

    public Mono<List<ResourceSegmentationModel>> getAllByProvidersAndKeys(
            YdbTxSession session,
            List<Tuple2<ResourceSegmentationModel.ProviderKey, TenantId>> providerKeys
    ) {
        if (providerKeys.isEmpty()) {
            return Mono.just(List.of());
        }
        String query = ydbQuerySource.getQuery("yql.queries.resourceSegmentations.getAllByProvidersAndKeys");
        ListValue idsParam = ListValue.of(providerKeys.stream().map(key -> TupleValue.of(
                PrimitiveValue.utf8(key.getT2().getId()),
                PrimitiveValue.utf8(key.getT1().getProviderId()),
                PrimitiveValue.utf8(key.getT1().getKey())
        )).toArray(TupleValue[]::new));
        Params params = Params.of("$provider_keys", idsParam);
        return session.executeDataQueryRetryable(query, params).map(this::toResourceTypeSegmentations);
    }

    public Mono<Void> upsertResourceSegmentationRetryable(YdbTxSession session,
                                                          ResourceSegmentationModel resourceSegmentation) {
        String query = ydbQuerySource.getQuery("yql.queries.resourceSegmentations.upsertOneResourceSegmentation");
        Map<String, Value> fields = prepareResourceSegmentationFields(resourceSegmentation);
        Params params = Params.of("$resource_segmentation", StructValue.of(fields));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<Void> upsertResourceSegmentationsRetryable(YdbTxSession session,
                                                           List<ResourceSegmentationModel> resourceSegmentations) {
        if (resourceSegmentations.isEmpty()) {
            return Mono.empty();
        }
        String query = ydbQuerySource.getQuery("yql.queries.resourceSegmentations.upsertManyResourceSegmentations");
        Params params = Params.of("$resource_segmentations", ListValue.of(resourceSegmentations.stream()
                .map(resourceSegmentation -> {
                    Map<String, Value> fields = prepareResourceSegmentationFields(resourceSegmentation);
                    return StructValue.of(fields);
                }).toArray(StructValue[]::new)));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<Void> updateResourceSegmentationRetryable(YdbTxSession session, ResourceSegmentationModel resource) {
        String query = ydbQuerySource.getQuery("yql.queries.resourceSegmentations.updateOneResourceSegmentation");
        Map<String, Value> fields = prepareResourceSegmentationFields(resource);
        Params params = Params.of("$resource_segmentation", StructValue.of(fields));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<Void> updateResourceSegmentationsRetryable(YdbTxSession session,
                                                           List<ResourceSegmentationModel> resources) {
        if (resources.isEmpty()) {
            return Mono.empty();
        }
        String query = ydbQuerySource.getQuery("yql.queries.resourceSegmentations.updateManyResourceSegmentations");
        Params params = Params.of("$resource_segmentations", ListValue.of(resources.stream()
                .map(resourceSegmentation -> {
                    Map<String, Value> fields = prepareResourceSegmentationFields(resourceSegmentation);
                    return StructValue.of(fields);
                }).toArray(StructValue[]::new)));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<List<ResourceSegmentationModel>> getByProvider(YdbTxSession session, String providerId,
                                                               TenantId tenantId, String resourceSegmentationIdFrom,
                                                               int limit, boolean withDeleted) {
        String query;
        Params params;
        ListValue deletedParam = withDeleted
                ? ListValue.of(PrimitiveValue.bool(true), PrimitiveValue.bool(false))
                : ListValue.of(PrimitiveValue.bool(false));
        if (resourceSegmentationIdFrom != null) {
            query = ydbQuerySource.getQuery("yql.queries.resourceSegmentations.getByProviderNextPage");
            params = Params.of("$from_id", PrimitiveValue.utf8(resourceSegmentationIdFrom),
                    "$limit", PrimitiveValue.uint64(limit),
                    "$provider_id", PrimitiveValue.utf8(providerId),
                    "$deleted", deletedParam,
                    "$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        } else {
            query = ydbQuerySource.getQuery("yql.queries.resourceSegmentations.getByProviderFirstPage");
            params = Params.of("$limit", PrimitiveValue.uint64(limit),
                    "$provider_id", PrimitiveValue.utf8(providerId),
                    "$deleted", deletedParam,
                    "$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        }
        return session.executeDataQueryRetryable(query, params).map(this::toResourceTypeSegmentations);
    }

    public Mono<Boolean> existsByProviderAndKey(YdbTxSession session, String providerId, TenantId tenantId,
                                                String key, boolean withDeleted) {
        ListValue deletedParam = withDeleted
                ? ListValue.of(PrimitiveValue.bool(true), PrimitiveValue.bool(false))
                : ListValue.of(PrimitiveValue.bool(false));
        String query = ydbQuerySource.getQuery("yql.queries.resourceSegmentations.existsByProviderAndKey");
        Params params = Params.of("$provider_id", PrimitiveValue.utf8(providerId),
                "$key", PrimitiveValue.utf8(key),
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$deleted", deletedParam);
        return session.executeDataQueryRetryable(query, params).map(this::toExists);
    }

    private boolean toExists(DataQueryResult result) {
        if (result.isEmpty() || result.getResultSetCount() > 1) {
            throw new IllegalStateException("Exactly one result set is required");
        }
        ResultSetReader reader = result.getResultSet(0);
        if (!reader.next() || reader.getRowCount() > 1) {
            throw new IllegalStateException("Exactly one result is required");
        }
        return reader.getColumn("resource_segmentation_exists").getBool();
    }

    private Optional<ResourceSegmentationModel> toResourceSegmentation(DataQueryResult result) {
        if (result.isEmpty()) {
            return Optional.empty();
        }
        if (result.getResultSetCount() > 1) {
            throw new IllegalStateException("Too many result sets");
        }
        ResultSetReader reader = result.getResultSet(0);
        if (!reader.next()) {
            return Optional.empty();
        }
        if (reader.getRowCount() > 1) {
            throw new IllegalStateException("Non unique resource segmentation");
        }
        ResourceSegmentationModel resourceSegmentation = readOneResourceSegmentation(reader, new HashMap<>());
        return Optional.of(resourceSegmentation);
    }

    private List<ResourceSegmentationModel> toResourceTypeSegmentations(DataQueryResult result) {
        if (result.isEmpty()) {
            return List.of();
        }
        if (result.getResultSetCount() > 1) {
            throw new IllegalStateException("Too many result sets");
        }
        ResultSetReader reader = result.getResultSet(0);
        List<ResourceSegmentationModel> resourceSegmentations = new ArrayList<>();
        Map<String, TenantId> tenantIdCache = new HashMap<>();
        while (reader.next()) {
            resourceSegmentations.add(readOneResourceSegmentation(reader, tenantIdCache));
        }
        return resourceSegmentations;
    }

    private ResourceSegmentationModel readOneResourceSegmentation(ResultSetReader reader,
                                                                  Map<String, TenantId> tenantIdCache) {
        String id = reader.getColumn("id").getUtf8();
        String tenantIdString = reader.getColumn("tenant_id").getUtf8();
        TenantId tenantId = tenantIdCache.computeIfAbsent(tenantIdString, TenantId::new);
        String providerId = reader.getColumn("provider_id").getUtf8();
        long version = reader.getColumn("version").getInt64();
        String key = reader.getColumn("key").getUtf8();
        String nameEn = reader.getColumn("name_en").getUtf8();
        String nameRu = reader.getColumn("name_ru").getUtf8();
        String descriptionEn = reader.getColumn("description_en").getUtf8();
        String descriptionRu = reader.getColumn("description_ru").getUtf8();
        boolean deleted = reader.getColumn("deleted").getBool();
        int groupingOrder = Ydb.int32OrDefault(reader.getColumn("grouping_order"), 0);
        SegmentationUISettings uiSettings = uiSettingsHelper.read(reader.getColumn("ui_settings"));
        return new ResourceSegmentationModel(id, tenantId, providerId, version, key, nameEn, nameRu, descriptionEn,
                descriptionRu, deleted, groupingOrder, uiSettings);
    }

    private Map<String, Value> prepareResourceSegmentationFields(ResourceSegmentationModel resourceSegmentation) {
        Map<String, Value> fields = new HashMap<>();
        fields.put("id", PrimitiveValue.utf8(resourceSegmentation.getId()));
        fields.put("tenant_id", PrimitiveValue.utf8(resourceSegmentation.getTenantId().getId()));
        fields.put("provider_id", PrimitiveValue.utf8(resourceSegmentation.getProviderId()));
        fields.put("version", PrimitiveValue.int64(resourceSegmentation.getVersion()));
        fields.put("key", PrimitiveValue.utf8(resourceSegmentation.getKey()));
        fields.put("name_en", PrimitiveValue.utf8(resourceSegmentation.getNameEn()));
        fields.put("name_ru", PrimitiveValue.utf8(resourceSegmentation.getNameRu()));
        fields.put("description_en", PrimitiveValue.utf8(resourceSegmentation.getDescriptionEn()));
        fields.put("description_ru", PrimitiveValue.utf8(resourceSegmentation.getDescriptionRu()));
        fields.put("deleted", PrimitiveValue.bool(resourceSegmentation.isDeleted()));
        fields.put("grouping_order", PrimitiveValue.int32(resourceSegmentation.getGroupingOrder()));
        fields.put("ui_settings",
                uiSettingsHelper.writeOptional(resourceSegmentation.getUiSettings().orElse(null))
        );
        return fields;
    }

}
