package ru.yandex.intranet.d.dao.resources.segments;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import com.yandex.ydb.table.query.DataQueryResult;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.ListValue;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.StructValue;
import com.yandex.ydb.table.values.TupleValue;
import com.yandex.ydb.table.values.Value;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.datasource.Ydb;
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;

/**
 * Resource segments DAO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class ResourceSegmentsDao {

    private final YdbQuerySource ydbQuerySource;

    public ResourceSegmentsDao(YdbQuerySource ydbQuerySource) {
        this.ydbQuerySource = ydbQuerySource;
    }

    public Mono<Optional<ResourceSegmentModel>> getById(YdbTxSession session, String id, TenantId tenantId) {
        String query = ydbQuerySource.getQuery("yql.queries.resourceSegments.getOneById");
        Params params = Params.of("$id", PrimitiveValue.utf8(id),
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        return session.executeDataQueryRetryable(query, params).map(this::toResourceSegment);
    }

    public Mono<Tuple2<Optional<ResourceSegmentModel>, String>> getByIdStartTx(YdbTxSession session, String id,
                                                                               TenantId tenantId) {
        String query = ydbQuerySource.getQuery("yql.queries.resourceSegments.getOneById");
        Params params = Params.of("$id", PrimitiveValue.utf8(id),
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        return session.executeDataQueryRetryable(query, params).map(r -> Tuples.of(toResourceSegment(r), r.getTxId()));
    }

    public Mono<List<ResourceSegmentModel>> getByIds(YdbTxSession session, List<Tuple2<String, TenantId>> ids) {
        if (ids.isEmpty()) {
            return Mono.just(List.of());
        }
        String query = ydbQuerySource.getQuery("yql.queries.resourceSegments.getByIds");
        ListValue idsParam = ListValue.of(ids.stream().map(id -> TupleValue.of(PrimitiveValue.utf8(id.getT1()),
                PrimitiveValue.utf8(id.getT2().getId()))).toArray(TupleValue[]::new));
        Params params = Params.of("$ids", idsParam);
        return session.executeDataQueryRetryable(query, params).map(this::toResourceSegments);
    }

    public Mono<Void> upsertResourceSegmentRetryable(YdbTxSession session, ResourceSegmentModel resourceSegment) {
        String query = ydbQuerySource.getQuery("yql.queries.resourceSegments.upsertOneResourceSegment");
        Map<String, Value> fields = prepareResourceSegmentFields(resourceSegment);
        Params params = Params.of("$resource_segment", StructValue.of(fields));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<Void> upsertResourceSegmentsRetryable(YdbTxSession session,
                                                      List<ResourceSegmentModel> resourceSegments) {
        if (resourceSegments.isEmpty()) {
            return Mono.empty();
        }
        String query = ydbQuerySource.getQuery("yql.queries.resourceSegments.upsertManyResourceSegments");
        Params params = Params.of("$resource_segments", ListValue.of(resourceSegments.stream()
                .map(resourceSegment -> {
                    Map<String, Value> fields = prepareResourceSegmentFields(resourceSegment);
                    return StructValue.of(fields);
                }).toArray(StructValue[]::new)));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<Void> updateResourceSegmentRetryable(YdbTxSession session, ResourceSegmentModel resourceSegment) {
        String query = ydbQuerySource.getQuery("yql.queries.resourceSegments.updateOneResourceSegment");
        Map<String, Value> fields = prepareResourceSegmentFields(resourceSegment);
        Params params = Params.of("$resource_segment", StructValue.of(fields));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<Void> updateResourceSegmentsRetryable(YdbTxSession session,
                                                      List<ResourceSegmentModel> resourceSegments) {
        if (resourceSegments.isEmpty()) {
            return Mono.empty();
        }
        String query = ydbQuerySource.getQuery("yql.queries.resourceSegments.updateManyResourceSegments");
        Params params = Params.of("$resource_segments", ListValue.of(resourceSegments.stream()
                .map(resourceSegment -> {
                    Map<String, Value> fields = prepareResourceSegmentFields(resourceSegment);
                    return StructValue.of(fields);
                }).toArray(StructValue[]::new)));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<List<ResourceSegmentModel>> getBySegmentation(YdbTxSession session, String segmentationId,
                                                              TenantId tenantId, String resourceSegmentIdFrom,
                                                              int limit, boolean withDeleted) {
        String query;
        Params params;
        ListValue deletedParam = withDeleted
                ? ListValue.of(PrimitiveValue.bool(true), PrimitiveValue.bool(false))
                : ListValue.of(PrimitiveValue.bool(false));
        if (resourceSegmentIdFrom != null) {
            query = ydbQuerySource.getQuery("yql.queries.resourceSegments.getBySegmentationNextPage");
            params = Params.of("$from_id", PrimitiveValue.utf8(resourceSegmentIdFrom),
                    "$limit", PrimitiveValue.uint64(limit),
                    "$resource_segmentation_id", PrimitiveValue.utf8(segmentationId),
                    "$deleted", deletedParam,
                    "$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        } else {
            query = ydbQuerySource.getQuery("yql.queries.resourceSegments.getBySegmentationFirstPage");
            params = Params.of("$limit", PrimitiveValue.uint64(limit),
                    "$resource_segmentation_id", PrimitiveValue.utf8(segmentationId),
                    "$deleted", deletedParam,
                    "$tenant_id", PrimitiveValue.utf8(tenantId.getId()));
        }
        return session.executeDataQueryRetryable(query, params).map(this::toResourceSegments);
    }

    public Mono<Boolean> existsBySegmentationAndKey(YdbTxSession session, String segmentationId, TenantId tenantId,
                                                    String key, boolean withDeleted) {
        ListValue deletedParam = withDeleted
                ? ListValue.of(PrimitiveValue.bool(true), PrimitiveValue.bool(false))
                : ListValue.of(PrimitiveValue.bool(false));
        String query = ydbQuerySource.getQuery("yql.queries.resourceSegments.existsBySegmentationAndKey");
        Params params = Params.of("$resource_segmentation_id", PrimitiveValue.utf8(segmentationId),
                "$key", PrimitiveValue.utf8(key),
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$deleted", deletedParam);
        return session.executeDataQueryRetryable(query, params).map(this::toExists);
    }

    public Mono<List<ResourceSegmentModel>> getAllBySegmentationAndKey(
            YdbTxSession session,
            List<Tuple2<ResourceSegmentModel.SegmentationAndKey, TenantId>> keys
    ) {
        if (keys.isEmpty()) {
            return Mono.just(List.of());
        }
        String query = ydbQuerySource.getQuery("yql.queries.resourceSegments.getAllBySegmentationAndKeys");
        ListValue keysParam = ListValue.of(keys.stream().map(key -> TupleValue.of(
                PrimitiveValue.utf8(key.getT2().getId()),
                PrimitiveValue.utf8(key.getT1().getSegmentationId()),
                PrimitiveValue.utf8(key.getT1().getKey())
        )).toArray(TupleValue[]::new));
        Params params = Params.of("$keys", keysParam);
        return session.executeDataQueryRetryable(query, params).map(this::toResourceSegments);
    }

    private boolean toExists(DataQueryResult result) {
        if (result.isEmpty() || result.getResultSetCount() > 1) {
            throw new IllegalStateException("Exactly one result set is required");
        }
        ResultSetReader reader = result.getResultSet(0);
        if (!reader.next() || reader.getRowCount() > 1) {
            throw new IllegalStateException("Exactly one result is required");
        }
        return reader.getColumn("resource_segment_exists").getBool();
    }

    private Optional<ResourceSegmentModel> toResourceSegment(DataQueryResult result) {
        if (result.isEmpty()) {
            return Optional.empty();
        }
        if (result.getResultSetCount() > 1) {
            throw new IllegalStateException("Too many result sets");
        }
        ResultSetReader reader = result.getResultSet(0);
        if (!reader.next()) {
            return Optional.empty();
        }
        if (reader.getRowCount() > 1) {
            throw new IllegalStateException("Non unique resource segment");
        }
        ResourceSegmentModel resourceSegment = readOneResourceSegment(reader, new HashMap<>());
        return Optional.of(resourceSegment);
    }

    private List<ResourceSegmentModel> toResourceSegments(DataQueryResult result) {
        if (result.isEmpty()) {
            return List.of();
        }
        if (result.getResultSetCount() > 1) {
            throw new IllegalStateException("Too many result sets");
        }
        ResultSetReader reader = result.getResultSet(0);
        List<ResourceSegmentModel> resourceSegments = new ArrayList<>();
        Map<String, TenantId> tenantIdCache = new HashMap<>();
        while (reader.next()) {
            resourceSegments.add(readOneResourceSegment(reader, tenantIdCache));
        }
        return resourceSegments;
    }

    private ResourceSegmentModel readOneResourceSegment(ResultSetReader reader, Map<String, TenantId> tenantIdCache) {
        String id = reader.getColumn("id").getUtf8();
        String tenantIdString = reader.getColumn("tenant_id").getUtf8();
        TenantId tenantId = tenantIdCache.computeIfAbsent(tenantIdString, TenantId::new);
        String segmentationId = reader.getColumn("resource_segmentation_id").getUtf8();
        long version = reader.getColumn("version").getInt64();
        String key = reader.getColumn("key").getUtf8();
        String nameEn = reader.getColumn("name_en").getUtf8();
        String nameRu = reader.getColumn("name_ru").getUtf8();
        String descriptionEn = reader.getColumn("description_en").getUtf8();
        String descriptionRu = reader.getColumn("description_ru").getUtf8();
        boolean deleted = reader.getColumn("deleted").getBool();
        Boolean uncommon = Ydb.boolOrNull(reader.getColumn("uncommon"));
        return new ResourceSegmentModel(id, tenantId, segmentationId, version, key, nameEn, nameRu, descriptionEn,
                descriptionRu, deleted, uncommon);
    }

    private Map<String, Value> prepareResourceSegmentFields(ResourceSegmentModel resourceSegment) {
        Map<String, Value> fields = new HashMap<>();
        fields.put("id", PrimitiveValue.utf8(resourceSegment.getId()));
        fields.put("tenant_id", PrimitiveValue.utf8(resourceSegment.getTenantId().getId()));
        fields.put("resource_segmentation_id", PrimitiveValue.utf8(resourceSegment.getSegmentationId()));
        fields.put("version", PrimitiveValue.int64(resourceSegment.getVersion()));
        fields.put("key", PrimitiveValue.utf8(resourceSegment.getKey()));
        fields.put("name_en", PrimitiveValue.utf8(resourceSegment.getNameEn()));
        fields.put("name_ru", PrimitiveValue.utf8(resourceSegment.getNameRu()));
        fields.put("description_en", PrimitiveValue.utf8(resourceSegment.getDescriptionEn()));
        fields.put("description_ru", PrimitiveValue.utf8(resourceSegment.getDescriptionRu()));
        fields.put("deleted", PrimitiveValue.bool(resourceSegment.isDeleted()));
        fields.put("uncommon", Ydb.nullableBool(resourceSegment.getUncommon().orElse(null)));
        return fields;
    }

}
