package ru.yandex.intranet.d.dao.resources.types

import com.fasterxml.jackson.core.type.TypeReference
import com.yandex.ydb.table.query.Params
import com.yandex.ydb.table.result.ResultSetReader
import com.yandex.ydb.table.values.ListValue
import com.yandex.ydb.table.values.PrimitiveValue
import com.yandex.ydb.table.values.StructValue
import kotlinx.coroutines.reactor.awaitSingle
import kotlinx.coroutines.reactor.awaitSingleOrNull
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.stereotype.Component
import ru.yandex.intranet.d.dao.DaoReader
import ru.yandex.intranet.d.dao.JsonFieldHelper
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource
import ru.yandex.intranet.d.datasource.model.YdbTxSession
import ru.yandex.intranet.d.kotlin.ProviderId
import ru.yandex.intranet.d.model.TenantId
import ru.yandex.intranet.d.model.resources.types.ExchangeableResourceTypeKey
import ru.yandex.intranet.d.model.resources.types.ExchangeableResourceTypeModel
import ru.yandex.intranet.d.model.resources.types.ExchangeableResourceTypeSegments
import ru.yandex.intranet.d.util.ObjectMapperHolder

@Component
class ExchangeableResourceTypesDao(
    private val ydbQuerySource: YdbQuerySource,
    @Qualifier("ydbJsonObjectMapper") private val objectMapper: ObjectMapperHolder
) {
    private val segmentsJsonFieldHelper: JsonFieldHelper<ExchangeableResourceTypeSegments> =
        JsonFieldHelper(objectMapper, object : TypeReference<ExchangeableResourceTypeSegments>() {})

    suspend fun getById(session: YdbTxSession, id: ExchangeableResourceTypeKey): ExchangeableResourceTypeModel? {
        val query = ydbQuerySource.getQuery("yql.queries.exchangeable_resource_types.getById")
        val params = toIdParams(id)
        return DaoReader.toModel(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    suspend fun getByIds(
        session: YdbTxSession, ids: Collection<ExchangeableResourceTypeKey>
    ): List<ExchangeableResourceTypeModel> {
        if (ids.isEmpty()) {
            return listOf()
        }
        return ids.distinct().chunked(1000).map {
            val query = ydbQuerySource.getQuery("yql.queries.exchangeable_resource_types.getByIds")
            val params = toIdListParams(it)
            DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
        }.flatten()
    }

    suspend fun getProviders(session: YdbTxSession): List<ProviderId> {
        val query = ydbQuerySource.getQuery("yql.queries.exchangeable_resource_types.getProviders")
        return DaoReader.toModels(
            session.executeDataQueryRetryable(query, Params.empty()).awaitSingle(), this::toProviderId
        )
    }

    suspend fun upsertOneRetryable(session: YdbTxSession, model: ExchangeableResourceTypeModel) {
        val query = ydbQuerySource.getQuery("yql.queries.exchangeable_resource_types.upsertOne")
        val params = toUpsertOneParams(model)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun upsertManyRetryable(session: YdbTxSession, models: Collection<ExchangeableResourceTypeModel>) {
        if (models.isEmpty()) {
            return
        }
        val query = ydbQuerySource.getQuery("yql.queries.exchangeable_resource_types.upsertMany")
        val params = toUpsertManyParams(models)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    suspend fun getByProvider(
        session: YdbTxSession, tenantId: TenantId, providerId: ProviderId
    ): List<ExchangeableResourceTypeModel> {
        val query = ydbQuerySource.getQuery("yql.queries.exchangeable_resource_types.getByProvider")
        val params = toGetByProviderParams(tenantId, providerId)
        return DaoReader.toModels(session.executeDataQueryRetryable(query, params).awaitSingle(), this::toModel)
    }

    private fun toIdParams(id: ExchangeableResourceTypeKey): Params = Params.of("\$id", toIdStruct(id))

    private fun toUpsertOneParams(model: ExchangeableResourceTypeModel) = Params.of(
        "\$value", toUpsertStruct(model)
    )

    private fun toUpsertManyParams(values: Collection<ExchangeableResourceTypeModel>) = Params.of(
        "\$values", ListValue.of(*values.map { toUpsertStruct(it) }.toTypedArray())
    )

    private fun toGetByProviderParams(tenantId: TenantId, providerId: ProviderId) = Params.of(
        "\$tenant_id", PrimitiveValue.utf8(tenantId.id),
        "\$provider_id", PrimitiveValue.utf8(providerId)
    )

    private fun toIdStruct(id: ExchangeableResourceTypeKey) = StructValue.of(
        mapOf(
            "tenant_id" to PrimitiveValue.utf8(id.tenantId.id),
            "provider_id" to PrimitiveValue.utf8(id.providerId),
            "to_resource_type_id" to PrimitiveValue.utf8(id.toResourceTypeId),
            "from_resource_type_id" to PrimitiveValue.utf8(id.fromResourceTypeId)
        )
    )

    private fun toIdListParams(ids: Collection<ExchangeableResourceTypeKey>) = Params.of(
        "\$ids", ListValue.of(*ids.map { toIdStruct(it) }.toTypedArray())
    )

    private fun toUpsertStruct(
        model: ExchangeableResourceTypeModel
    ) = StructValue.of(
        mapOf(
            "tenant_id" to PrimitiveValue.utf8(model.key.tenantId.id),
            "provider_id" to PrimitiveValue.utf8(model.key.providerId),
            "to_resource_type_id" to PrimitiveValue.utf8(model.key.toResourceTypeId),
            "from_resource_type_id" to PrimitiveValue.utf8(model.key.fromResourceTypeId),
            "numerator" to PrimitiveValue.int64(model.numerator),
            "denominator" to PrimitiveValue.int64(model.denominator),
            "available_segments" to segmentsJsonFieldHelper.write(model.available_segments)
        )
    )

    private fun toModel(reader: ResultSetReader): ExchangeableResourceTypeModel = ExchangeableResourceTypeModel(
        key = ExchangeableResourceTypeKey(
            tenantId = TenantId(reader.getColumn("tenant_id").utf8),
            providerId = reader.getColumn("provider_id").utf8,
            toResourceTypeId = reader.getColumn("to_resource_type_id").utf8,
            fromResourceTypeId = reader.getColumn("from_resource_type_id").utf8
        ),
        numerator = reader.getColumn("numerator").int64,
        denominator = reader.getColumn("denominator").int64,
        available_segments = segmentsJsonFieldHelper.read(reader.getColumn("available_segments"))!!
    )

    private fun toProviderId(reader: ResultSetReader): ProviderId = reader.getColumn("provider_id").utf8
}
