package ru.yandex.intranet.d.dao.resources.types;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import com.fasterxml.jackson.core.type.TypeReference;
import com.yandex.ydb.table.query.DataQueryResult;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.ListValue;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.StructValue;
import com.yandex.ydb.table.values.TupleValue;
import com.yandex.ydb.table.values.Value;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.JsonFieldHelper;
import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.providers.AggregationSettings;
import ru.yandex.intranet.d.model.resources.types.ResourceTypeModel;
import ru.yandex.intranet.d.util.ObjectMapperHolder;

import static com.yandex.ydb.table.values.PrimitiveValue.int64;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static ru.yandex.intranet.d.datasource.Ydb.int64OrNull;
import static ru.yandex.intranet.d.datasource.Ydb.nullableInt64;
import static ru.yandex.intranet.d.datasource.Ydb.nullableUtf8;
import static ru.yandex.intranet.d.datasource.Ydb.utf8OrNull;

/**
 * Resource types DAO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class ResourceTypesDao {

    private final YdbQuerySource ydbQuerySource;
    private final JsonFieldHelper<AggregationSettings> aggregationSettingsHelper;

    public ResourceTypesDao(YdbQuerySource ydbQuerySource,
                            @Qualifier("ydbJsonObjectMapper") ObjectMapperHolder objectMapper) {
        this.ydbQuerySource = ydbQuerySource;
        this.aggregationSettingsHelper = new JsonFieldHelper<>(objectMapper,
                new TypeReference<AggregationSettings>() { });
    }

    public Mono<Optional<ResourceTypeModel>> getById(YdbTxSession session, String id, TenantId tenantId) {
        String query = ydbQuerySource.getQuery("yql.queries.resourceTypes.getOneById");
        Params params = Params.of("$id", utf8(id),
                "$tenant_id", utf8(tenantId.getId()));
        return session.executeDataQueryRetryable(query, params).map(this::toResourceType);
    }

    public Mono<Tuple2<Optional<ResourceTypeModel>, String>> getByIdStartTx(YdbTxSession session, String id,
                                                                            TenantId tenantId) {
        String query = ydbQuerySource.getQuery("yql.queries.resourceTypes.getOneById");
        Params params = Params.of("$id", utf8(id),
                "$tenant_id", utf8(tenantId.getId()));
        return session.executeDataQueryRetryable(query, params).map(r -> Tuples.of(toResourceType(r), r.getTxId()));
    }

    public Mono<List<ResourceTypeModel>> getByIds(YdbTxSession session, List<Tuple2<String, TenantId>> ids) {
        if (ids.isEmpty()) {
            return Mono.just(List.of());
        }
        String query = ydbQuerySource.getQuery("yql.queries.resourceTypes.getByIds");
        ListValue idsParam = ListValue.of(ids.stream().map(id -> TupleValue.of(utf8(id.getT1()),
                utf8(id.getT2().getId()))).toArray(TupleValue[]::new));
        Params params = Params.of("$ids", idsParam);
        return session.executeDataQueryRetryable(query, params).map(this::toResourceTypes);
    }

    public Mono<Optional<ResourceTypeModel>> getByProviderAndKey(YdbTxSession session, String providerId, String key,
                                                                 TenantId tenantId) {
        String query = ydbQuerySource.getQuery("yql.queries.resourceTypes.getByProviderAndKey");
        Params params = Params.of("$provider_id", utf8(providerId),
                "$key", utf8(key),
                "$tenant_id", utf8(tenantId.getId()));
        return session.executeDataQueryRetryable(query, params).map(this::toResourceType);
    }

    public Mono<List<ResourceTypeModel>> getByProviderAndKeys(YdbTxSession session,
                                                              List<Tuple2<Tuple2<String, String>, TenantId>> ids) {
        if (ids.isEmpty()) {
            return Mono.just(List.of());
        }
        String query = ydbQuerySource.getQuery("yql.queries.resourceTypes.getByProviderAndKeys");
        ListValue idsParam = ListValue.of(ids.stream()
                .map(id -> TupleValue.of(utf8(id.getT1().getT1()),
                        utf8(id.getT1().getT2()),
                        utf8(id.getT2().getId())))
                .toArray(TupleValue[]::new));
        Params params = Params.of("$ids", idsParam);
        return session.executeDataQueryRetryable(query, params).map(this::toResourceTypes);
    }

    public Mono<Void> upsertResourceTypeRetryable(YdbTxSession session, ResourceTypeModel resourceType) {
        String query = ydbQuerySource.getQuery("yql.queries.resourceTypes.upsertOneResourceType");
        Map<String, Value> fields = prepareResourceTypeFields(resourceType);
        Params params = Params.of("$resource_type", StructValue.of(fields));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<Void> upsertResourceTypesRetryable(YdbTxSession session, List<ResourceTypeModel> resourceTypes) {
        if (resourceTypes.isEmpty()) {
            return Mono.empty();
        }
        String query = ydbQuerySource.getQuery("yql.queries.resourceTypes.upsertManyResourceTypes");
        Params params = Params.of("$resource_types", ListValue.of(resourceTypes.stream().map(resourceType -> {
            Map<String, Value> fields = prepareResourceTypeFields(resourceType);
            return StructValue.of(fields);
        }).toArray(StructValue[]::new)));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<Void> updateResourceTypeRetryable(YdbTxSession session, ResourceTypeModel resourceType) {
        String query = ydbQuerySource.getQuery("yql.queries.resourceTypes.updateOneResourceType");
        Map<String, Value> fields = prepareResourceTypeFields(resourceType);
        Params params = Params.of("$resource_type", StructValue.of(fields));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<Void> updateResourceTypesRetryable(YdbTxSession session, List<ResourceTypeModel> resourceTypes) {
        if (resourceTypes.isEmpty()) {
            return Mono.empty();
        }
        String query = ydbQuerySource.getQuery("yql.queries.resourceTypes.updateManyResourceTypes");
        Params params = Params.of("$resource_types", ListValue.of(resourceTypes.stream().map(resourceType -> {
            Map<String, Value> fields = prepareResourceTypeFields(resourceType);
            return StructValue.of(fields);
        }).toArray(StructValue[]::new)));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<List<ResourceTypeModel>> getByProvider(YdbTxSession session, String providerId, TenantId tenantId,
                                                       String resourceTypeIdFrom, int limit, boolean withDeleted) {
        String query;
        Params params;
        ListValue deletedParam = withDeleted
                ? ListValue.of(PrimitiveValue.bool(true), PrimitiveValue.bool(false))
                : ListValue.of(PrimitiveValue.bool(false));
        if (resourceTypeIdFrom != null) {
            query = ydbQuerySource.getQuery("yql.queries.resourceTypes.getByProviderNextPage");
            params = Params.of("$from_id", utf8(resourceTypeIdFrom),
                    "$limit", PrimitiveValue.uint64(limit),
                    "$provider_id", utf8(providerId),
                    "$deleted", deletedParam,
                    "$tenant_id", utf8(tenantId.getId()));
        } else {
            query = ydbQuerySource.getQuery("yql.queries.resourceTypes.getByProviderFirstPage");
            params = Params.of("$limit", PrimitiveValue.uint64(limit),
                    "$provider_id", utf8(providerId),
                    "$deleted", deletedParam,
                    "$tenant_id", utf8(tenantId.getId()));
        }
        return session.executeDataQueryRetryable(query, params).map(this::toResourceTypes);
    }

    public Mono<Boolean> existsByProviderAndKey(YdbTxSession session, String providerId, TenantId tenantId,
                                                String key, boolean withDeleted) {
        ListValue deletedParam = withDeleted
                ? ListValue.of(PrimitiveValue.bool(true), PrimitiveValue.bool(false))
                : ListValue.of(PrimitiveValue.bool(false));
        String query = ydbQuerySource.getQuery("yql.queries.resourceTypes.existsByProviderAndKey");
        Params params = Params.of("$provider_id", utf8(providerId),
                "$key", utf8(key),
                "$tenant_id", utf8(tenantId.getId()),
                "$deleted", deletedParam);
        return session.executeDataQueryRetryable(query, params).map(this::toExists);
    }

    private boolean toExists(DataQueryResult result) {
        if (result.isEmpty() || result.getResultSetCount() > 1) {
            throw new IllegalStateException("Exactly one result set is required");
        }
        ResultSetReader reader = result.getResultSet(0);
        if (!reader.next() || reader.getRowCount() > 1) {
            throw new IllegalStateException("Exactly one result is required");
        }
        return reader.getColumn("resource_type_exists").getBool();
    }

    private Optional<ResourceTypeModel> toResourceType(DataQueryResult result) {
        if (result.isEmpty()) {
            return Optional.empty();
        }
        if (result.getResultSetCount() > 1) {
            throw new IllegalStateException("Too many result sets");
        }
        ResultSetReader reader = result.getResultSet(0);
        if (!reader.next()) {
            return Optional.empty();
        }
        if (reader.getRowCount() > 1) {
            throw new IllegalStateException("Non unique resource type");
        }
        ResourceTypeModel resourceType = readOneResourceType(reader, new HashMap<>());
        return Optional.of(resourceType);
    }

    private List<ResourceTypeModel> toResourceTypes(DataQueryResult result) {
        if (result.isEmpty()) {
            return List.of();
        }
        if (result.getResultSetCount() > 1) {
            throw new IllegalStateException("Too many result sets");
        }
        ResultSetReader reader = result.getResultSet(0);
        List<ResourceTypeModel> resourceTypes = new ArrayList<>();
        Map<String, TenantId> tenantIdCache = new HashMap<>();
        while (reader.next()) {
            resourceTypes.add(readOneResourceType(reader, tenantIdCache));
        }
        return resourceTypes;
    }

    private ResourceTypeModel readOneResourceType(ResultSetReader reader, Map<String, TenantId> tenantIdCache) {
        ResourceTypeModel.Builder builder = ResourceTypeModel.builder();
        for (Fields field : Fields.values()) {
            field.read(reader, builder, aggregationSettingsHelper);
        }
        return builder.build();
    }

    @SuppressWarnings("rawtypes")
    private Map<String, Value> prepareResourceTypeFields(ResourceTypeModel resourceType) {
        Map<String, Value> fields = new HashMap<>();
        for (Fields field : Fields.values()) {
            field.write(resourceType, fields, aggregationSettingsHelper);
        }
        return fields;
    }

    @SuppressWarnings({"unused", "RedundantSuppression", "rawtypes"})
    public enum Fields {
        ID {
            @Override
            public void read(ResultSetReader reader, ResourceTypeModel.Builder builder,
                             JsonFieldHelper<AggregationSettings> aggregationSettingsHelper) {
                builder.id(reader.getColumn(field()).getUtf8());
            }

            @Override
            public void write(ResourceTypeModel resourceType, Map<String, Value> fields,
                              JsonFieldHelper<AggregationSettings> aggregationSettingsHelper) {
                fields.put(field(), utf8(resourceType.getId()));
            }
        },
        TENANT_ID {
            @Override
            public void read(ResultSetReader reader, ResourceTypeModel.Builder builder,
                             JsonFieldHelper<AggregationSettings> aggregationSettingsHelper) {
                builder.tenantId(Tenants.getInstance(reader.getColumn(field()).getUtf8()));
            }

            @Override
            public void write(ResourceTypeModel resourceType, Map<String, Value> fields,
                              JsonFieldHelper<AggregationSettings> aggregationSettingsHelper) {
                fields.put(field(), utf8(resourceType.getTenantId().getId()));
            }
        },
        PROVIDER_ID {
            @Override
            public void read(ResultSetReader reader, ResourceTypeModel.Builder builder,
                             JsonFieldHelper<AggregationSettings> aggregationSettingsHelper) {
                builder.providerId(reader.getColumn(field()).getUtf8());
            }

            @Override
            public void write(ResourceTypeModel resourceType, Map<String, Value> fields,
                              JsonFieldHelper<AggregationSettings> aggregationSettingsHelper) {
                fields.put(field(), utf8(resourceType.getProviderId()));
            }
        },
        VERSION {
            @Override
            public void read(ResultSetReader reader, ResourceTypeModel.Builder builder,
                             JsonFieldHelper<AggregationSettings> aggregationSettingsHelper) {
                builder.version(reader.getColumn(field()).getInt64());
            }

            @Override
            public void write(ResourceTypeModel resourceType, Map<String, Value> fields,
                              JsonFieldHelper<AggregationSettings> aggregationSettingsHelper) {
                fields.put(field(), int64(resourceType.getVersion()));
            }
        },
        KEY {
            @Override
            public void read(ResultSetReader reader, ResourceTypeModel.Builder builder,
                             JsonFieldHelper<AggregationSettings> aggregationSettingsHelper) {
                builder.key(reader.getColumn(field()).getUtf8());
            }

            @Override
            public void write(ResourceTypeModel resourceType, Map<String, Value> fields,
                              JsonFieldHelper<AggregationSettings> aggregationSettingsHelper) {
                fields.put(field(), utf8(resourceType.getKey()));
            }
        },
        NAME_EN {
            @Override
            public void read(ResultSetReader reader, ResourceTypeModel.Builder builder,
                             JsonFieldHelper<AggregationSettings> aggregationSettingsHelper) {
                builder.nameEn(reader.getColumn(field()).getUtf8());
            }

            @Override
            public void write(ResourceTypeModel resourceType, Map<String, Value> fields,
                              JsonFieldHelper<AggregationSettings> aggregationSettingsHelper) {
                fields.put(field(), utf8(resourceType.getNameEn()));
            }
        },
        NAME_RU {
            @Override
            public void read(ResultSetReader reader, ResourceTypeModel.Builder builder,
                             JsonFieldHelper<AggregationSettings> aggregationSettingsHelper) {
                builder.nameRu(reader.getColumn(field()).getUtf8());
            }

            @Override
            public void write(ResourceTypeModel resourceType, Map<String, Value> fields,
                              JsonFieldHelper<AggregationSettings> aggregationSettingsHelper) {
                fields.put(field(), utf8(resourceType.getNameRu()));
            }
        },
        DESCRIPTION_EN {
            @Override
            public void read(ResultSetReader reader, ResourceTypeModel.Builder builder,
                             JsonFieldHelper<AggregationSettings> aggregationSettingsHelper) {
                builder.descriptionEn(reader.getColumn(field()).getUtf8());
            }

            @Override
            public void write(ResourceTypeModel resourceType, Map<String, Value> fields,
                              JsonFieldHelper<AggregationSettings> aggregationSettingsHelper) {
                fields.put(field(), utf8(resourceType.getDescriptionEn()));
            }
        },
        DESCRIPTION_RU {
            @Override
            public void read(ResultSetReader reader, ResourceTypeModel.Builder builder,
                             JsonFieldHelper<AggregationSettings> aggregationSettingsHelper) {
                builder.descriptionRu(reader.getColumn(field()).getUtf8());
            }

            @Override
            public void write(ResourceTypeModel resourceType, Map<String, Value> fields,
                              JsonFieldHelper<AggregationSettings> aggregationSettingsHelper) {
                fields.put(field(), utf8(resourceType.getDescriptionRu()));
            }
        },
        DELETED {
            @Override
            public void read(ResultSetReader reader, ResourceTypeModel.Builder builder,
                             JsonFieldHelper<AggregationSettings> aggregationSettingsHelper) {
                builder.deleted(reader.getColumn(field()).getBool());
            }

            @Override
            public void write(ResourceTypeModel resourceType, Map<String, Value> fields,
                              JsonFieldHelper<AggregationSettings> aggregationSettingsHelper) {
                fields.put(field(), PrimitiveValue.bool(resourceType.isDeleted()));
            }
        },
        UNITS_ENSEMBLE_ID {
            @Override
            public void read(ResultSetReader reader, ResourceTypeModel.Builder builder,
                             JsonFieldHelper<AggregationSettings> aggregationSettingsHelper) {
                builder.unitsEnsembleId(utf8OrNull(reader.getColumn(field())));
            }

            @Override
            public void write(ResourceTypeModel resourceType, Map<String, Value> fields,
                              JsonFieldHelper<AggregationSettings> aggregationSettingsHelper) {
                fields.put(field(), nullableUtf8(resourceType.getUnitsEnsembleId()));
            }
        },
        BASE_UNIT_ID {
            @Override
            public void read(ResultSetReader reader, ResourceTypeModel.Builder builder,
                             JsonFieldHelper<AggregationSettings> aggregationSettingsHelper) {
                builder.baseUnitId(utf8OrNull(reader.getColumn(field())));
            }

            @Override
            public void write(ResourceTypeModel resourceType, Map<String, Value> fields,
                              JsonFieldHelper<AggregationSettings> aggregationSettingsHelper) {
                fields.put(field(), nullableUtf8(resourceType.getBaseUnitId()));
            }
        },
        SORTING_ORDER {
            @Override
            public void read(ResultSetReader reader, ResourceTypeModel.Builder builder,
                             JsonFieldHelper<AggregationSettings> aggregationSettingsHelper) {
                builder.sortingOrder(int64OrNull(reader.getColumn(field())));
            }

            @Override
            public void write(ResourceTypeModel resourceType, Map<String, Value> fields,
                              JsonFieldHelper<AggregationSettings> aggregationSettingsHelper) {
                fields.put(field(), nullableInt64(resourceType.getSortingOrder()));
            }
        },
        AGGREGATION_SETTINGS {
            @Override
            public void read(ResultSetReader reader, ResourceTypeModel.Builder builder,
                             JsonFieldHelper<AggregationSettings> aggregationSettingsHelper) {
                builder.aggregationSettings(aggregationSettingsHelper.read(reader.getColumn(field())));
            }

            @Override
            public void write(ResourceTypeModel resourceType, Map<String, Value> fields,
                              JsonFieldHelper<AggregationSettings> aggregationSettingsHelper) {
                fields.put(field(), aggregationSettingsHelper.writeOptional(resourceType.getAggregationSettings()
                        .orElse(null)));
            }
        };

        private final String fieldName;

        Fields() {
            fieldName = name().toLowerCase();
        }

        public String field() {
            return fieldName;
        }

        public abstract void read(ResultSetReader reader, ResourceTypeModel.Builder builder,
                                  JsonFieldHelper<AggregationSettings> aggregationSettingsHelper);

        public abstract void write(ResourceTypeModel resourceType, Map<String, Value> fields,
                                   JsonFieldHelper<AggregationSettings> aggregationSettingsHelper);
    }
}
