package ru.yandex.intranet.d.dao.services;

import java.io.UncheckedIOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.yandex.ydb.table.query.DataQueryResult;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.result.ValueReader;
import com.yandex.ydb.table.values.ListValue;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.StructValue;
import it.unimi.dsi.fastutil.longs.LongOpenHashSet;
import it.unimi.dsi.fastutil.longs.LongSet;
import it.unimi.dsi.fastutil.longs.LongSets;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.datasource.Ydb;
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource;
import ru.yandex.intranet.d.datasource.model.WithTxId;
import ru.yandex.intranet.d.datasource.model.YdbReadTableSettings;
import ru.yandex.intranet.d.datasource.model.YdbSession;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.services.ServiceIdAndParentId;
import ru.yandex.intranet.d.model.services.ServiceMinimalModel;
import ru.yandex.intranet.d.model.services.ServiceModel;
import ru.yandex.intranet.d.model.services.ServiceNode;
import ru.yandex.intranet.d.model.services.ServiceReadOnlyState;
import ru.yandex.intranet.d.model.services.ServiceRecipeModel;
import ru.yandex.intranet.d.model.services.ServiceSlugNode;
import ru.yandex.intranet.d.model.services.ServiceSlugWithParent;
import ru.yandex.intranet.d.model.services.ServiceState;
import ru.yandex.intranet.d.model.services.ServiceWithStatesModel;
import ru.yandex.intranet.d.util.Long2LongMultimap;
import ru.yandex.intranet.d.util.ObjectMapperHolder;

/**
 * Services DAO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class ServicesDao {

    private final YdbQuerySource ydbQuerySource;
    private final ObjectReader slugReader;
    private final ObjectWriter slugWriter;
    private final ObjectReader idsReader;
    private final ObjectWriter idsWriter;

    public ServicesDao(
            YdbQuerySource ydbQuerySource,
            @Qualifier("ydbJsonObjectMapper") ObjectMapperHolder objectMapper
    ) {
        this.ydbQuerySource = ydbQuerySource;
        ObjectMapper mapper = objectMapper.getObjectMapper();
        this.slugReader = mapper.readerFor(String.class);
        this.slugWriter = mapper.writerFor(String.class);
        TypeReference<Set<Long>> idsTypeReference = new TypeReference<>() {
        };
        this.idsReader = mapper.readerFor(idsTypeReference);
        this.idsWriter = mapper.writerFor(idsTypeReference);
    }

    public Mono<WithTxId<Optional<ServiceMinimalModel>>> getByIdMinimal(YdbTxSession session, long id) {
        String query = ydbQuerySource.getAbcQuery("yql.queries.services.getOneByIdMinimal");
        Params params = Params.of("$id", PrimitiveValue.int64(id));
        return session.executeDataQueryRetryable(query, params)
                .map(result -> new WithTxId<>(toMinimalService(result), result.getTxId()));
    }

    public Mono<Optional<ServiceMinimalModel>> getBySlugMinimal(YdbTxSession session, String slug) {
        String query = ydbQuerySource.getAbcQuery("yql.queries.services.getOneBySlugMinimal");
        Params params = Params.of("$slug", PrimitiveValue.utf8(slug));
        return session.executeDataQueryRetryable(query, params)
                .map(this::toMinimalService);
    }

    public Mono<Optional<ServiceModel>> getBySlug(YdbTxSession session, String slug) {
        String query = ydbQuerySource.getAbcQuery("yql.queries.services.getOneBySlug");
        Params params = Params.of("$slug", PrimitiveValue.utf8(slug));
        return session.executeDataQueryRetryable(query, params)
                .map(this::toService);
    }

    public Mono<List<ServiceMinimalModel>> getByIdsMinimal(YdbTxSession session, List<Long> ids) {
        String query = ydbQuerySource.getAbcQuery("yql.queries.services.getByIdsMinimal");
        ListValue idsParam = ListValue.of(ids.stream().map(PrimitiveValue::int64).toArray(PrimitiveValue[]::new));
        Params params = Params.of("$ids", idsParam);
        return session.executeDataQueryRetryable(query, params).map(this::toMinimalServices);
    }

    public Flux<Long> getAllServiceIds(YdbSession session) {
        return session.readTable(ydbQuerySource.preprocessAbcTableName("public_services_service"),
                YdbReadTableSettings.builder().ordered(true).addColumn("id").build())
                .flatMapIterable(reader -> {
                    List<Long> page = new ArrayList<>();
                    while (reader.next()) {
                        page.add(reader.getColumn("id").getInt64());
                    }
                    return page;
                });
    }

    public Flux<ServiceIdAndParentId> getAllServiceIdsWithParents(YdbSession session) {
        return session.readTable(
                ydbQuerySource.preprocessAbcTableName("public_services_service"),
                YdbReadTableSettings.builder().ordered(true).addColumns("id", "parent_id").build()
        )
                .flatMapIterable(reader -> {
                    List<ServiceIdAndParentId> page = new ArrayList<>();
                    while (reader.next()) {
                        page.add(new ServiceIdAndParentId(
                                reader.getColumn("id").getInt64(),
                                reader.getColumn("parent_id").getInt64()
                        ));
                    }
                    return page;
                });
    }

    public Mono<LongSet> getAllParents(YdbTxSession session, long serviceId, TenantId tenantId) {
        String query = ydbQuerySource.getQuery("yql.queries.services.getAllParents");
        Params params = Params.of(
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$service_id", PrimitiveValue.int64(serviceId)
        );
        return session.executeDataQueryRetryable(query, params).map(this::readIds);
    }

    public Mono<Map<Long, LongSet>> getAllParentsForServices(YdbTxSession session, List<Long> serviceIds,
                                                               TenantId tenantId) {
        String query = ydbQuerySource.getQuery("yql.queries.services.getAllParentsForServices");
        ListValue idsParam = ListValue.of(serviceIds.stream().map(PrimitiveValue::int64)
                .toArray(PrimitiveValue[]::new));
        Params params = Params.of(
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$service_ids", idsParam);
        return session.executeDataQueryRetryable(query, params).map(this::readMultiIds);
    }

    public Mono<Void> upsertAllParentsRetryable(YdbTxSession session, Long2LongMultimap data, TenantId tenantId) {
        String query = ydbQuerySource.getQuery("yql.queries.services.upsertAllParents");
        Params params = Params.of("$dataList", toAllParentsListValue(data, tenantId));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<Void> upsertRecipeRetryable(YdbTxSession session, ServiceRecipeModel service) {
        String query = ydbQuerySource.getAbcQuery("yql.queries.services.upsertRecipe");
        Params params = Params.of("$service", StructValue.of(Map.of(
                "id", PrimitiveValue.int64(service.getId()),
                "name", PrimitiveValue.utf8(service.getName()),
                "name_en", PrimitiveValue.utf8(service.getNameEn()),
                "slug", PrimitiveValue.json(writeSlug(service.getSlug())),
                "state", PrimitiveValue.utf8(service.getState().toString()),
                "readonly_state", Ydb.nullableUtf8(Optional.ofNullable(service.getReadOnlyState())
                        .map(ServiceReadOnlyState::toString).orElse(null)),
                "is_exportable", PrimitiveValue.bool(service.isExportable()),
                "parent_id", Ydb.nullableInt64(service.getParentId().orElse(null))
        )));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<Void> upsertRecipeManyRetryable(YdbTxSession session, List<ServiceRecipeModel> services) {
        String query = ydbQuerySource.getAbcQuery("yql.queries.services.upsertRecipeMany");
        Params params = Params.of("$services", ListValue.of(services.stream().map(service ->
                StructValue.of(Map.of(
                        "id", PrimitiveValue.int64(service.getId()),
                        "name", PrimitiveValue.utf8(service.getName()),
                        "name_en", PrimitiveValue.utf8(service.getNameEn()),
                        "slug", PrimitiveValue.json(writeSlug(service.getSlug())),
                        "state", PrimitiveValue.utf8(service.getState().toString()),
                        "readonly_state", Ydb.nullableUtf8(Optional.ofNullable(service.getReadOnlyState())
                                .map(ServiceReadOnlyState::toString).orElse(null)),
                        "is_exportable", PrimitiveValue.bool(service.isExportable()),
                        "parent_id", Ydb.nullableInt64(service.getParentId().orElse(null))
        ))).toArray(StructValue[]::new)));
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<List<ServiceSlugWithParent>> getServiceSlugsWithParent(YdbTxSession session, List<String> parentSlugs) {
        if (parentSlugs.isEmpty()) {
            return Mono.just(Collections.emptyList());
        }

        String query = ydbQuerySource.getAbcQuery("yql.queries.services.getServiceSlugsWithParent");
        ListValue slugsParam = ListValue.of(parentSlugs.stream()
                .map(slug -> PrimitiveValue.json(writeSlug(slug)))
                .toArray(PrimitiveValue[]::new));
        Params params = Params.of("$parent_slugs", slugsParam);
        return session.executeDataQueryRetryable(query, params).map(this::toServiceSlugWithParent);
    }

    public Mono<WithTxId<Optional<ServiceWithStatesModel>>> getServiceStatesById(YdbTxSession session, long id) {
        String query = ydbQuerySource.getAbcQuery("yql.queries.services.getOneServiceWithStatesById");
        Params params = Params.of("$id", PrimitiveValue.int64(id));
        return session.executeDataQueryRetryable(query, params)
                .map(result -> new WithTxId<>(toServiceWithStates(result), result.getTxId()));
    }

    public Mono<List<ServiceWithStatesModel>> getServiceStatesByIds(YdbTxSession session, List<Long> ids) {
        String query = ydbQuerySource.getAbcQuery("yql.queries.services.getServicesWithStatesByIds");
        ListValue idsParam = ListValue.of(ids.stream().map(PrimitiveValue::int64).toArray(PrimitiveValue[]::new));
        Params params = Params.of("$ids", idsParam);
        return session.executeDataQueryRetryable(query, params).map(this::toServicesWithStates);
    }

    public Mono<List<ServiceNode>> getAllServiceNodes(YdbSession session) {
        return session.readTable(ydbQuerySource.preprocessAbcTableName("public_services_service"),
                YdbReadTableSettings.builder().ordered(true).addColumns("id", "parent_id").build())
                .flatMapIterable(reader -> {
                    List<ServiceNode> page = new ArrayList<>();
                    while (reader.next()) {
                        page.add(new ServiceNode(
                                reader.getColumn("id").getInt64(),
                                Ydb.int64OrNull(reader.getColumn("parent_id"))
                        ));
                    }
                    return page;
                }).collectList();
    }

    public Mono<List<ServiceSlugNode>> getAllServiceSlugNodes(YdbSession session) {
        return session.readTable(ydbQuerySource.preprocessAbcTableName("public_services_service"),
                        YdbReadTableSettings.builder().ordered(true).addColumns("id", "parent_id", "slug").build())
                .flatMapIterable(reader -> {
                    List<ServiceSlugNode> page = new ArrayList<>();
                    while (reader.next()) {
                        page.add(new ServiceSlugNode(
                                reader.getColumn("id").getInt64(),
                                Ydb.int64OrNull(reader.getColumn("parent_id")),
                                readSlug(reader.getColumn("slug").getJson())
                        ));
                    }
                    return page;
                }).collectList();
    }

    private ListValue toAllParentsListValue(Long2LongMultimap data, TenantId tenantId) {
        ArrayList<StructValue> values = new ArrayList<>();
        data.forEach((serviceId, parents) -> {
            parents.remove(0);
            values.add(StructValue.of(
                    "tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                    "service_id", PrimitiveValue.int64(serviceId),
                    "all_parents_service_ids", writeIds(parents)
            ));
        });
        return ListValue.of(values.toArray(new StructValue[0]));
    }

    private LongSet readIds(DataQueryResult result) {
        if (result.isEmpty()) {
            return LongSets.EMPTY_SET;
        }
        if (result.getResultSetCount() > 1) {
            throw new IllegalStateException("Too many result sets");
        }
        ResultSetReader reader = result.getResultSet(0);
        if (!reader.next()) {
            return LongSets.EMPTY_SET;
        }
        if (reader.getRowCount() > 1) {
            throw new IllegalStateException("Non unique resource");
        }
        if (reader.getColumnCount() > 1) {
            throw new IllegalStateException("Too many columns");
        }

        String json = reader.getColumn(0).getJsonDocument();
        try {
            Set<Long> parents = idsReader.readValue(json);
            return new LongOpenHashSet(parents);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }

    private Map<Long, LongSet> readMultiIds(DataQueryResult result) {
        if (result.isEmpty()) {
            return new HashMap<>();
        }
        if (result.getResultSetCount() > 1) {
            throw new IllegalStateException("Too many result sets");
        }
        ResultSetReader reader = result.getResultSet(0);
        Map<Long, LongSet> map = new HashMap<>();
        while (reader.next()) {
            long serviceId = reader.getColumn("service_id").getInt64();
            try {
                Set<Long> parents = idsReader.readValue(reader
                        .getColumn("all_parents_service_ids").getJsonDocument());
                map.put(serviceId, new LongOpenHashSet(parents));
            } catch (JsonProcessingException e) {
                throw new UncheckedIOException(e);
            }
        }
        return map;
    }

    private PrimitiveValue writeIds(Set<Long> tags) {
        try {
            String json = idsWriter.writeValueAsString(tags);
            return PrimitiveValue.jsonDocument(json);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }

    private Optional<ServiceMinimalModel> toMinimalService(DataQueryResult result) {
        Optional<ResultSetReader> reader = getReaderToOneService(result);
        if (reader.isEmpty()) {
            return Optional.empty();
        }
        return Optional.of(readOneServiceMinimal(reader.get()));
    }

    private Optional<ServiceModel> toService(DataQueryResult result) {
        Optional<ResultSetReader> reader = getReaderToOneService(result);
        if (reader.isEmpty()) {
            return Optional.empty();
        }
        return Optional.of(readOneService(reader.get()));
    }

    private List<ServiceMinimalModel> toMinimalServices(DataQueryResult result) {
        Optional<ResultSetReader> optionalReader = getReaderToManyServices(result);
        if (optionalReader.isEmpty()) {
            return List.of();
        }
        ResultSetReader reader = optionalReader.get();
        List<ServiceMinimalModel> services = new ArrayList<>();
        while (reader.next()) {
            services.add(readOneServiceMinimal(reader));
        }
        return services;
    }

    private List<ServiceSlugWithParent> toServiceSlugWithParent(DataQueryResult result) {
        Optional<ResultSetReader> optionalReader = getReaderToManyServices(result);
        if (optionalReader.isEmpty()) {
            return List.of();
        }
        ResultSetReader reader = optionalReader.get();
        List<ServiceSlugWithParent> services = new ArrayList<>();
        while (reader.next()) {
            ValueReader slugValueReader = reader.getColumn("slug");
            String slug = slugValueReader.isOptionalItemPresent() ?
                    readSlug(slugValueReader.getJson()) : null;
            String parentSlug = readSlug(reader.getColumn("parent_slug").getJson());
            ServiceSlugWithParent serviceSlugWithParent = new ServiceSlugWithParent(slug, parentSlug);
            services.add(serviceSlugWithParent);
        }
        return services;
    }

    private ServiceMinimalModel readOneServiceMinimal(ResultSetReader reader) {
        return new ServiceMinimalModel(readOneService(reader));
    }

    private ServiceModel readOneService(ResultSetReader reader) {
        long id = reader.getColumn("id").getInt64();
        long parentId = reader.getColumn("parent_id").getInt64();
        String name = reader.getColumn("name").getUtf8();
        String nameEn = reader.getColumn("name_en").getUtf8();
        String slug = readSlug(reader.getColumn("slug").getJson());
        ServiceState state = ServiceState.fromString(reader.getColumn("state").getUtf8());
        ValueReader valueReader = reader.getColumn("readonly_state");
        ServiceReadOnlyState readOnlyState = valueReader.isOptionalItemPresent() ?
                ServiceReadOnlyState.fromString(valueReader.getUtf8()) : null;
        boolean exportable = reader.getColumn("is_exportable").getBool();
        return new ServiceModel(id, parentId, name, nameEn, slug, state, readOnlyState, exportable);
    }

    private ServiceWithStatesModel readOneServiceWithStates(ResultSetReader reader) {
        long id = reader.getColumn("id").getInt64();
        ServiceState state = ServiceState.fromString(reader.getColumn("state").getUtf8());
        ValueReader valueReader = reader.getColumn("readonly_state");
        ServiceReadOnlyState readOnlyState = valueReader.isOptionalItemPresent() ?
                ServiceReadOnlyState.fromString(valueReader.getUtf8()) : null;
        boolean exportable = reader.getColumn("is_exportable").getBool();
        String slug = readSlug(reader.getColumn("slug").getJson());
        return new ServiceWithStatesModel(id, state, readOnlyState, exportable, slug);
    }

    private String readSlug(String json) {
        try {
            return slugReader.readValue(json);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }

    private String writeSlug(String slug) {
        try {
            return slugWriter.writeValueAsString(slug);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }

    private Optional<ServiceWithStatesModel> toServiceWithStates(DataQueryResult result) {
        Optional<ResultSetReader> reader = getReaderToOneService(result);
        if (reader.isEmpty()) {
            return Optional.empty();
        }
        return Optional.of(readOneServiceWithStates(reader.get()));
    }

    private List<ServiceWithStatesModel> toServicesWithStates(DataQueryResult result) {
        Optional<ResultSetReader> optionalReader = getReaderToManyServices(result);
        if (optionalReader.isEmpty()) {
            return List.of();
        }
        ResultSetReader reader = optionalReader.get();
        List<ServiceWithStatesModel> services = new ArrayList<>();
        while (reader.next()) {
            services.add(readOneServiceWithStates(reader));
        }
        return services;
    }

    private Optional<ResultSetReader> getReaderToOneService(DataQueryResult result) {
        if (result.isEmpty()) {
            return Optional.empty();
        }
        if (result.getResultSetCount() > 1) {
            throw new IllegalStateException("Too many result sets");
        }
        ResultSetReader reader = result.getResultSet(0);
        if (!reader.next()) {
            return Optional.empty();
        }
        if (reader.getRowCount() > 1) {
            throw new IllegalStateException("Non unique service");
        }
        return Optional.of(reader);
    }

    private Optional<ResultSetReader> getReaderToManyServices(DataQueryResult result) {
        if (result.isEmpty()) {
            return Optional.empty();
        }
        if (result.getResultSetCount() > 1) {
            throw new IllegalStateException("Too many result sets");
        }
        ResultSetReader reader = result.getResultSet(0);
        return Optional.of(reader);
    }
}
