package ru.yandex.intranet.d.dao.settings

import com.fasterxml.jackson.core.JsonProcessingException
import com.fasterxml.jackson.core.type.TypeReference
import com.yandex.ydb.table.query.Params
import com.yandex.ydb.table.result.ResultSetReader
import com.yandex.ydb.table.values.OptionalType
import com.yandex.ydb.table.values.OptionalValue
import com.yandex.ydb.table.values.PrimitiveType
import com.yandex.ydb.table.values.PrimitiveValue
import com.yandex.ydb.table.values.StructValue
import kotlinx.coroutines.reactor.awaitSingle
import kotlinx.coroutines.reactor.awaitSingleOrNull
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.stereotype.Component
import ru.yandex.intranet.d.dao.DaoReader
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource
import ru.yandex.intranet.d.datasource.model.YdbTxSession
import ru.yandex.intranet.d.model.TenantId
import ru.yandex.intranet.d.model.settings.SettingsKey
import ru.yandex.intranet.d.util.ObjectMapperHolder
import java.io.UncheckedIOException

/**
 * Runtime settings DAO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
class RuntimeSettingsDao(private val ydbQuerySource: YdbQuerySource,
                         @Qualifier("ydbJsonObjectMapper") private val objectMapper: ObjectMapperHolder) {

    suspend fun <T: Any> get(session: YdbTxSession, tenantId: TenantId, id: SettingsKey<T>): T? {
        val query = ydbQuerySource.getQuery("yql.queries.runtimeSettings.getById")
        val params = toKeyParams(tenantId, id)
        return DaoReader.toModel(session
            .executeDataQueryRetryable(query, params).awaitSingle()) { reader -> toModel(reader, id.datatype) }
    }

    suspend fun <T: Any> upsertOneRetryable(session: YdbTxSession, tenantId: TenantId, id: SettingsKey<T>, value: T?) {
        val query = ydbQuerySource.getQuery("yql.queries.runtimeSettings.upsertOne")
        val params = toUpsertOneParams(tenantId, id, value)
        session.executeDataQueryRetryable(query, params).awaitSingleOrNull()
    }

    private fun <T: Any> toModel(reader: ResultSetReader, datatype: TypeReference<T>): T? {
        val columnReader = reader.getColumn("value")
        if (!columnReader.isOptionalItemPresent) {
            return null
        }
        val json = columnReader.jsonDocument
        return try {
            objectMapper.objectMapper.readValue(json, datatype)
        } catch (e: JsonProcessingException) {
            throw UncheckedIOException(e)
        }
    }

    private fun <T: Any> toKeyParams(tenantId: TenantId, id: SettingsKey<T>) = Params.of(
        "\$id", toKeyStruct(tenantId, id)
    )

    private fun <T: Any> toKeyStruct(tenantId: TenantId, id: SettingsKey<T>) = StructValue.of(mapOf(
        "tenant_id" to PrimitiveValue.utf8(tenantId.id),
        "id" to PrimitiveValue.utf8(id.id)
    ))

    private fun <T: Any> toUpsertOneParams(tenantId: TenantId, id: SettingsKey<T>, value: T?) = Params.of(
        "\$value", toUpsertStruct(tenantId, id, value)
    )

    private fun <T: Any> toUpsertStruct(tenantId: TenantId, id: SettingsKey<T>, value: T?) = StructValue.of(
        mapOf(
            "tenant_id" to PrimitiveValue.utf8(tenantId.id),
            "id" to PrimitiveValue.utf8(id.id),
            "value" to toValue(value)
        )
    )

    private fun <T: Any> toValue(value: T?): OptionalValue {
        val type = OptionalType.of(PrimitiveType.jsonDocument())
        if (value == null) {
            return type.emptyValue()
        }
        val jsonDocument = try {
            val json = objectMapper.objectMapper.writeValueAsString(value)
            PrimitiveValue.jsonDocument(json)
        } catch (e: JsonProcessingException) {
            throw UncheckedIOException(e)
        }
        return type.newValue(jsonDocument)
    }

}
