package ru.yandex.intranet.d.dao.sync;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.core.type.TypeReference;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.result.ValueReader;
import com.yandex.ydb.table.values.OptionalType;
import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.Value;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.dao.AbstractDao;
import ru.yandex.intranet.d.dao.JsonFieldHelper;
import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource;
import ru.yandex.intranet.d.datasource.model.WithTxId;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.sync.Errors;
import ru.yandex.intranet.d.model.sync.ProvidersSyncErrorsModel;
import ru.yandex.intranet.d.model.sync.ProvidersSyncErrorsModel.Identity;
import ru.yandex.intranet.d.util.ObjectMapperHolder;
import ru.yandex.intranet.d.util.Util;

import static java.util.Optional.ofNullable;
import static ru.yandex.intranet.d.datasource.Ydb.nullToEmptyUtf8;
import static ru.yandex.intranet.d.datasource.Ydb.timestamp;
import static ru.yandex.intranet.d.datasource.Ydb.utf8;
import static ru.yandex.intranet.d.datasource.Ydb.utf8EmptyToNull;

/**
 * ProvidersSyncErrorsDao.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 01-04-2021
 */
@Component
public class ProvidersSyncErrorsDao extends AbstractDao<ProvidersSyncErrorsModel, Identity> {
    private static final Fields[] IDENTITY_FIELDS = {
            Fields.PROVIDER_ID,
            Fields.ACCOUNT_SPACE_ID,
            Fields.SYNC_ID,
            Fields.ERROR_ID
    };

    private final JsonFieldHelper<Errors> errorsHelper;

    protected ProvidersSyncErrorsDao(
            YdbQuerySource ydbQuerySource,
            @Qualifier("ydbJsonObjectMapper") ObjectMapperHolder objectMapper
    ) {
        super(ydbQuerySource);
        this.errorsHelper = new JsonFieldHelper<>(objectMapper, new TypeReference<>() {
        });
    }

    public Mono<WithTxId<List<ProvidersSyncErrorsModel>>> getBySyncId(
            YdbTxSession session,
            TenantId tenantId,
            String providerId,
            String accountSpaceId,
            String syncId
    ) {
        String query = ydbQuerySource.getQuery("yql.queries.providers_sync_errors.getBySyncId");
        Params params = Params.of(
                "$tenant_id", utf8(tenantId.getId()),
                "$provider_id", utf8(providerId),
                "$account_space_id", nullToEmptyUtf8(ofNullable(accountSpaceId)),
                "$sync_id", utf8(syncId)
        );
        return session.executeDataQueryRetryable(query, params)
                .map(result -> new WithTxId<>(toModels(result), result.getTxId()));
    }

    public Mono<WithTxId<Void>> clearOldErrorsRetryable(
            YdbTxSession session,
            TenantId tenantId,
            String providerId,
            String lastsSyncIdToSave
    ) {
        String query = ydbQuerySource.getQuery("yql.queries.providers_sync_errors.clearOldErrors");
        Params params = Params.of(
                "$tenant_id", utf8(tenantId.getId()),
                "$provider_id", utf8(providerId),
                "$lasts_sync_id_to_save", utf8(lastsSyncIdToSave)
        );
        return session.executeDataQueryRetryable(query, params)
                .map(result -> new WithTxId<>(null, result.getTxId()));
    }

    public Mono<WithTxId<List<ProvidersSyncErrorsModel>>> getAllByProvider(
            YdbTxSession session,
            TenantId tenantId,
            String providerId
    ) {
        String query = ydbQuerySource.getQuery("yql.queries.providers_sync_errors.getAllByProvider");
        Params params = Params.of(
                "$tenant_id", utf8(tenantId.getId()),
                "$provider_id", utf8(providerId)
        );
        return session.executeDataQueryRetryable(query, params)
                .map(result -> new WithTxId<>(toModels(result), result.getTxId()));
    }

    @Override
    protected WithTenant<Identity> getIdentityWithTenant(ProvidersSyncErrorsModel model) {
        return new WithTenant<>(model.getTenantId(), model.getIdentity());
    }

    @Override
    protected Params getIdentityParams(Identity id) {
        Params params = Params.create();
        for (Fields field : IDENTITY_FIELDS) {
            field.toIdentityParam(id, params);
        }
        return params;
    }

    @SuppressWarnings("rawtypes")
    @Override
    protected Map<String, Value> prepareFieldValues(ProvidersSyncErrorsModel model) {
        Map<String, Value> fields = new HashMap<>();
        for (Fields field : Fields.values()) {
            field.write(model, fields, this);
        }
        return fields;
    }

    @Override
    protected ProvidersSyncErrorsModel readOneRow(ResultSetReader reader, Map<String, TenantId> tenantIdCache) {
        ProvidersSyncErrorsModel.Builder builder = new ProvidersSyncErrorsModel.Builder();
        for (Fields field : Fields.values()) {
            field.read(reader, builder, this);
        }
        return builder.build();
    }

    @Override
    protected String queryKeyPrefix() {
        return "yql.queries.providers_sync_errors";
    }

    @SuppressWarnings({"unused", "RedundantSuppression", "rawtypes"})
    public enum Fields {
        TENANT_ID {
            @Override
            public void read(ResultSetReader reader, ProvidersSyncErrorsModel.Builder builder,
                             ProvidersSyncErrorsDao dao) {
                builder.setTenantId(Tenants.getInstance(utf8(reader.getColumn(field()))));
            }

            @Override
            public void write(ProvidersSyncErrorsModel model, Map<String, Value> fields, ProvidersSyncErrorsDao dao) {
                fields.put(field(), utf8(model.getTenantId().getId()));
            }
        },
        PROVIDER_ID {
            @Override
            public void read(ResultSetReader reader, ProvidersSyncErrorsModel.Builder builder,
                             ProvidersSyncErrorsDao dao) {
                builder.setProviderId(utf8(reader.getColumn(field())));
            }

            @Override
            public void write(ProvidersSyncErrorsModel model, Map<String, Value> fields, ProvidersSyncErrorsDao dao) {
                fields.put(field(), utf8(model.getProviderId()));
            }

            @Override
            public void toIdentityParam(Identity identity, Params params) {
                params.put("$" + field(), utf8(identity.getProviderId()));
            }
        },
        ACCOUNT_SPACE_ID {
            @Override
            public void read(ResultSetReader reader, ProvidersSyncErrorsModel.Builder builder,
                             ProvidersSyncErrorsDao dao) {
                builder.setAccountSpaceId(utf8EmptyToNull(reader.getColumn(field())));
            }

            @Override
            public void write(ProvidersSyncErrorsModel model, Map<String, Value> fields, ProvidersSyncErrorsDao dao) {
                fields.put(field(), nullToEmptyUtf8(model.getAccountSpaceId()));
            }

            @Override
            public void toIdentityParam(Identity identity, Params params) {
                params.put("$" + field(), nullToEmptyUtf8(identity.getAccountSpaceId()));
            }
        },
        SYNC_ID {
            @Override
            public void read(ResultSetReader reader, ProvidersSyncErrorsModel.Builder builder,
                             ProvidersSyncErrorsDao dao) {
                builder.setSyncId(utf8(reader.getColumn(field())));
            }

            @Override
            public void write(ProvidersSyncErrorsModel model, Map<String, Value> fields, ProvidersSyncErrorsDao dao) {
                fields.put(field(), utf8(model.getSyncId()));
            }

            @Override
            public void toIdentityParam(Identity identity, Params params) {
                params.put("$" + field(), utf8(identity.getSyncId()));
            }
        },
        ERROR_ID {
            @Override
            public void read(ResultSetReader reader, ProvidersSyncErrorsModel.Builder builder,
                             ProvidersSyncErrorsDao dao) {
                builder.setErrorId(utf8(reader.getColumn(field())));
            }

            @Override
            public void write(ProvidersSyncErrorsModel model, Map<String, Value> fields, ProvidersSyncErrorsDao dao) {
                fields.put(field(), utf8(model.getErrorId()));
            }

            @Override
            public void toIdentityParam(Identity identity, Params params) {
                params.put("$" + field(), utf8(identity.getErrorId()));
            }
        },
        SYNC_START {
            @Override
            public void read(ResultSetReader reader, ProvidersSyncErrorsModel.Builder builder,
                             ProvidersSyncErrorsDao dao) {
                builder.setSyncStart(timestamp(reader.getColumn(field())));
            }

            @Override
            public void write(ProvidersSyncErrorsModel model, Map<String, Value> fields, ProvidersSyncErrorsDao dao) {
                fields.put(field(), timestamp(model.getSyncStart()));
            }
        },
        REQUEST_TIMESTAMP {
            @Override
            public void read(ResultSetReader reader, ProvidersSyncErrorsModel.Builder builder,
                             ProvidersSyncErrorsDao dao) {
                builder.setRequestTimestamp(timestamp(reader.getColumn(field())));
            }

            @Override
            public void write(ProvidersSyncErrorsModel model, Map<String, Value> fields, ProvidersSyncErrorsDao dao) {
                fields.put(field(), timestamp(model.getRequestTimestamp()));
            }
        },
        REQUEST {
            @Override
            public void read(ResultSetReader reader, ProvidersSyncErrorsModel.Builder builder,
                             ProvidersSyncErrorsDao dao
            ) {
                ValueReader valueReader = reader.getColumn(field());
                if (valueReader.isOptionalItemPresent()) {
                    builder.setRequest(valueReader.getJsonDocument());
                } else {
                    builder.setRequest(null);
                }
            }

            @Override
            public void write(ProvidersSyncErrorsModel model, Map<String, Value> fields, ProvidersSyncErrorsDao dao) {
                String request = model.getRequest();
                OptionalType type = OptionalType.of(PrimitiveType.jsonDocument());
                if (Util.isEmpty(request)) {
                    fields.put(field(), type.emptyValue());
                } else {
                    fields.put(field(), type.newValue(PrimitiveValue.jsonDocument(request)));
                }
            }
        },
        ERRORS {
            @Override
            public void read(ResultSetReader reader, ProvidersSyncErrorsModel.Builder builder,
                             ProvidersSyncErrorsDao dao) {
                builder.setErrors(dao.errorsHelper.read(reader.getColumn(field())));
            }

            @Override
            public void write(ProvidersSyncErrorsModel model, Map<String, Value> fields, ProvidersSyncErrorsDao dao) {
                fields.put(field(), dao.errorsHelper.write(model.getErrors()));
            }
        };

        private final String fieldName;

        Fields() {
            fieldName = name().toLowerCase();
        }

        public String field() {
            return fieldName;
        }

        public abstract void read(ResultSetReader reader, ProvidersSyncErrorsModel.Builder builder,
                                  ProvidersSyncErrorsDao dao);

        public abstract void write(ProvidersSyncErrorsModel model, Map<String, Value> fields,
                                   ProvidersSyncErrorsDao dao);

        public void toIdentityParam(Identity identity, Params params) {
            throw new UnsupportedOperationException();
        }
    }
}
