package ru.yandex.intranet.d.dao.sync;

import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.core.type.TypeReference;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.Value;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.dao.AbstractDao;
import ru.yandex.intranet.d.dao.JsonFieldHelper;
import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.datasource.Ydb;
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource;
import ru.yandex.intranet.d.datasource.model.WithTxId;
import ru.yandex.intranet.d.datasource.model.YdbReadTableSettings;
import ru.yandex.intranet.d.datasource.model.YdbSession;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.sync.ProvidersSyncStatusModel;
import ru.yandex.intranet.d.model.sync.SyncStats;
import ru.yandex.intranet.d.util.ObjectMapperHolder;

import static ru.yandex.intranet.d.datasource.Ydb.nullableTimestamp;
import static ru.yandex.intranet.d.datasource.Ydb.nullableUtf8;
import static ru.yandex.intranet.d.datasource.Ydb.timestamp;
import static ru.yandex.intranet.d.datasource.Ydb.timestampOrNull;
import static ru.yandex.intranet.d.datasource.Ydb.utf8;
import static ru.yandex.intranet.d.datasource.Ydb.utf8OrNull;

/**
 * ProvidersSyncStatusDao.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 01-04-2021
 */
@Component
public class ProvidersSyncStatusDao extends AbstractDao<ProvidersSyncStatusModel, String> {
    private final JsonFieldHelper<SyncStats> lastSyncStatsHelper;

    public ProvidersSyncStatusDao(
            YdbQuerySource ydbQuerySource,
            @Qualifier("ydbJsonObjectMapper") ObjectMapperHolder objectMapper
    ) {
        super(ydbQuerySource);
        this.lastSyncStatsHelper = new JsonFieldHelper<>(objectMapper, new TypeReference<>() {
        });
    }

    public Mono<WithTxId<Void>> upsertNewSyncRetryable(
            YdbTxSession session,
            TenantId tenantId,
            String providerId,
            Instant newSyncStart,
            String newSyncId,
            ProvidersSyncStatusModel.SyncStatuses newSyncStatus
    ) {
        String query = ydbQuerySource.getQuery("yql.queries.providers_sync_status.upsertNewSync");
        Params params = Params.of(
                "$tenant_id", utf8(tenantId.getId()),
                "$provider_id", utf8(providerId),
                "$new_sync_start", timestamp(newSyncStart),
                "$new_sync_id", utf8(newSyncId),
                "$new_sync_status", utf8(newSyncStatus.name())
        );
        return session.executeDataQueryRetryable(query, params).map(result -> new WithTxId<>(null, result.getTxId()));
    }

    public Flux<ProvidersSyncStatusModel> getAll(YdbSession session) {
        YdbReadTableSettings.Builder settings = YdbReadTableSettings.builder();
        settings.ordered(true);
        for (Fields field : Fields.values()) {
            settings.addColumn(field.field());
        }
        return session.readTable(
                ydbQuerySource.preprocessTableName("providers_sync_status"), settings.build()
        ).flatMapIterable(reader -> {
            List<ProvidersSyncStatusModel> page = new ArrayList<>();
            while (reader.next()) {
                page.add(readOneRow(reader));
            }
            return page;
        });
    }

    @Override
    protected WithTenant<String> getIdentityWithTenant(ProvidersSyncStatusModel model) {
        return new WithTenant<>(model.getTenantId(), model.getProviderId());
    }

    @Override
    protected Params getIdentityParams(String id) {
        return Params.create().put("$provider_id", utf8(id));
    }

    @SuppressWarnings("rawtypes")
    @Override
    protected Map<String, Value> prepareFieldValues(ProvidersSyncStatusModel model) {
        Map<String, Value> fields = new HashMap<>();
        for (Fields field : Fields.values()) {
            field.write(model, fields, this);
        }
        return fields;
    }

    @Override
    protected ProvidersSyncStatusModel readOneRow(ResultSetReader reader, Map<String, TenantId> tenantIdCache) {
        return readOneRow(reader);
    }

    protected ProvidersSyncStatusModel readOneRow(ResultSetReader reader) {
        ProvidersSyncStatusModel.Builder builder = new ProvidersSyncStatusModel.Builder();
        for (Fields field : Fields.values()) {
            field.read(reader, builder, this);
        }
        return builder.build();
    }

    @Override
    protected String queryKeyPrefix() {
        return "yql.queries.providers_sync_status";
    }

    @SuppressWarnings({"unused", "RedundantSuppression", "rawtypes"})
    private enum Fields {
        TENANT_ID {
            @Override
            public void read(ResultSetReader reader, ProvidersSyncStatusModel.Builder builder,
                             ProvidersSyncStatusDao dao) {
                builder.setTenantId(Tenants.getInstance(utf8(reader.getColumn(field()))));
            }

            @Override
            public void write(ProvidersSyncStatusModel model, Map<String, Value> fields, ProvidersSyncStatusDao dao) {
                fields.put(field(), utf8(model.getTenantId().getId()));
            }
        },
        PROVIDER_ID {
            @Override
            public void read(ResultSetReader reader, ProvidersSyncStatusModel.Builder builder,
                             ProvidersSyncStatusDao dao) {
                builder.setProviderId(utf8(reader.getColumn(field())));
            }

            @Override
            public void write(ProvidersSyncStatusModel model, Map<String, Value> fields, ProvidersSyncStatusDao dao) {
                fields.put(field(), Ydb.utf8(model.getProviderId()));
            }
        },
        LAST_SYNC_START {
            @Override
            public void read(ResultSetReader reader, ProvidersSyncStatusModel.Builder builder,
                             ProvidersSyncStatusDao dao) {
                builder.setLastSyncStart(timestampOrNull(reader.getColumn(field())));
            }

            @Override
            public void write(ProvidersSyncStatusModel model, Map<String, Value> fields, ProvidersSyncStatusDao dao) {
                fields.put(field(), nullableTimestamp(model.getLastSyncStart()));
            }
        },
        LAST_SYNC_ID {
            @Override
            public void read(ResultSetReader reader, ProvidersSyncStatusModel.Builder builder,
                             ProvidersSyncStatusDao dao) {
                builder.setLastSyncId(utf8OrNull(reader.getColumn(field())));
            }

            @Override
            public void write(ProvidersSyncStatusModel model, Map<String, Value> fields, ProvidersSyncStatusDao dao) {
                fields.put(field(), nullableUtf8(model.getLastSyncId()));
            }
        },
        LAST_SYNC_FINISH {
            @Override
            public void read(ResultSetReader reader, ProvidersSyncStatusModel.Builder builder,
                             ProvidersSyncStatusDao dao) {
                builder.setLastSyncFinish(timestampOrNull(reader.getColumn(field())));
            }

            @Override
            public void write(ProvidersSyncStatusModel model, Map<String, Value> fields, ProvidersSyncStatusDao dao) {
                fields.put(field(), nullableTimestamp(model.getLastSyncFinish()));
            }
        },
        LAST_SYNC_STATUS {
            @Override
            public void read(ResultSetReader reader, ProvidersSyncStatusModel.Builder builder,
                             ProvidersSyncStatusDao dao) {
                builder.setLastSyncStatus(utf8OrNull(reader.getColumn(field())));
            }

            @Override
            public void write(ProvidersSyncStatusModel model, Map<String, Value> fields, ProvidersSyncStatusDao dao) {
                String value = model.getLastSyncStatus() != null ? model.getLastSyncStatus().name() : null;
                fields.put(field(), nullableUtf8(value));
            }
        },
        LAST_SYNC_STATS {
            @Override
            public void read(ResultSetReader reader, ProvidersSyncStatusModel.Builder builder,
                             ProvidersSyncStatusDao dao) {
                builder.setLastSyncStats(dao.lastSyncStatsHelper.read(reader.getColumn(field())));
            }

            @Override
            public void write(ProvidersSyncStatusModel model, Map<String, Value> fields, ProvidersSyncStatusDao dao) {
                fields.put(field(), dao.lastSyncStatsHelper.writeOptional(model.getLastSyncStats()));
            }
        },
        NEW_SYNC_START {
            @Override
            public void read(ResultSetReader reader, ProvidersSyncStatusModel.Builder builder,
                             ProvidersSyncStatusDao dao) {
                builder.setNewSyncStart(timestamp(reader.getColumn(field())));
            }

            @Override
            public void write(ProvidersSyncStatusModel model, Map<String, Value> fields, ProvidersSyncStatusDao dao) {
                fields.put(field(), timestamp(model.getNewSyncStart()));
            }
        },
        NEW_SYNC_ID {
            @Override
            public void read(ResultSetReader reader, ProvidersSyncStatusModel.Builder builder,
                             ProvidersSyncStatusDao dao) {
                builder.setNewSyncId(utf8(reader.getColumn(field())));
            }

            @Override
            public void write(ProvidersSyncStatusModel model, Map<String, Value> fields, ProvidersSyncStatusDao dao) {
                fields.put(field(), utf8(model.getNewSyncId()));
            }
        },
        NEW_SYNC_STATUS {
            @Override
            public void read(ResultSetReader reader, ProvidersSyncStatusModel.Builder builder,
                             ProvidersSyncStatusDao dao) {
                builder.setNewSyncStatus(utf8(reader.getColumn(field())));
            }

            @Override
            public void write(ProvidersSyncStatusModel model, Map<String, Value> fields, ProvidersSyncStatusDao dao) {
                fields.put(field(), utf8(model.getNewSyncStatus().name()));
            }
        },
        LAST_SUCCESSFUL_SYNC_FINISH {
            @Override
            public void read(ResultSetReader reader, ProvidersSyncStatusModel.Builder builder,
                             ProvidersSyncStatusDao dao) {
                builder.setLastSuccessfulSyncFinish(timestampOrNull(reader.getColumn(field())));
            }

            @Override
            public void write(ProvidersSyncStatusModel model, Map<String, Value> fields, ProvidersSyncStatusDao dao) {
                fields.put(field(), nullableTimestamp(model.getLastSuccessfulSyncFinish()));
            }
        };

        private final String fieldName;

        Fields() {
            fieldName = name().toLowerCase();
        }

        public String field() {
            return fieldName;
        }

        public abstract void read(ResultSetReader reader, ProvidersSyncStatusModel.Builder builder,
                                  ProvidersSyncStatusDao dao);

        public abstract void write(ProvidersSyncStatusModel model, Map<String, Value> fields,
                                   ProvidersSyncStatusDao dao);
    }
}
