package ru.yandex.intranet.d.dao.transfers;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.Value;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.AbstractDao;
import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.datasource.Ydb;
import ru.yandex.intranet.d.datasource.impl.YdbQuerySource;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.transfers.PendingTransferRequestsModel;

/**
 * PendingTransferRequestsDao.
 *
 * @author Petr Surkov <petrsurkov@yandex-team.ru>
 */
@Component
public class PendingTransferRequestsDao extends AbstractDao<PendingTransferRequestsModel, String> {
    public PendingTransferRequestsDao(YdbQuerySource ydbQuerySource) {
        super(ydbQuerySource);
    }

    public Mono<List<PendingTransferRequestsModel>> getAllByTenant(YdbTxSession session, TenantId tenantId) {
        String firstPageQuery = ydbQuerySource
                .getQuery(queryKeyPrefix() + ".getByTenantFirstPage");
        Params firstPageParams = Params.of("$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS));
        return session.executeDataQueryRetryable(firstPageQuery, firstPageParams).flatMap(firstPageResult -> {
            List<PendingTransferRequestsModel> models = toModels(firstPageResult);
            if (!firstPageResult.getResultSet(0).isTruncated() && models.size() < Ydb.MAX_RESPONSE_ROWS) {
                return Mono.just(models);
            }
            return getNextPageByTenant(session, tenantId, models.get(models.size() - 1)).expand(tuple -> {
                if (!tuple.getT2() && tuple.getT1().size() < Ydb.MAX_RESPONSE_ROWS) {
                    return Mono.empty();
                } else {
                    return getNextPageByTenant(session, tenantId, tuple.getT1().get(tuple.getT1().size() - 1));
                }
            }).map(Tuple2::getT1).reduce(models, (l, r) -> Stream.concat(l.stream(), r.stream())
                    .collect(Collectors.toList()));
        });
    }

    private Mono<Tuple2<List<PendingTransferRequestsModel>, Boolean>> getNextPageByTenant(
            YdbTxSession session, TenantId tenantId, PendingTransferRequestsModel from) {
        String nextPageQuery = ydbQuerySource
                .getQuery(queryKeyPrefix() + ".getByTenantNextPage");
        Map<String, Value<?>> paramsMap = Map.of("$limit", PrimitiveValue.uint64(Ydb.MAX_RESPONSE_ROWS),
                "$tenant_id", PrimitiveValue.utf8(tenantId.getId()),
                "$from_request_id", PrimitiveValue.utf8(from.getRequestId()));
        Params nextPageParams = Params.copyOf(paramsMap);
        return session.executeDataQueryRetryable(nextPageQuery, nextPageParams).map(nextPageResult -> {
            List<PendingTransferRequestsModel> models = toModels(nextPageResult);
            return Tuples.of(models, nextPageResult.getResultSet(0).isTruncated());
        });
    }

    public Mono<Void> deleteOneRetryable(YdbTxSession session,
                                         WithTenant<String> idWithTenant) {
        String query = ydbQuerySource.getQuery(queryKeyPrefix() + ".deleteOne");
        Params params = getIdentityWithTenantParams(idWithTenant.getIdentity(), idWithTenant.getTenantId());
        return session.executeDataQueryRetryable(query, params).then();
    }

    public Mono<Void> deleteManyRetryable(YdbTxSession session,
                                          List<WithTenant<String>> ids) {
        if (ids.isEmpty()) {
            return Mono.empty();
        }
        String query = ydbQuerySource.getQuery(queryKeyPrefix() + ".deleteMany");
        Params params = Params.of("$ids", toWithTenantsListValue(ids));
        return session.executeDataQueryRetryable(query, params).then();
    }

    @Override
    protected WithTenant<String> getIdentityWithTenant(
            PendingTransferRequestsModel model) {
        return new WithTenant<>(model.getTenantId(), model.getRequestId());
    }

    @Override
    protected Params getIdentityParams(String requestId) {
        return Params.create()
                .put("$request_id", PrimitiveValue.utf8(requestId));
    }

    @SuppressWarnings("rawtypes")
    @Override
    protected Map<String, Value> prepareFieldValues(PendingTransferRequestsModel model) {
        HashMap<String, Value> fields = new HashMap<>();

        fields.put(Fields.TENANT_ID.field(), PrimitiveValue.utf8(model.getTenantId().getId()));
        fields.put(Fields.REQUEST_ID.field(), PrimitiveValue.utf8(model.getRequestId()));

        return fields;
    }

    @Override
    protected PendingTransferRequestsModel readOneRow(ResultSetReader reader, Map<String, TenantId> tenantIdCache) {
        return PendingTransferRequestsModel.builder()
                .tenantId(Tenants.getInstance(reader.getColumn(Fields.TENANT_ID.field()).getUtf8()))
                .requestId(reader.getColumn(Fields.REQUEST_ID.field()).getUtf8())
                .build();
    }

    @Override
    protected String queryKeyPrefix() {
        return "yql.queries.pendingTransferRequests";
    }

    public enum Fields {
        TENANT_ID,
        REQUEST_ID;

        public String field() {
            return name().toLowerCase();
        }
    }
}
